# Super App TudoAqui - Copilot Instructions

## Project Overview
**TudoAqui** (internal codename: "Mixeiro") é um super app marketplace agregador para o mercado angolano que conecta utilizadores a produtos e serviços através de uma única plataforma.

**Lema:** "Encontre tudo num só lugar"

**Categorias:** Imobiliária, Automóveis, Turismo, Serviços Diversos + Integrações de logística/mobilidade

**Tech Stack (2026):**
- **Frontend:** Flutter/Dart com motor Impeller (120 FPS, compilação nativa)
  - Código único para Android, iOS e Web
  - Segurança: Compilação para código de máquina nativo (anti-pirataria)
- **Backend:** Node.js + NestJS (TypeScript obrigatório)
  - Arquitetura de Microserviços (módulos independentes)
  - Organização "nível bancário" para escalabilidade
- **Databases:**
  - PostgreSQL: Dados estruturados (contratos, especificações)
  - Redis: Localização GPS em tempo real (táxis, entregas)
  - Firebase: Notificações push e chat em tempo real
- **Autenticação & Segurança:**
  - Google Identity / Auth0: Login seguro (Biometria, 2FA)
  - Amazon Rekognition / Google Vision AI: KYC automático (OCR de BI, Liveness Detection)
- **Integrações Angolanas:** Tupuca (logística), Heetch/Yango (mobilidade), Multicaixa Express + GPO (pagamentos)
- **Maps:** Google Maps API para geolocalização

**Deployment:** XAMPP local development (`c:\xampp\htdocs\TudoAqui`)

## Development Phases (Roadmap)

### Fase 1: Fundação e Segurança (Meses 1-2)
- **Autenticação:** Google Identity ou Auth0 com 2FA via SMS/WhatsApp
- **Módulo KYC:** Amazon Rekognition para OCR de BI angolano + Liveness Detection ([verificacao_documentos.dart](../lib/verificacao_documentos.dart))
- **Arquitetura:** PostgreSQL (dados complexos) + Redis (cache/GPS) + NestJS microserviços

### Fase 2: Módulos de Marketplace (Meses 3-4)
- Listagens de Imobiliária e Automóveis com filtros avançados (província/município, preço, ano)
- Geolocalização "perto de mim" via Google Maps
- Chat interno para proteger números de telefone dos utilizadores

### Fase 3: Integrações e Ecossistema (Meses 5-6)
- **Logística:** Botão "Pedir Entrega" integrado à API Tupuca
- **Mobilidade:** Chamar taxi (Heetch/Yango) ao clicar numa localização
- **Pagamentos:** Multicaixa Express + GPO para transações

### Fase 4: Turismo e Serviços (Mês 7)
- Booking local (restaurantes, pacotes turísticos)
- Seguros (simulação imediata com seguradoras locais)

### Fase 5: Lançamento (Mês 8)
- Beta test em Luanda
- Campanha "Segurança de comprar de quem é verificado"

## Architecture & Key Patterns

### UI/UX Design System
- **Primary Brand Color:** `Color(0xFF0052CC)` (Cobalt Blue - confiança/segurança)
- **Success State:** Green (`Colors.green`)
- **Standard Padding:** `20.0` para bordas de ecrã
- **Border Radius:** `15` para containers, `10` para botões
- **Button Height:** `55` pixels
- **Comentários justificam escolhas:** Ex: "Azul Cobalto para confiança"

### State Management Pattern
StatefulWidget com flags booleanas para estado simples (escalar para Provider/Bloc se necessário):
```dart
class _VerificacaoDocumentosState extends State<VerificacaoDocumentos> {
  bool _documentoEnviado = false;
  // Usar setState() para atualizações de UI
}
```

### Language & Localization
- **Todo o texto UI em Português (PT-AO)** - mercado angolano
- **BI** = Bilhete de Identidade
- **Alvará** = Licença comercial para empresas
- Termos comuns: "Vendedor" (seller), "Selo de Verificado" (verified badge), "Perto de mim" (near me)

### Document Verification Flow (KYC)
Ver [verificacao_documentos.dart](../lib/verificacao_documentos.dart):
1. Utilizador toca na área de upload para capturar foto do BI
2. Feedback visual muda ícone de `camera_alt` para `check_circle`
3. Botão "ENVIAR PARA ANÁLISE" ativa apenas após upload
4. Backend Node.js valida com IA (reconhecimento facial + OCR via `google_ml_kit`)
5. **Marca d'água automática:** Todas as fotos recebem watermark "TudoAqui" para prevenir fraude
6. **Selo de Verificado:** Apenas vendedores aprovados podem publicar anúncios

## Development Workflows

### File Organization
```
lib/
├── screens/
│   ├── verificacao_documentos.dart  # KYC verification (Fase 1)
│   ├── home_marketplace.dart        # Dashboard principal
│   ├── imoveis/                     # Imobiliária (Fase 2)
│   ├── automoveis/                  # Automóveis (Fase 2)
│   ├── turismo/                     # Turismo e reservas (Fase 4)
│   └── chat/                        # Chat interno (Fase 2)
├── services/
│   ├── api_tupuca.dart              # Integração logística (Fase 3)
│   ├── api_pagamentos.dart          # Multicaixa/GPO (Fase 3)
│   └── firebase_service.dart        # Notificações e chat real-time
├── models/
│   ├── vendedor.dart                # Dados do vendedor verificado
│   └── anuncio.dart                 # Estrutura de listagens
└── utils/
    └── constants.dart               # Cores, tamanhos, API endpoints
```

### Adding New Screens
1. Criar `lib/screens/nome_da_tela.dart`
2. Usar StatefulWidget para ecrãs interativos
3. Scaffold com AppBar usando cor da marca `0xFF0052CC`
4. Adicionar comentários em português para lógica de negócio
5. Integrar funcionalidades via packages (não inline): `image_picker`, `google_ml_kit`, `google_maps_flutter`, `firebase_messaging`

### Button Patterns
Full-width buttons at bottom of screen:
```dart
SizedBox(
  width: double.infinity,
  height: 55,
  child: ElevatedButton(
    style: ElevatedButton.styleFrom(
      backgroundColor: Color(0xFF0052CC),
      shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(10)),
    ),
    // ...
  ),
)
```

### Placeholder Implementation Strategy
Use comments to mark future integrations:
```dart
// Nota: Em um projeto real, você usaria o package 'image_picker' para a câmera
// Aqui chamaria a função de abrir câmera
```
Keep mockable state for UI testing before backend integration.

## Backend Integration

### Node.js API Endpoints
**Base URL:** `https://api.tudoaqui.ao` (produção) / `http://localhost:3000` (dev)

#### KYC & Autenticação
- `POST /auth/register` - Registo com 2FA
- `POST /auth/login` - Login com biometria/SMS
- `POST /kyc/upload-bi` - Upload de BI para validação
- `GET /kyc/status/:userId` - Estado da verificação

#### Marketplace
- `Business Model

### Fontes de Receita
1. **Subscrição Premium:** Empresas pagam por destaque nos resultados de pesquisa
2. **Taxa de Transação:** 2-5% em serviços reservados pelo app (turismo, booking)
3. **Publicidade:** Bancos e seguradoras nas categorias de casas/carros
4. **Taxa de Logística:** Comissão em entregas via Tupuca

## Key Dependencies (pubspec.yaml)

```yaml
dependencies:
  flutter:
    sdk: flutter
  
  # UI & Navigation
  google_fonts: ^6.1.0
  
  # Camera & ML
  image_picker: ^1.0.4
  google_ml_kit: ^0.16.0
  
  # Maps & Location
  google_maps_flutter: ^2.5.0
  geolocator: ^10.1.0
  
  # Backend & Database
  http: ^1.1.0
  postgres: ^2.6.2
  
  # Firebase
  firebase_core: ^2.24.0
  firebase_messaging: ^14.7.4
  firebase_database: ^10.4.0
  
  # Authentication
  local_auth: ^2.1.7  # Biometria
  
  # State Management (escolher conforme escala)
  provider: ^6.1.1
```

## Critical Workflows

### Comando para Executar App (Dev)
```bash
flutter run -d chrome  # Web testing
flutter run            # Android/iOS emulator
```

### Testar Integração KYC
1. Abrir [verificacao_documentos.dart](../lib/verificacao_documentos.dart)
2. Simular upload de foto (muda estado `_documentoEnviado`)
3. Verificar chamada ao endpoint `/kyc/upload-bi`
4. Confirmar recebimento de `selo_verificado: true` do backend

### Deploy para Produção
```bash
flutter build apk --release         # Android
flutter build ipa --release         # iOS (requer macOS)
firebase deploy --only hosting      # Web version
```

---

## Coding Guidelines

**Ao gerar código:**
- **Língua:** Todo o texto UI em Português (PT-AO)
- **Cores:** Usar constantes da marca (`Color(0xFF0052CC)`, verde para sucesso)
- **Comentários:** Justificar decisões de negócio em português
- **Estado:** StatefulWidget para ecrãs simples; migrar para Provider/Bloc se complexidade aumentar
- **Segurança:** Nunca armazenar dados de BI em logs; sempre encriptar antes de envio
- **UX:** Feedback visual imediato (ícones mudam, botões desabilitam até validação)
- **Geolocalização:** Sempre pedir permissão antes de aceder localização
- **Marcas angolanas:** Referenciar Tupuca, Multicaixa, Heetch/Yango nos comentários quando integrar APIs
  kyc_status ENUM('pendente', 'aprovado', 'rejeitado'),
  selo_verificado BOOLEAN DEFAULT FALSE,
  created_at TIMESTAMP
);

-- Tabela de anúncios
CREATE TABLE anuncios (
  id UUID PRIMARY KEY,
  vendedor_id UUID REFERENCES vendedores(id),
  categoria ENUM('imoveis', 'automoveis', 'turismo', 'servicos'),
  titulo VARCHAR(255),
  preco DECIMAL(10,2),
  localizacao GEOGRAPHY(POINT, 4326), -- PostGIS para geolocalização
  fotos JSONB
);
```

### Firebase Integration
- **Cloud Messaging:** Push notifications para novos chats/ofertas
- **Realtime Database:** Chat interno entre comprador-vendedor
- **Storage:** Armazenamento temporário de fotos de documentos

## Security & Compliance

### Sistema de Segurança Anti-Burla
1. **KYC Obrigatório:** Apenas vendedores com BI validado podem publicar
2. **Marca d'água Automática:** Watermark "TudoAqui" em todas as fotos de documentos
3. **Reconhecimento Facial:** Comparação entre selfie e foto do BI
4. **Sistema de Denúncia:** IA identifica padrões de fraude (ex: múltiplas contas com mesmo BI)
5. **Criptografia:** Ponta-a-ponta para dados sensíveis (BI, alvarás)
6. **Privacidade:** OCR com `google_ml_kit` no dispositivo antes de envio ao servidor (minimiza exposição de PII)

## TODO/Future Enhancements
- [ ] Add `image_picker` package for camera access
- [ ] Integrate `google_ml_kit` for document text extraction
- [ ] Implement Node.js backend endpoints
- [ ] Add error handling for failed document uploads
- [ ] Create buyer-side screens (currently seller-focused)
- [ ] Add `pubspec.yaml` and Flutter project structure

---

**When generating code:**
- Write all UI strings in Portuguese
- Use the established color palette and spacing constants
- Follow StatefulWidget pattern for interactive screens
- Add business logic comments in Portuguese
- Maintain trust-building UX (green success states, clear CTAs)
