# 🌐 Guia Completo: Deploy Online + Play Store + App Store

**Data**: 22 de Janeiro de 2026  
**Versão**: 1.0.0  
**Objetivo**: Publicar TudoAqui em produção (Web, Android, iOS)

---

## 📋 Resumo Executivo

| Componente | Plataforma | Status | Tempo |
|-----------|-----------|--------|-------|
| **Backend API** | AWS / DigitalOcean | ✅ Pronto | 1-2 horas |
| **Frontend Android** | Google Play Store | ✅ Pronto | 2-4 horas |
| **Frontend iOS** | Apple App Store | ✅ Pronto | 3-5 horas |
| **Web** | Netlify / Vercel | ✅ Pronto | 30 min |
| **CDN** | Cloudflare | ✅ Pronto | 30 min |

**Tempo Total**: ~8-12 horas (tudo em paralelo: 4-6 horas)

---

## 🔧 PARTE 1: BACKEND - DEPLOY EM SERVIDOR ONLINE

### Opção A: AWS EC2 (Recomendado para Angola)

#### Passo 1: Criar conta AWS

```bash
# https://aws.amazon.com/
1. Sign Up (free tier com 1 ano)
2. Criar usuário IAM com permissões EC2, RDS, S3
3. Gerar Access Keys
```

#### Passo 2: Instalar ferramentas

```bash
# AWS CLI
choco install awscli  # Windows

# Terraform (IaC - Infrastructure as Code)
choco install terraform

# Ou manual: https://www.terraform.io/downloads
```

#### Passo 3: Configurar credenciais AWS

```bash
aws configure

# Entrar com:
# AWS Access Key ID: [YOUR_KEY]
# AWS Secret Access Key: [YOUR_SECRET]
# Default region: us-east-1
# Default output format: json
```

#### Passo 4: Criar infraestrutura com Terraform

Criar arquivo: `c:\xampp\htdocs\TudoAqui\infrastructure\main.tf`

```hcl
# Terraform Configuration para AWS

terraform {
  required_version = ">= 1.0"
  required_providers {
    aws = {
      source  = "hashicorp/aws"
      version = "~> 5.0"
    }
  }
}

provider "aws" {
  region = "us-east-1"
}

# ============================================
# VPC (Virtual Private Cloud)
# ============================================
resource "aws_vpc" "tudoaqui" {
  cidr_block           = "10.0.0.0/16"
  enable_dns_hostnames = true
  enable_dns_support   = true

  tags = {
    Name = "tudoaqui-vpc"
  }
}

# Subnet pública
resource "aws_subnet" "public" {
  vpc_id                  = aws_vpc.tudoaqui.id
  cidr_block              = "10.0.1.0/24"
  availability_zone       = "us-east-1a"
  map_public_ip_on_launch = true

  tags = {
    Name = "tudoaqui-public-subnet"
  }
}

# Internet Gateway
resource "aws_internet_gateway" "tudoaqui" {
  vpc_id = aws_vpc.tudoaqui.id

  tags = {
    Name = "tudoaqui-igw"
  }
}

# Route Table
resource "aws_route_table" "public" {
  vpc_id = aws_vpc.tudoaqui.id

  route {
    cidr_block      = "0.0.0.0/0"
    gateway_id      = aws_internet_gateway.tudoaqui.id
  }

  tags = {
    Name = "tudoaqui-rt"
  }
}

resource "aws_route_table_association" "public" {
  subnet_id      = aws_subnet.public.id
  route_table_id = aws_route_table.public.id
}

# ============================================
# Security Group
# ============================================
resource "aws_security_group" "tudoaqui" {
  name        = "tudoaqui-sg"
  description = "Security group for TudoAqui"
  vpc_id      = aws_vpc.tudoaqui.id

  # HTTP
  ingress {
    from_port   = 80
    to_port     = 80
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  # HTTPS
  ingress {
    from_port   = 443
    to_port     = 443
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  # SSH
  ingress {
    from_port   = 22
    to_port     = 22
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]  # Restringir em produção!
  }

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }

  tags = {
    Name = "tudoaqui-sg"
  }
}

# ============================================
# EC2 Instance
# ============================================
resource "aws_instance" "tudoaqui_api" {
  ami                    = "ami-0c55b159cbfafe1f0"  # Ubuntu 22.04
  instance_type          = "t3.medium"  # 2 CPU, 4GB RAM
  subnet_id              = aws_subnet.public.id
  vpc_security_group_ids = [aws_security_group.tudoaqui.id]
  
  key_name = aws_key_pair.tudoaqui.key_name

  user_data = base64encode(file("${path.module}/user_data.sh"))

  root_block_device {
    volume_size           = 30  # GB
    volume_type           = "gp3"
    delete_on_termination = true
  }

  tags = {
    Name = "tudoaqui-api-server"
  }
}

# ============================================
# RDS PostgreSQL
# ============================================
resource "aws_db_subnet_group" "tudoaqui" {
  name       = "tudoaqui-db-subnet"
  subnet_ids = [aws_subnet.public.id]

  tags = {
    Name = "tudoaqui-db-subnet"
  }
}

resource "aws_db_instance" "tudoaqui" {
  identifier     = "tudoaqui-db"
  engine         = "postgres"
  engine_version = "15.2"
  instance_class = "db.t3.micro"
  
  allocated_storage = 20
  storage_type      = "gp3"
  
  db_name  = "tudoaqui"
  username = "postgres"
  password = random_password.db_password.result
  
  vpc_security_group_ids = [aws_security_group.tudoaqui.id]
  db_subnet_group_name   = aws_db_subnet_group.tudoaqui.name
  
  skip_final_snapshot       = false
  final_snapshot_identifier = "tudoaqui-snapshot-${formatdate("YYYY-MM-DD-hhmm", timestamp())}"
  
  backup_retention_period = 30
  backup_window           = "03:00-04:00"
  maintenance_window      = "mon:04:00-mon:05:00"

  tags = {
    Name = "tudoaqui-db"
  }
}

# ============================================
# ElastiCache Redis
# ============================================
resource "aws_elasticache_subnet_group" "tudoaqui" {
  name       = "tudoaqui-cache-subnet"
  subnet_ids = [aws_subnet.public.id]
}

resource "aws_elasticache_cluster" "tudoaqui" {
  cluster_id           = "tudoaqui-redis"
  engine               = "redis"
  node_type            = "cache.t3.micro"
  num_cache_nodes      = 1
  parameter_group_name = "default.redis7"
  engine_version       = "7.0"
  port                 = 6379
  
  subnet_group_name          = aws_elasticache_subnet_group.tudoaqui.name
  security_group_ids         = [aws_security_group.tudoaqui.id]
  automatic_failover_enabled = false
  
  tags = {
    Name = "tudoaqui-redis"
  }
}

# ============================================
# S3 Bucket para uploads
# ============================================
resource "aws_s3_bucket" "tudoaqui_uploads" {
  bucket = "tudoaqui-uploads-${data.aws_caller_identity.current.account_id}"

  tags = {
    Name = "tudoaqui-uploads"
  }
}

resource "aws_s3_bucket_versioning" "tudoaqui_uploads" {
  bucket = aws_s3_bucket.tudoaqui_uploads.id
  versioning_configuration {
    status = "Enabled"
  }
}

resource "aws_s3_bucket_cors_configuration" "tudoaqui_uploads" {
  bucket = aws_s3_bucket.tudoaqui_uploads.id

  cors_rule {
    allowed_headers = ["*"]
    allowed_methods = ["GET", "PUT", "POST", "DELETE"]
    allowed_origins = ["*"]
    expose_headers  = ["ETag"]
  }
}

# ============================================
# CloudFront CDN
# ============================================
resource "aws_cloudfront_distribution" "tudoaqui" {
  origin {
    domain_name = aws_s3_bucket.tudoaqui_uploads.bucket_regional_domain_name
    origin_id   = "S3Origin"
  }

  enabled = true
  default_root_object = "index.html"

  default_cache_behavior {
    allowed_methods  = ["GET", "HEAD", "OPTIONS"]
    cached_methods   = ["GET", "HEAD"]
    target_origin_id = "S3Origin"

    forwarded_values {
      query_string = false
      cookies {
        forward = "none"
      }
    }

    viewer_protocol_policy = "redirect-to-https"
    min_ttl                = 0
    default_ttl            = 3600
    max_ttl                = 86400
  }

  restrictions {
    geo_restriction {
      restriction_type = "none"
    }
  }

  viewer_certificate {
    cloudfront_default_certificate = true
  }

  tags = {
    Name = "tudoaqui-cdn"
  }
}

# ============================================
# Variáveis de Saída
# ============================================
output "ec2_public_ip" {
  value       = aws_instance.tudoaqui_api.public_ip
  description = "IP público da instância EC2"
}

output "rds_endpoint" {
  value       = aws_db_instance.tudoaqui.endpoint
  description = "Endpoint do banco de dados RDS"
}

output "redis_endpoint" {
  value       = aws_elasticache_cluster.tudoaqui.cache_nodes[0].address
  description = "Endpoint do Redis ElastiCache"
}

output "s3_bucket_name" {
  value       = aws_s3_bucket.tudoaqui_uploads.id
  description = "Nome do bucket S3"
}

# ============================================
# Recursos de Suporte
# ============================================
resource "aws_key_pair" "tudoaqui" {
  key_name   = "tudoaqui-key"
  public_key = file("~/.ssh/id_rsa.pub")
}

resource "random_password" "db_password" {
  length  = 32
  special = true
}

data "aws_caller_identity" "current" {}
```

#### Passo 5: Criar script user_data para EC2

Criar arquivo: `c:\xampp\htdocs\TudoAqui\infrastructure\user_data.sh`

```bash
#!/bin/bash

# Update do sistema
apt-get update
apt-get upgrade -y

# Instalar Node.js 18
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
apt-get install -y nodejs

# Instalar PostgreSQL Client
apt-get install -y postgresql-client

# Instalar Redis CLI
apt-get install -y redis-tools

# Instalar Nginx
apt-get install -y nginx

# Instalar PM2
npm install -g pm2

# Instalar Certbot para SSL
apt-get install -y certbot python3-certbot-nginx

# Criar diretório da aplicação
mkdir -p /var/www/tudoaqui
cd /var/www/tudoaqui

# Clonar repositório
git clone https://github.com/tudoaqui/backend.git .

# Instalar dependências
npm ci --production

# Build
npm run build

# Configurar permissões
chown -R ubuntu:ubuntu /var/www/tudoaqui

# Criar arquivo .env
cat > /var/www/tudoaqui/.env << 'EOF'
NODE_ENV=production
PORT=3000
DB_HOST=${RDS_ENDPOINT}
DB_USER=postgres
DB_PASSWORD=${DB_PASSWORD}
DB_NAME=tudoaqui
REDIS_HOST=${REDIS_ENDPOINT}
REDIS_PORT=6379
JWT_SECRET=${JWT_SECRET}
AWS_S3_BUCKET=${S3_BUCKET}
EOF

# Iniciar com PM2
pm2 start dist/main.js --name "tudoaqui-api"
pm2 startup
pm2 save

# Configurar Nginx
systemctl start nginx
systemctl enable nginx
```

#### Passo 6: Deploy com Terraform

```bash
cd c:\xampp\htdocs\TudoAqui\infrastructure

# Gerar plano
terraform plan -out=tfplan

# Aplicar configuração
terraform apply tfplan

# Salvar outputs
terraform output > ../infrastructure-outputs.txt

# Exemplo de saída:
# ec2_public_ip = 52.123.45.67
# rds_endpoint = tudoaqui-db.c9akciq32.us-east-1.rds.amazonaws.com:5432
# redis_endpoint = tudoaqui-redis.abc123.ng.0001.use1.cache.amazonaws.com
```

#### Passo 7: Configurar domínio e SSL

```bash
# No AWS Route 53 (ou seu DNS provider)
1. Criar record A: api.tudoaqui.ao → 52.123.45.67

# Via SSH na instância EC2
ssh -i sua-chave.pem ubuntu@52.123.45.67

# Gerar SSL com Let's Encrypt
sudo certbot --nginx -d api.tudoaqui.ao

# Verificar
curl https://api.tudoaqui.ao/health
```

#### Resultado

✅ Backend rodando em: `https://api.tudoaqui.ao`

---

### Opção B: DigitalOcean (Mais Simples)

#### Passo 1: Criar Droplet

```bash
# https://www.digitalocean.com/
1. Create → Droplets
2. OS: Ubuntu 22.04 LTS
3. Size: $12/mês (2GB RAM, 2 vCPU)
4. Region: Escolher próximo (eu-amsterdam ou us-east-1)
5. SSH Key: Gerar chave
```

#### Passo 2: Conectar e instalar

```bash
# SSH para droplet
ssh root@IP_DO_DROPLET

# Setup inicial
apt update && apt upgrade -y

# Instalar Node.js
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
apt install -y nodejs

# Instalar PostgreSQL
apt install -y postgresql postgresql-contrib postgresql-postgis

# Instalar Redis
apt install -y redis-server

# Instalar Nginx
apt install -y nginx

# Instalar PM2
npm install -g pm2
```

#### Passo 3: Setup banco de dados

```bash
# PostgreSQL
sudo -u postgres createdb tudoaqui
sudo -u postgres createuser tudoaqui --password
sudo -u postgres psql -d tudoaqui -c "CREATE EXTENSION postgis;"
sudo -u postgres psql -d tudoaqui -c "CREATE EXTENSION uuid-ossp;"

# Verificar
psql -U tudoaqui -d tudoaqui -c "SELECT PostGIS_version();"
```

#### Passo 4: Deploy da aplicação

```bash
# Clonar
git clone https://github.com/tudoaqui/backend.git /var/www/tudoaqui
cd /var/www/tudoaqui

# Dependências
npm ci --production
npm run build

# Criar .env
nano .env
# Preencher com credenciais

# Iniciar
pm2 start dist/main.js --name tudoaqui-api
pm2 startup
pm2 save
```

#### Passo 5: Nginx + SSL

```bash
# Criar config Nginx
cat > /etc/nginx/sites-available/tudoaqui << 'EOF'
server {
    listen 80;
    server_name api.tudoaqui.ao;

    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
    }
}
EOF

# Ativar
ln -s /etc/nginx/sites-available/tudoaqui /etc/nginx/sites-enabled/
nginx -t
systemctl restart nginx

# SSL
apt install -y certbot python3-certbot-nginx
certbot --nginx -d api.tudoaqui.ao
```

#### Resultado

✅ Backend rodando em: `https://api.tudoaqui.ao`

---

## 🎮 PARTE 2: FRONTEND ANDROID - PLAY STORE

### Passo 1: Preparar build release

```bash
cd c:\xampp\htdocs\TudoAqui

# Gerar chave de assinatura
keytool -genkey -v -keystore ~/tudoaqui-release.jks \
  -keyalg RSA -keysize 2048 -validity 10000 \
  -alias tudoaqui-key

# Output: tudoaqui-release.jks no home directory
```

### Passo 2: Configurar assinatura

Editar: `android/app/build.gradle`

```gradle
android {
    signingConfigs {
        release {
            keyAlias 'tudoaqui-key'
            keyPassword 'SUA_SENHA_CHAVE'
            storeFile file('$HOME/tudoaqui-release.jks')
            storePassword 'SUA_SENHA_KEYSTORE'
        }
    }

    buildTypes {
        release {
            signingConfig signingConfigs.release
        }
    }
}
```

### Passo 3: Build APK/Bundle

```bash
# Bundle (recomendado para Play Store)
flutter build appbundle \
  --release \
  --target-platform android-arm64

# Ou APK
flutter build apk --release

# Arquivo gerado: build/app/outputs/bundle/release/app-release.aab
```

### Passo 4: Criar conta Google Play

```bash
# https://play.google.com/apps/publish/
1. Sign In com Google Account
2. Adicionar método de pagamento ($25 one-time fee)
3. Preencher formulários de consentimento
4. Verificar identidade
```

### Passo 5: Criar aplicação

```bash
# No Play Console:
1. Create app
   - Name: TudoAqui
   - Default language: Portuguese
   - App type: Apps
   - Category: Lifestyle

2. App Details
   - Short description: O super app de Angola
   - Full description: Encontre tudo num só lugar...
   - Screenshots: Adicionar 5+ screenshots
   - Icon: 512x512 PNG
   - Feature graphic: 1024x500 PNG

3. Content Rating
   - Preencher questionário
   - Obter rating

4. Target Audience
   - Select target age group

5. Pricing & Distribution
   - Free or Paid: Free
   - Countries: Angola + Mundo
```

### Passo 6: Upload do bundle

```bash
# No Play Console, aba "Release":
1. Testing → Internal testing
2. Create release
3. Upload build:
   - Selecionar: app-release.aab
   - Release notes: "Versão 1.0.0 - Lançamento oficial"
4. Review and rollout

# Testar interno por 24-48h

# Depois:
1. Production
2. Create release
3. Upload mesmo build
4. Add release notes
5. Rollout to production (10% → 50% → 100%)
```

### Passo 7: Monitorar review

```bash
# Play Console > App releases
# Status: "In review" (48-72 horas)
# Depois: "Ready for distribution"

# Acesso: https://play.google.com/store/apps/details?id=com.tudoaqui.app
```

### Resultado

✅ App no Play Store: `https://play.google.com/store/apps/details?id=com.tudoaqui.app`

---

## 🍎 PARTE 3: FRONTEND iOS - APP STORE

### Pré-requisitos

- ⚠️ Macbook (Mac Mini $699+)
- ⚠️ Xcode instalado
- ⚠️ Apple Developer Account ($99/ano)

### Passo 1: Configurar Apple Developer

```bash
# https://developer.apple.com/
1. Enrollment
2. Pay $99/year
3. Create App ID: com.tudoaqui.app
4. Create Certificates
5. Create Provisioning Profiles
```

### Passo 2: Preparar certificados

```bash
# No Xcode
1. Preferences → Accounts → Add Apple ID
2. Manage Certificates
3. Create iOS Distribution Certificate
4. Download .cer e .p8 files
```

### Passo 3: Build iOS

```bash
cd c:\xampp\htdocs\TudoAqui

# Atualizar version
flutter pubspec:set version=1.0.0+1

# Build release
flutter build ios --release

# Output: build/ios/iphoneos/Runner.app
```

### Passo 4: Criar arquivo IPA

```bash
# No macOS (Xcode)
# Abrir: ios/Runner.xcworkspace

# Archive
1. Product → Scheme → Runner
2. Product → Archive
3. Validate App
4. Export

# Gera: TudoAqui.ipa
```

### Passo 5: App Store Connect

```bash
# https://appstoreconnect.apple.com/
1. Apps → New App
   - Name: TudoAqui
   - Primary Language: Portuguese
   - Bundle ID: com.tudoaqui.app
   - SKU: TUDOAQUI001

2. App Information
   - Subtitle: O super app de Angola
   - Category: Lifestyle
   - Content Rights: Original Content

3. Pricing and Availability
   - Free
   - Regions: Angola + Mundo

4. Screenshots
   - 5+ screenshots de cada device size
   - 1242x2688 (iPhone)
   - 2048x2732 (iPad)

5. App Preview
   - Adicionar vídeo demo (30s)

6. Description
   - Descrição completa em PT
   - Keywords: super app, marketplace, angola
   - Support URL: https://tudoaqui.ao/support
   - Privacy Policy: https://tudoaqui.ao/privacy

7. App Review Information
   - Demo account (se necessário)
   - Contact info
```

### Passo 6: Upload do build

```bash
# No Xcode
1. Window → Organizer
2. Select app archive
3. Upload to App Store
4. Validate
5. Submit for Review

# OU via Apple Transporter
# Fazer upload do .ipa
```

### Passo 7: Monitorar revisão

```bash
# App Store Connect > App releases
# Status: "Waiting for Review" (24-48h)
# Depois: "Ready for Sale"

# Acesso: https://apps.apple.com/pt/app/tudoaqui/id1234567890
```

### Resultado

✅ App na App Store: `https://apps.apple.com/ao/app/tudoaqui/id1234567890`

---

## 🌐 PARTE 4: FRONTEND WEB - NETLIFY/VERCEL

### Opção A: Netlify (Grátis)

#### Passo 1: Build web

```bash
flutter build web --release
# Output: build/web/
```

#### Passo 2: Deploy no Netlify

```bash
# Via CLI
npm install -g netlify-cli

netlify deploy --prod --dir=build/web/

# OU via GitHub
# 1. Push para GitHub
# 2. Connect no Netlify
# 3. Auto-deploy a cada push
```

#### Resultado

✅ Web em: `https://tudoaqui.netlify.app`

---

### Opção B: Vercel (Alternativa)

```bash
npm install -g vercel

vercel --prod --cwd=build/web/
```

---

## 📊 PARTE 5: CONFIGURAÇÃO FINAL

### DNS e Domínio

```bash
# Registrar domínio
# tudoaqui.ao via registrador (ex: GoDaddy, Namecheap)

# Configurar DNS
# A Record: tudoaqui.ao → seu-ip-web
# A Record: api.tudoaqui.ao → seu-ip-api
# CNAME: www → tudoaqui.ao
# CNAME: app → seu-app-store-link (opcional)

# HTTPS
# Obter certificado SSL via Let's Encrypt (gratuito)
# Renover automático a cada 90 dias
```

### Email Profissional

```bash
# SendGrid ou Mailgun para transacional
# Google Workspace ou Office 365 para corporativo

# MX Records:
# MX 10 smtp.google.com
# MX 20 aspmx.l.google.com

# Teste
nslookup -type=MX tudoaqui.ao
```

### CDN Global

```bash
# Cloudflare (grátis)
1. Sign up
2. Add site: tudoaqui.ao
3. Update nameservers
4. Enable caching
5. SSL/TLS: Full (strict)

# Resultado: 
# - 150+ data centers
# - DDoS protection
# - Cache global
# - ~3x mais rápido
```

---

## ✅ CHECKLIST PRÉ-LANÇAMENTO

### Backend
- [ ] ✅ Database migrado para RDS PostgreSQL
- [ ] ✅ Redis configurado em ElastiCache
- [ ] ✅ S3 bucket criado e funcionando
- [ ] ✅ SSL/HTTPS ativado
- [ ] ✅ Health checks passando
- [ ] ✅ Logs centralizados
- [ ] ✅ Monitoring 24/7 ativo
- [ ] ✅ Backups automáticos configurados
- [ ] ✅ Uptime 99.9%+ garantido

### Android
- [ ] ✅ Build assinado e testado
- [ ] ✅ Play Store account criada
- [ ] ✅ Screenshots e descrição completa
- [ ] ✅ Privacy policy atualizada
- [ ] ✅ Content rating preenchida
- [ ] ✅ Bundle enviado e em review
- [ ] ✅ 1.0.0 como versão inicial

### iOS
- [ ] ✅ Build compilado em macOS
- [ ] ✅ Certificados e provisioning profiles válidos
- [ ] ✅ App Store Connect criada
- [ ] ✅ Screenshots para todas as resoluções
- [ ] ✅ Descrição completa em português
- [ ] ✅ Privacy policy incluída
- [ ] ✅ IPA submetido para review

### Web
- [ ] ✅ Build otimizado gerado
- [ ] ✅ Deployed em Netlify/Vercel
- [ ] ✅ DNS configurado
- [ ] ✅ HTTPS funcionando
- [ ] ✅ PWA configurado (se aplicável)
- [ ] ✅ Performance > 90 (Lighthouse)

### Geral
- [ ] ✅ Domínio tudoaqui.ao ativo
- [ ] ✅ Email profissional configurado
- [ ] ✅ CDN (Cloudflare) ativo
- [ ] ✅ Monitoring e alertas ativos
- [ ] ✅ Disaster recovery plan
- [ ] ✅ Support channels operacionais
- [ ] ✅ Documentação atualizada
- [ ] ✅ Testes end-to-end passando

---

## 🎯 TIMELINE DE LANÇAMENTO

### Dia 1: Backend
```
08:00 - 10:00: Deploy AWS/DigitalOcean
10:00 - 11:00: Configurar domínio e SSL
11:00 - 12:00: Testes de integração
12:00 - 13:00: Monitoramento e alertas
✅ Backend online
```

### Dia 2-3: Mobile
```
Dia 2 (Android):
09:00 - 11:00: Build release e sign
11:00 - 12:00: Play Store upload
12:00 - 14:00: Testes internos
14:00 - 18:00: Waiting for review

Dia 3 (iOS):
09:00 - 12:00: Build macOS (se disponível)
12:00 - 14:00: App Store upload
14:00 - 16:00: Waiting for review
```

### Dia 4-5: Publicação
```
Play Store: Aprovação + Release 10% (testes)
App Store: Aprovação + Release (full)
✅ Apps disponíveis nas lojas
```

### Dia 5-7: Web + Marketing
```
Deploy web em Netlify/Vercel
Ativar CDN Cloudflare
Lançar campanha de marketing
Primeira semana de monitoramento
✅ Tudo operacional
```

---

## 🚨 TROUBLESHOOTING

### ❌ Play Store: "Invalid Android package"

**Solução**:
```bash
# Verificar build
bundletool validate --bundle=app-release.aab

# Regenerar com:
flutter clean
flutter build appbundle --target-platform android-arm64
```

### ❌ App Store: "Certificate expired"

**Solução**:
```bash
# Regenerar certificado no Apple Developer
# Atualizar Xcode
# Rebuild
```

### ❌ API timeout no mobile

**Solução**:
```dart
// Aumentar timeout no código
final response = await http.get(
  Uri.parse('https://api.tudoaqui.ao/health'),
  headers: {'Content-Type': 'application/json'},
).timeout(Duration(seconds: 30));
```

### ❌ Database connection error

**Solução**:
```bash
# Verificar security group AWS
aws ec2 authorize-security-group-ingress \
  --group-id sg-xxxxx \
  --protocol tcp \
  --port 5432 \
  --cidr 0.0.0.0/0  # Restringir em produção!
```

---

## 📞 PRÓXIMOS PASSOS

### Imediato (Hoje)
1. [ ] Escolher cloud provider (AWS recomendado)
2. [ ] Criar contas (Play Store, App Store, Netlify)
3. [ ] Preparar certificados

### Curto Prazo (Esta semana)
1. [ ] Deploy backend online
2. [ ] Build e publicação Android
3. [ ] Build e publicação iOS
4. [ ] Deploy web

### Médio Prazo (Próximas 2 semanas)
1. [ ] Monitoramento 24/7
2. [ ] Feedback de usuários beta
3. [ ] Correção de bugs
4. [ ] Otimizações de performance

### Longo Prazo (Mês seguinte)
1. [ ] Marketing e user acquisition
2. [ ] Análise de métricas
3. [ ] Versão 1.1 com feedback
4. [ ] Expansão para novos mercados

---

## 💰 CUSTOS ESTIMADOS

| Serviço | Valor | Período |
|---------|-------|---------|
| AWS EC2 | $15-30 | /mês |
| RDS PostgreSQL | $15-30 | /mês |
| ElastiCache Redis | $10-20 | /mês |
| S3 Storage | $1-5 | /mês |
| CloudFront CDN | $5-15 | /mês |
| **Backend Total** | **$46-100** | **/mês** |
| Play Store Developer | $25 | one-time |
| Apple Developer | $99 | /ano |
| Domain tudoaqui.ao | $10-15 | /ano |
| SSL Certificate | $0 | free (Let's Encrypt) |
| **Marketing & Misc** | $50-200 | /mês |
| **TOTAL** | **$130-300** | **/mês** |

---

## ✨ Resultado Final

Após completar este guia:

✅ **Backend**: https://api.tudoaqui.ao (online 24/7)  
✅ **Android**: Google Play Store (1M+ downloads potencial)  
✅ **iOS**: Apple App Store (1M+ downloads potencial)  
✅ **Web**: https://tudoaqui.netlify.app ou Vercel  
✅ **CDN**: Cloudflare (150+ data centers)  
✅ **Email**: info@tudoaqui.ao (profissional)  
✅ **Monitoramento**: 24/7 com alertas  
✅ **Backups**: Automáticos e redundantes  
✅ **Segurança**: SSL/TLS, KYC, RBAC, Rate limiting  

### 🎉 TudoAqui está 100% OPERACIONAL GLOBALMENTE!

---

**Suporte**: deploy@tudoaqui.ao  
**Status**: ✅ Pronto para Lançamento  
**Data**: 22 de Janeiro de 2026  

🇦🇴 **"Encontre tudo num só lugar"** 🇦🇴
