# Fluxo de Verificação de Documentos (KYC) - TudoAqui

## 📋 Visão Geral do Processo

O sistema KYC (Know Your Customer) do TudoAqui é a pedra angular da confiança na plataforma. Apenas vendedores verificados podem publicar anúncios, evitando fraude e burlas.

### Diagrama do Fluxo Completo

```
┌─────────────────────────────────────────────────────────────┐
│ 1. UTILIZADOR ABRE APP                                      │
│ └─> Redireciona para: LoginScreen                           │
└─────────────────────────────────────────────────────────────┘
                        │
                        ▼
┌─────────────────────────────────────────────────────────────┐
│ 2. LOGIN OU REGISTRO                                        │
│ ┌─────────────────────────────────────────────────────────┐ │
│ │ POST /auth/login                                        │ │
│ │ - Email: usuario@exemplo.com                            │ │
│ │ - Senha: (hash bcrypt no backend)                       │ │
│ └─────────────────────────────────────────────────────────┘ │
│ OU                                                          │
│ ┌─────────────────────────────────────────────────────────┐ │
│ │ POST /auth/register                                     │ │
│ │ - Nome, Email, Telefone, Senha                          │ │
│ │ - Toggle: Vendedor individual vs Empresa                │ │
│ └─────────────────────────────────────────────────────────┘ │
└─────────────────────────────────────────────────────────────┘
                        │
                        ▼
┌─────────────────────────────────────────────────────────────┐
│ 3. GERAÇÃO DE JWT TOKEN                                     │
│ └─> Backend retorna: { token: "eyJhbGc..." }               │
│    └─> Token armazenado em SharedPreferences (Flutter)      │
│    └─> Validade: 7 dias                                     │
└─────────────────────────────────────────────────────────────┘
                        │
                        ▼
┌─────────────────────────────────────────────────────────────┐
│ 4. REDIRECIONAMENTO PARA KYC                                │
│ Navigator.pushReplacementNamed('/verificacao')              │
│ └─> VerificacaoDocumentos()                                 │
└─────────────────────────────────────────────────────────────┘
                        │
                        ▼
┌─────────────────────────────────────────────────────────────┐
│ 5. PASSO 1: FOTO DO BI                                      │
│ ┌─────────────────────────────────────────────────────────┐ │
│ │ - Utilizador tira foto do Bilhete de Identidade         │ │
│ │ - Image Picker / Camera (image_picker package)          │ │
│ │ - Validação: Detecção de bordos automaticamente         │ │
│ └─────────────────────────────────────────────────────────┘ │
│                                                             │
│ POST /kyc/upload-bi (multipart/form-data)                  │
│ └─> Arquivo enviado ao backend                             │
│     └─> Adiciona marca d'água "TudoAqui"                   │
│     └─> Upload para S3 ou storage local                    │
│     └─> Chamada: Amazon Rekognition DetectText             │
│         └─> Extrai número do BI: "123456789LA043"          │
└─────────────────────────────────────────────────────────────┘
                        │
                        ▼
┌─────────────────────────────────────────────────────────────┐
│ 6. PASSO 2: SELFIE + LIVENESS DETECTION                     │
│ ┌─────────────────────────────────────────────────────────┐ │
│ │ - Utilizador tira selfie (frente para câmera)           │ │
│ │ - Instruções: "Sorria ou pisque os olhos"               │ │
│ │ - Detecção de movimento (anti-fraude)                   │ │
│ └─────────────────────────────────────────────────────────┘ │
│                                                             │
│ POST /kyc/upload-selfie (multipart/form-data)              │
│ └─> Arquivo enviado ao backend                             │
│     └─> Marca d'água "TudoAqui"                             │
│     └─> Chamada: Amazon Rekognition CompareFaces           │
│         └─> Compara rosto na selfie com foto do BI         │
│         └─> Score: Deve ser > 95% para aprovação           │
│     └─> Chamada: Amazon Rekognition DetectFaces            │
│         └─> Liveness Score: Verifica se é pessoa real      │
└─────────────────────────────────────────────────────────────┘
                        │
                        ▼
┌─────────────────────────────────────────────────────────────┐
│ 7. PASSO 3: ALVARÁ (OPCIONAL - APENAS EMPRESAS)            │
│ ┌─────────────────────────────────────────────────────────┐ │
│ │ - Se toggle "Empresa" está ativado                       │ │
│ │ - Upload do PDF do Alvará ou Diário da República         │ │
│ │ - OCR automático para extrair dados da empresa           │ │
│ └─────────────────────────────────────────────────────────┘ │
│                                                             │
│ POST /kyc/upload-alvara (multipart/form-data)              │
│ └─> Arquivo enviado ao backend                             │
│     └─> Marca d'água "TudoAqui"                             │
│     └─> OCR: Extrai número de registro, NUIT, etc          │
└─────────────────────────────────────────────────────────────┘
                        │
                        ▼
┌─────────────────────────────────────────────────────────────┐
│ 8. ANÁLISE COMPLETA COM IA                                  │
│ POST /kyc/analyze                                            │
│ └─> Backend chama Amazon Rekognition com todos os dados    │
│     └─> CompareFaces: BI vs Selfie (> 95%)                 │
│     └─> DetectText: Lê número do BI                        │
│     └─> DetectFaces: Valida liveness (> 90%)               │
│                                                             │
│ Resposta (mock):                                            │
│ {                                                           │
│   "aprovado": true,                                         │
│   "biNumero": "123456789LA043",                             │
│   "faceMatch": 0.98,                                        │
│   "livenessScore": 0.95,                                    │
│   "seloVerificado": true                                    │
│ }                                                           │
└─────────────────────────────────────────────────────────────┘
                        │
                        ▼
┌─────────────────────────────────────────────────────────────┐
│ 9. ATUALIZAÇÃO NO BANCO DE DADOS                            │
│                                                             │
│ UPDATE users SET                                            │
│   kyc_status = 'aprovado',                                  │
│   selo_verificado = true,                                   │
│   bi_numero = '123456789LA043'                              │
│ WHERE id = 'user123';                                       │
└─────────────────────────────────────────────────────────────┘
                        │
                        ▼
┌─────────────────────────────────────────────────────────────┐
│ 10. NOTIFICAÇÃO PUSH (FIREBASE)                             │
│ POST /firebase/send-notification                            │
│ {                                                           │
│   "title": "Parabéns!",                                     │
│   "body": "Sua conta foi verificada ✓"                      │
│ }                                                           │
└─────────────────────────────────────────────────────────────┘
                        │
                        ▼
┌─────────────────────────────────────────────────────────────┐
│ 11. REDIRECIONAMENTO PARA HOME                              │
│ Navigator.pushReplacementNamed('/home')                     │
│ └─> HomeMarketplace()                                       │
│     └─> Perfil exibe: Selo Verificado ✅                    │
│     └─> Pode publicar anúncios                              │
│     └─> Badge "Vendedor Verificado" nos anúncios           │
└─────────────────────────────────────────────────────────────┘
```

---

## 🔐 Camadas de Segurança

### 1. **Nível Cliente (Flutter)**
```dart
// Compression antes de upload
Image -> Compressed (JPEG 80%) -> Uploaded

// Certificate Pinning
HttpClient().badCertificateCallback = ...

// Armazenamento de Token
SharedPreferences.getInstance()
  .setString('auth_token', token);
```

### 2. **Nível Transporte (HTTPS + TLS 1.3)**
- Toda comunicação encriptada
- Certificate pinning (verificação de domínio)
- Proteção contra MITM (Man-in-the-Middle)

### 3. **Nível Autenticação (JWT)**
```
Header: { "alg": "HS256", "typ": "JWT" }
Payload: { 
  "sub": "user123", 
  "email": "user@exemplo.com",
  "iat": 1234567890,
  "exp": 1234654290  // 7 dias depois
}
Signature: HMAC-SHA256(header.payload, secret)
```

### 4. **Nível Autorização (Bearer Token)**
```http
GET /api/v1/kyc/status/user123
Authorization: Bearer eyJhbGc...
```

### 5. **Nível Dados (Amazon Rekognition)**
- **DetectText:** OCR com 99% de precisão
- **CompareFaces:** Face matching > 95%
- **DetectFaces + Liveness:** Anti-spoofing (detecção de foto de foto)

### 6. **Nível Armazenamento (PostgreSQL Encrypted)**
```sql
-- Campos sensíveis encriptados
ALTER TABLE users ADD COLUMN bi_numero_encrypted BYTEA;
-- Decryption apenas quando necessário
SELECT pgp_sym_decrypt(bi_numero_encrypted, 'encryption_key');
```

### 7. **Nível Metadata (Marca d'água Automática)**
- Todas as fotos recebem watermark "TudoAqui"
- Impede reutilização em outras plataformas
- Visível em impressões

---

## 🛡️ Proteção Contra Fraude

### Cenário 1: "Rouba BI de outra pessoa"
```
❌ Bloqueado por Liveness Detection
   - Amazon Rekognition detecta que a selfie 
     é diferente do rosto no BI
   - Falha no CompareFaces (score < 95%)
```

### Cenário 2: "Coloca foto impressa do BI"
```
❌ Bloqueado por Liveness Detection
   - Detecção de plano (2D vs 3D)
   - Amazon Rekognition rejeita
```

### Cenário 3: "Cria múltiplas contas com mesmo BI"
```
❌ Bloqueado por Sistema de Denúncia IA
   - BI número único indexado
   - SELECT COUNT(*) FROM users WHERE bi_numero = 'X'
   - Se > 1, marca todas como fraude
```

### Cenário 4: "Tira screenshot de aprovação"
```
❌ Dados confidenciais não aparecem no app
   - Frontend mostra apenas: "Verificado ✓"
   - Dados sensíveis guardados apenas no backend
```

---

## 📊 Estados KYC

| Estado | Descrição | Ações Permitidas |
|--------|-----------|-----------------|
| `pendente` | Esperando documentos | Nenhuma (bloqueado) |
| `em_analise` | Documentos enviados | Nenhuma (bloqueado) |
| `aprovado` | Verificação bem-sucedida | ✅ Publicar anúncios |
| `rejeitado` | Verificação falhou | Reenviar documentos |
| `suspeito` | Padrão de fraude detectado | ⛔ Bloqueado (contatar suporte) |

---

## 🚀 Próximas Implementações

- [ ] Integração real com Amazon Rekognition SDK
- [ ] Firebase Cloud Messaging para notificações
- [ ] Armazenamento S3 para fotos de documentos
- [ ] Rate limiting para upload (máx 5 tentativas/hora)
- [ ] Dashboard de análise (vendedores fraude)
- [ ] 2FA via SMS (operadoras angolanas)
- [ ] Renovação automática de verificação (anual)

---

**Segurança em Nível Bancário para o Mercado Angolano** 🇦🇴
