# 📦 Guia de Publicação - TudoAqui

**Data**: 22 de Janeiro de 2026  
**Versão**: 1.0.0  
**Status**: ✅ Pronto para Publicação

---

## 📋 Opções de Publicação

| Plataforma | Tipo | Dificuldade | Tempo |
|-----------|------|------------|--------|
| **GitHub** | Código-fonte | ⭐ Fácil | 5 min |
| **npm Registry** | Pacote Node.js | ⭐⭐ Médio | 10 min |
| **Docker Hub** | Container | ⭐⭐ Médio | 15 min |
| **AWS** | Cloud Completo | ⭐⭐⭐ Complexo | 30 min |
| **Heroku** | PaaS Rápido | ⭐⭐ Médio | 10 min |
| **DigitalOcean** | VPS | ⭐⭐⭐ Complexo | 45 min |

---

## 🚀 Opção 1: Publicar no GitHub

### Passo 1: Criar repositório

```bash
cd c:\xampp\htdocs\TudoAqui

# Inicializar git (se não tiver)
git init

# Adicionar remote
git remote add origin https://github.com/tudoaqui/backend.git

# Verificar remote
git remote -v
```

### Passo 2: Preparar para publicação

```bash
# Verificar status
git status

# Adicionar todos os arquivos
git add .

# Fazer commit
git commit -m "feat: TudoAqui v1.0.0 - Backend completo com 8 módulos integrados"

# Push para main
git push -u origin main
```

### Passo 3: Adicionar informações do projeto

```bash
# Editar README.md (já pronto em: README.md)
# Editar package.json (já pronto)
# Editar LICENSE (adicionar MIT License)
```

### Passo 4: Criar releases

```bash
# Tag de versão
git tag -a v1.0.0 -m "Release v1.0.0 - Production Ready"

# Push tags
git push origin v1.0.0

# Ver todas as tags
git tag -l
```

### Resultado

✅ Repositório público em: `https://github.com/tudoaqui/backend`

---

## 📦 Opção 2: Publicar no npm Registry

### Pré-requisitos

```bash
# Criar conta em npmjs.com (gratuito)
# https://www.npmjs.com/signup

# Login no npm
npm login
# Entrar com: username, password, email
```

### Publicar pacote

```bash
cd c:\xampp\htdocs\TudoAqui\backend

# Verificar package.json
cat package.json

# Publicar
npm publish --public

# Verificar publicação
npm view tudoaqui-backend

# Instalar de qualquer lugar
npm install tudoaqui-backend
```

### package.json (verificar)

```json
{
  "name": "tudoaqui-backend",
  "version": "1.0.0",
  "description": "TudoAqui Super App - Backend Microservices",
  "repository": {
    "type": "git",
    "url": "https://github.com/tudoaqui/backend.git"
  },
  "main": "dist/main.js",
  "types": "dist/main.d.ts",
  "files": ["dist"],
  "keywords": [
    "tudoaqui",
    "super-app",
    "angola",
    "marketplace",
    "nestjs",
    "microservices"
  ]
}
```

### Atualizar versão

```bash
# Patch (1.0.0 → 1.0.1)
npm version patch

# Minor (1.0.0 → 1.1.0)
npm version minor

# Major (1.0.0 → 2.0.0)
npm version major

# Publicar
npm publish
```

### Resultado

✅ Pacote público em: `https://www.npmjs.com/package/tudoaqui-backend`

---

## 🐳 Opção 3: Publicar no Docker Hub

### Pré-requisitos

```bash
# Criar conta em hub.docker.com (gratuito)
# https://hub.docker.com/signup

# Login no Docker
docker login
# Entrar com: username, password
```

### Build da imagem

```bash
cd c:\xampp\htdocs\TudoAqui\backend

# Build local
docker build -t tudoaqui-backend:1.0.0 .

# Tag para Docker Hub
docker tag tudoaqui-backend:1.0.0 tudoaqui/backend:1.0.0
docker tag tudoaqui-backend:1.0.0 tudoaqui/backend:latest
```

### Publicar imagem

```bash
# Push para Docker Hub
docker push tudoaqui/backend:1.0.0
docker push tudoaqui/backend:latest

# Verificar publicação
docker pull tudoaqui/backend:latest
```

### Dockerfile (verificar)

```dockerfile
FROM node:18-alpine

WORKDIR /app

COPY package*.json ./

RUN npm ci --production

COPY dist/ ./dist/

ENV NODE_ENV=production
ENV PORT=3000

EXPOSE 3000

CMD ["node", "dist/main"]
```

### Resultado

✅ Imagem pública em: `https://hub.docker.com/r/tudoaqui/backend`

---

## ☁️ Opção 4: Deploy em AWS (EC2 + RDS + S3)

### Passo 1: Preparar AWS

```bash
# Criar conta AWS (free tier disponível)
# https://aws.amazon.com/free

# Instalar AWS CLI
# https://aws.amazon.com/cli/

# Configurar credenciais
aws configure
# Entrar com: AWS Access Key ID, AWS Secret Access Key, Region (us-east-1)
```

### Passo 2: Criar infraestrutura

```bash
# Opção A: Via AWS Console (manual)
1. EC2 → Instances → Launch Instance
2. RDS → Databases → Create database
3. S3 → Buckets → Create bucket
4. VPC → Security Groups → Configure

# Opção B: Via Terraform (IaC)
# Criar arquivo: terraform/main.tf
```

### Passo 3: Deploy via GitHub Actions

Criar arquivo: `.github/workflows/deploy.yml`

```yaml
name: Deploy to AWS

on:
  push:
    branches: [main]

jobs:
  deploy:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3
      
      - name: Build
        run: npm run build
      
      - name: Deploy to EC2
        env:
          AWS_PRIVATE_KEY: ${{ secrets.AWS_PRIVATE_KEY }}
        run: |
          aws ec2-instance-connect send-ssh-public-key \
            --os-user ec2-user \
            --ssh-public-key-body "file://${{ secrets.SSH_PUBLIC_KEY }}" \
            --instance-id ${{ secrets.AWS_INSTANCE_ID }}
          
          ssh -i ${{ secrets.SSH_PRIVATE_KEY }} \
            ec2-user@${{ secrets.AWS_INSTANCE_IP }} \
            "cd /var/www/tudoaqui && ./deploy.sh"
```

### Passo 4: Deploy manual

```bash
# SSH para EC2
ssh -i seu-key.pem ec2-user@seu-ip-publico

# No servidor
cd /var/www/tudoaqui
git pull origin main
npm ci --production
npm run build
pm2 restart tudoaqui-api
```

### Resultado

✅ App rodando em: `https://api.tudoaqui.ao`

---

## 🚀 Opção 5: Deploy em Heroku (Rápido)

### Passo 1: Preparar Heroku

```bash
# Criar conta em heroku.com (gratuito com limites)
# https://www.heroku.com/

# Instalar Heroku CLI
# https://devcenter.heroku.com/articles/heroku-cli

# Login
heroku login
```

### Passo 2: Criar app

```bash
cd c:\xampp\htdocs\TudoAqui\backend

# Criar app Heroku
heroku create tudoaqui-api

# Ou usar app existente
heroku apps:create --remote heroku
```

### Passo 3: Adicionar database

```bash
# PostgreSQL Heroku
heroku addons:create heroku-postgresql:standard-0 --app tudoaqui-api

# Redis Heroku
heroku addons:create heroku-redis:premium-0 --app tudoaqui-api
```

### Passo 4: Configurar variáveis

```bash
# Definir ENV vars
heroku config:set NODE_ENV=production --app tudoaqui-api
heroku config:set JWT_SECRET=seu_secret_super_seguro --app tudoaqui-api

# Verificar
heroku config --app tudoaqui-api
```

### Passo 5: Deploy

```bash
# Via Git
git push heroku main

# Ou via Docker
heroku container:push web --app tudoaqui-api
heroku container:release web --app tudoaqui-api
```

### Resultado

✅ App rodando em: `https://tudoaqui-api.herokuapp.com`

---

## 🌐 Opção 6: Deploy em DigitalOcean (VPS)

### Passo 1: Criar droplet

```bash
# https://www.digitalocean.com/
# 1. Create → Droplets
# 2. Escolher OS: Ubuntu 22.04 LTS
# 3. Tamanho: $6/mês (2GB RAM, 1vCPU)
# 4. SSH Key setup
```

### Passo 2: Conectar via SSH

```bash
# SSH para droplet
ssh root@seu-ip-do-droplet

# Ou com chave
ssh -i sua-chave root@seu-ip-do-droplet
```

### Passo 3: Setup inicial

```bash
# Update do sistema
apt update && apt upgrade -y

# Instalar Node.js
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
apt install -y nodejs

# Instalar PostgreSQL
apt install -y postgresql postgresql-contrib

# Instalar Redis
apt install -y redis-server

# Instalar Nginx
apt install -y nginx

# Instalar PM2
npm install -g pm2

# Instalar Certbot (SSL)
apt install -y certbot python3-certbot-nginx
```

### Passo 4: Setup do banco

```bash
# PostgreSQL
sudo -u postgres psql
CREATE DATABASE tudoaqui;
CREATE USER tudoaqui WITH PASSWORD 'sua_senha_segura';
GRANT ALL PRIVILEGES ON DATABASE tudoaqui TO tudoaqui;
\q

# PostGIS
sudo -u postgres psql -d tudoaqui -c "CREATE EXTENSION postgis;"
```

### Passo 5: Deploy da aplicação

```bash
# Clonar repositório
git clone https://github.com/tudoaqui/backend.git /var/www/tudoaqui
cd /var/www/tudoaqui

# Instalar dependências
npm ci --production

# Build
npm run build

# Iniciar com PM2
pm2 start dist/main.js --name "tudoaqui-api"
pm2 startup
pm2 save
```

### Passo 6: Configurar Nginx

Criar: `/etc/nginx/sites-available/tudoaqui`

```nginx
server {
    listen 80;
    server_name api.tudoaqui.ao;

    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
    }
}
```

```bash
# Ativar site
ln -s /etc/nginx/sites-available/tudoaqui /etc/nginx/sites-enabled/
nginx -t
systemctl restart nginx
```

### Passo 7: SSL com Let's Encrypt

```bash
certbot --nginx -d api.tudoaqui.ao
# Seguir instruções
```

### Resultado

✅ App rodando em: `https://api.tudoaqui.ao`

---

## 📋 Checklist de Publicação

### Antes de Publicar

- [ ] ✅ Todos os testes passando (`npm test`)
- [ ] ✅ Build sem erros (`npm run build`)
- [ ] ✅ Cobertura de testes 85%+
- [ ] ✅ README.md atualizado
- [ ] ✅ CHANGELOG.md atualizado
- [ ] ✅ LICENSE incluído
- [ ] ✅ .gitignore configurado
- [ ] ✅ .env.example pronto
- [ ] ✅ Documentação atualizada (Swagger)
- [ ] ✅ Vulnerabilidades corrigidas (`npm audit`)

### Configurações Essenciais

- [ ] ✅ JWT_SECRET definido e seguro
- [ ] ✅ Database URL configurada
- [ ] ✅ Redis URL configurada
- [ ] ✅ AWS S3 credenciais (se usar)
- [ ] ✅ Firebase credenciais (se usar)
- [ ] ✅ SendGrid API key (se usar)
- [ ] ✅ Africa's Talking key (se usar)
- [ ] ✅ CORS configurado
- [ ] ✅ HTTPS ativado

### Após Publicar

- [ ] ✅ Health check funcionando
- [ ] ✅ Swagger UI acessível
- [ ] ✅ Endpoints respondendo
- [ ] ✅ Database conectado
- [ ] ✅ Cache funcionando
- [ ] ✅ Notificações enviando
- [ ] ✅ Logs sendo capturados
- [ ] ✅ Monitoring ativo
- [ ] ✅ Backups configurados
- [ ] ✅ DNS apontando

---

## 🔒 Segurança na Publicação

### Secrets & Variáveis

```bash
# Nunca commitar:
.env
.env.local
.env.production
secrets.json
api-keys.txt
ssh-keys/

# GitHub Secrets (para CI/CD):
Settings → Secrets → New repository secret
- AWS_ACCESS_KEY_ID
- AWS_SECRET_ACCESS_KEY
- DATABASE_URL
- JWT_SECRET
- etc
```

### SSL/TLS

```bash
# Obrigatório em produção
# Let's Encrypt (gratuito):
sudo certbot certonly --standalone -d api.tudoaqui.ao

# Ou via Nginx:
sudo certbot --nginx -d api.tudoaqui.ao
```

### Rate Limiting

```nginx
# Nginx config
limit_req_zone $binary_remote_addr zone=api_limit:10m rate=100r/m;

server {
    location /api/ {
        limit_req zone=api_limit burst=200;
        proxy_pass http://localhost:3000;
    }
}
```

---

## 📊 Monitoramento Pós-Deploy

### Health Checks

```bash
# Verificar status
curl https://api.tudoaqui.ao/health

# Verificar detalhes
curl https://api.tudoaqui.ao/health/detailed

# Uptime
curl -I https://api.tudoaqui.ao
```

### Logs

```bash
# Ver logs em tempo real
pm2 logs tudoaqui-api

# Ou via Docker
docker logs -f tudoaqui-api

# Ou via Heroku
heroku logs --tail --app tudoaqui-api
```

### Métricas

```bash
# Prometheus
curl https://api.tudoaqui.ao/metrics

# Grafana
https://seu-dominio/grafana
```

---

## 🎯 Timeline de Deploy

### Fase 1: Desenvolvimento (Concluída ✅)
- Semana 1-4: Backend completo
- Semana 3-4: Testes e documentação
- Status: **PRONTO**

### Fase 2: Staging (Próxima)
- Deploy em servidor de teste
- Teste de carga
- Teste de segurança
- Duração: 1 semana

### Fase 3: Produção (Após Staging)
- Deploy em production
- Monitoramento 24/7
- Backup diário
- Duração: Contínuo

### Fase 4: Frontend (Paralelo)
- Flutter development
- Mobile testing
- Web deployment
- Duração: 2-4 semanas

---

## 📞 Suporte Pós-Deploy

### Contatos

- **Email**: deploy@tudoaqui.ao
- **WhatsApp**: +244 923 123 456
- **GitHub Issues**: https://github.com/tudoaqui/backend/issues
- **Slack**: #tudoaqui-ops

### SLA (Service Level Agreement)

- ⚡ P1 (Crítico): Resposta em 1 hora
- ⚡ P2 (Alto): Resposta em 4 horas
- ⚡ P3 (Médio): Resposta em 24 horas
- ⚡ P4 (Baixo): Resposta em 5 dias

---

## 🎉 Próximos Passos

1. ✅ **Escolher plataforma** (recomendado: AWS ou DigitalOcean)
2. ✅ **Preparar infraestrutura** (DNS, SSL, Database, Cache)
3. ✅ **Configurar CI/CD** (GitHub Actions, GitLab CI)
4. ✅ **Primeiro deploy** (staging)
5. ✅ **Testes de integração** (end-to-end)
6. ✅ **Deploy em produção** (com monitoramento)
7. ✅ **Ativação do frontend** (Flutter)
8. ✅ **Beta testing** (com usuários reais)

---

**Pronto para publicar?** 🚀

Escolha uma das opções acima e execute os passos!

Dúvidas? Contate: suporte@tudoaqui.ao

---

**TudoAqui v1.0.0**  
**"Encontre tudo num só lugar"** 🇦🇴
