# 🚀 Guia Rápido de Deploy - Passo a Passo

**Status**: ✅ Todos os ficheiros prontos para deploy  
**Data**: 22 de Janeiro de 2026

---

## 📋 Checklist Antes de Começar

- [ ] Conta AWS criada (free tier)
- [ ] GitHub account com repositório criado
- [ ] Certificados gerados localmente
- [ ] Variables preenchiidas em `terraform.tfvars`

---

## 🔧 PASSO 1: Gerar Certificados e Chaves

### Windows (PowerShell)

```powershell
# Executar script de certificados
.\scripts\certificates.ps1

# Output esperado:
# ✅ SSH key gerada: C:\Users\seu_user\.ssh\tudoaqui-aws
# ✅ Keystore gerada: C:\Users\seu_user\tudoaqui-release.jks
# ✅ Secrets salvos em: .\certificates\secrets.txt
```

**Ficheiros gerados:**
- `~/.ssh/tudoaqui-aws` (chave privada SSH)
- `~/.ssh/tudoaqui-aws.pub` (chave pública)
- `~/tudoaqui-release.jks` (keystore Android)
- `./certificates/secrets.txt` (secrets para .env)

---

## 🌐 PASSO 2: Setup AWS com Terraform

### 2.1 Instalar Terraform

```powershell
# Windows
choco install terraform

# Verificar
terraform --version
```

### 2.2 Preencher variáveis

Editar: `infrastructure/terraform.tfvars`

```hcl
# Substitua estes valores:
aws_region       = "us-east-1"
db_password      = "SUA_SENHA_FORTE_32_CHARS"
jwt_secret       = "SEU_JWT_SECRET_64_CHARS"

tags = {
  Owner = "Seu Nome"
}
```

### 2.3 Inicializar Terraform

```bash
cd infrastructure

# Baixar providers
terraform init

# Validar configuração
terraform validate

# Planejar infraestrutura
terraform plan -out=tfplan
```

### 2.4 Aplicar configuração

```bash
# Aplicar
terraform apply tfplan

# Esperar ~5-10 minutos

# Output será exibido:
# ec2_public_ip = 52.123.45.67
# rds_endpoint = tudoaqui-db.c9akciq32.us-east-1.rds.amazonaws.com
```

---

## 📝 PASSO 3: Configurar .env

### 3.1 Copiar template

```bash
cp .env.example .env
```

### 3.2 Preencher valores

Usar outputs do Terraform:

```bash
# Abrir ficheiro
nano .env

# Actualizar com:
DB_HOST=tudoaqui-db.c9akciq32.us-east-1.rds.amazonaws.com
DB_PASSWORD=SUA_SENHA_FORTE
REDIS_HOST=tudoaqui-redis.abc123.ng.0001.use1.cache.amazonaws.com
JWT_SECRET=seu_jwt_secret
AWS_S3_BUCKET=tudoaqui-uploads-123456789
```

### 3.3 Gerar secrets faltando

```bash
# De volta ao ficheiro secrets.txt
cat ./certificates/secrets.txt

# Copiar JWT_SECRET e WEBHOOK_SECRET para .env
```

---

## 🔐 PASSO 4: Setup SSH e Deploy Inicial

### 4.1 Copiar SSH key para AWS

```powershell
# Ler chave pública
Get-Content $HOME\.ssh\tudoaqui-aws.pub

# Copiar e colar na AWS Console:
# 1. EC2 > Key Pairs > Import
# 2. Cole a chave pública
# 3. Name: tudoaqui-key
```

### 4.2 Conectar via SSH

```bash
# SSH com chave privada
ssh -i ~/.ssh/tudoaqui-aws ubuntu@52.123.45.67

# Esperar cloud-init finalizar (~5 min)
# Ver logs:
tail -f /var/log/tudoaqui-deploy.log
```

### 4.3 Verificar aplicação

```bash
# Na instância EC2
pm2 status
pm2 logs

# Testar API local
curl http://localhost:3000/health
```

---

## 🌍 PASSO 5: Configurar Domínio e SSL

### 5.1 Configurar DNS

Em seu registrador de domínio (ex: GoDaddy):

```
DNS Records:
A record: tudoaqui.ao → 52.123.45.67
A record: api.tudoaqui.ao → 52.123.45.67
CNAME: www.tudoaqui.ao → tudoaqui.ao
```

Verificar:
```bash
nslookup api.tudoaqui.ao
```

### 5.2 Gerar certificado SSL

Na instância EC2:

```bash
# SSH para instância
ssh -i ~/.ssh/tudoaqui-aws ubuntu@52.123.45.67

# Gerar certificado com Let's Encrypt
sudo /usr/local/bin/setup-ssl.sh api.tudoaqui.ao

# Auto-renew a cada 90 dias
sudo systemctl enable certbot.timer
```

### 5.3 Verificar HTTPS

```bash
curl -I https://api.tudoaqui.ao/health

# Deve retornar 200 OK
```

---

## 🎮 PASSO 6: Deploy Android (Play Store)

### 6.1 Build release

```bash
flutter build appbundle --release

# Output: build/app/outputs/bundle/release/app-release.aab
```

### 6.2 Google Play Console

```
1. https://play.google.com/apps/publish/
2. Create app
3. Nome: TudoAqui
4. Categoria: Lifestyle
5. Preencher store listing
6. Upload app-release.aab
7. Submeter para review
```

**Timeline**: 48-72 horas para aprovação

---

## 🍎 PASSO 7: Deploy iOS (App Store)

⚠️ **Requer Mac!**

### 7.1 Build em macOS

```bash
flutter build ios --release

# Gerar IPA via Xcode
cd ios
xcodebuild -workspace Runner.xcworkspace -scheme Runner -configuration Release archive
```

### 7.2 App Store Connect

```
1. https://appstoreconnect.apple.com/
2. Apps > New App
3. Name: TudoAqui
4. Bundle ID: com.tudoaqui.app
5. Preencher app information
6. Upload IPA
7. Submeter para review
```

**Timeline**: 24-48 horas para aprovação

---

## 🌐 PASSO 8: CI/CD com GitHub Actions

### 8.1 Criar repositório

```bash
git init
git add .
git commit -m "Initial commit"
git remote add origin https://github.com/seu_username/tudoaqui.git
git branch -M main
git push -u origin main
```

### 8.2 Configurar secrets no GitHub

No repositório GitHub:

```
Settings > Secrets and variables > Actions > New repository secret
```

Adicionar:

```
AWS_ACCESS_KEY_ID = seu_access_key
AWS_SECRET_ACCESS_KEY = seu_secret_key
AWS_ACCOUNT_ID = 123456789
EC2_STAGING_HOST = staging-api.tudoaqui.ao
EC2_PROD_HOST = api.tudoaqui.ao
EC2_SSH_KEY_STAGING = [chave SSH]
EC2_SSH_KEY_PROD = [chave SSH]
GOOGLE_PLAY_SERVICE_ACCOUNT_JSON = [JSON do Google Play]
APPLE_ID = seu_apple_id
APPLE_APP_SPECIFIC_PASSWORD = [app password]
SLACK_WEBHOOK_URL = https://hooks.slack.com/...
```

### 8.3 Workflows automáticos

Ficheiros já criados:

- `.github/workflows/deploy-backend.yml` - Deploy automático
- `.github/workflows/deploy-mobile.yml` - Build mobile

Triggers:

```yaml
# Deploy Backend
- Push para main → Deploy production
- Push para develop → Deploy staging
- Pull request → Testes

# Deploy Mobile
- Push para main com changes em lib/, android/, ios/ → Build
```

---

## 🔍 PASSO 9: Monitoramento e Alertas

### 9.1 Logs em tempo real

```bash
# Na instância EC2
pm2 logs tudoaqui-api

# Ou via AWS CloudWatch
aws logs tail /tudoaqui/api --follow
```

### 9.2 Health checks

```bash
# Configurar alarmes no AWS CloudWatch
1. CloudWatch > Alarms > Create Alarm
2. Métrica: api.tudoaqui.ao/health
3. Threshold: 5xx errors > 10 por minuto
4. Action: SNS notification
```

### 9.3 Backups automáticos

Já configurado em Terraform:

```hcl
backup_retention_period = 30  # 30 dias
backup_window = "03:00-04:00"  # 3AM UTC
```

---

## ✅ Checklist Final

Antes de lançar:

- [ ] Backend online em https://api.tudoaqui.ao
- [ ] Health check respondendo (200 OK)
- [ ] Swagger docs em /api-docs
- [ ] Database migrations executadas
- [ ] Testes passando
- [ ] Monitoring ativo
- [ ] Logs centralizados
- [ ] Backups configurados
- [ ] App Android no Play Store (em review)
- [ ] App iOS na App Store (em review)
- [ ] Web deployado em Netlify/Vercel
- [ ] Email profissional funcionando
- [ ] SSL/HTTPS em todos os domínios
- [ ] CDN Cloudflare ativo

---

## 🚨 Troubleshooting

### ❌ "psql: connection refused"

```bash
# Na instância EC2
aws rds describe-db-instances --query "DBInstances[0].Endpoint"

# Conectar via RDS endpoint
psql -h tudoaqui-db.xxx.rds.amazonaws.com -U postgres -d tudoaqui
```

### ❌ "Docker: image not found"

```bash
# Fazer login no ECR
aws ecr get-login-password | docker login --username AWS --password-stdin 123456789.dkr.ecr.us-east-1.amazonaws.com

# Pull manualmente
docker pull 123456789.dkr.ecr.us-east-1.amazonaws.com/tudoaqui:latest
```

### ❌ "Certificate not found"

```bash
# Regenerar SSL
sudo /usr/local/bin/setup-ssl.sh api.tudoaqui.ao

# Recarregar Nginx
sudo systemctl reload nginx
```

---

## 📞 Suporte

- **Docs**: [DEPLOY_ONLINE_STORES.md](DEPLOY_ONLINE_STORES.md)
- **Logs**: `/var/log/tudoaqui/`
- **Email**: deploy@tudoaqui.ao
- **Status**: https://status.tudoaqui.ao

---

## 🎉 Resultado Final

Após completar todos os passos:

✅ **Backend**: https://api.tudoaqui.ao (rodando 24/7)  
✅ **Android**: Google Play Store (públicamente disponível)  
✅ **iOS**: Apple App Store (públicamente disponível)  
✅ **Web**: https://tudoaqui.ao (via Netlify/Vercel)  
✅ **Email**: info@tudoaqui.ao (profissional)  
✅ **Monitoramento**: 24/7 com alertas automáticos  
✅ **Backups**: Automáticos e redundantes  
✅ **Segurança**: SSL, KYC, RBAC, Rate limiting  

### 🇦🇴 **TudoAqui está 100% OPERACIONAL!** 🇦🇴

---

**Tempo total estimado**: 4-6 horas (tudo em paralelo)

🚀 **Bom deployment!**
