# ============================================
# SETUP PRODUÇÃO COMPLETO - TudoAqui
# ============================================
# Este script automatiza:
# 1. Geração de certificados e chaves
# 2. Configuração do Terraform
# 3. Setup de GitHub Secrets
# 4. Deploy automático

# Cores para output
$colors = @{
    Green = 'Green'
    Red = 'Red'
    Yellow = 'Yellow'
    Cyan = 'Cyan'
}

function Write-Status {
    param([string]$Message, [string]$Status = "INFO")
    $color = switch ($Status) {
        "SUCCESS" { "Green" }
        "ERROR" { "Red" }
        "WARNING" { "Yellow" }
        default { "Cyan" }
    }
    Write-Host "[$Status] $Message" -ForegroundColor $color
}

# ============================================
# 1. GERAR CERTIFICADOS E CHAVES
# ============================================
function Generate-Certificates {
    Write-Status "====== GERANDO CERTIFICADOS ======" "INFO"
    
    $sshDir = "$env:USERPROFILE\.ssh"
    $certDir = ".\certificates"
    
    # Criar diretórios
    if (-not (Test-Path $sshDir)) {
        New-Item -ItemType Directory -Path $sshDir | Out-Null
        Write-Status "Diretório SSH criado: $sshDir" "SUCCESS"
    }
    
    if (-not (Test-Path $certDir)) {
        New-Item -ItemType Directory -Path $certDir | Out-Null
        Write-Status "Diretório certificados criado: $certDir" "SUCCESS"
    }
    
    # Gerar SSH key com OpenSSH (requer Windows 10+)
    $sshKeyPath = "$sshDir\tudoaqui-aws"
    if (-not (Test-Path $sshKeyPath)) {
        Write-Status "Gerando chave SSH..." "INFO"
        # Usar ssh-keygen (disponível no Windows 10+)
        $process = Start-Process -FilePath "ssh-keygen" -ArgumentList "-t rsa -b 4096 -f $sshKeyPath -N """"" -NoNewWindow -Wait -PassThru
        if ($process.ExitCode -eq 0) {
            Write-Status "Chave SSH gerada: $sshKeyPath" "SUCCESS"
        } else {
            Write-Status "Falha ao gerar SSH key. Criando manualmente..." "WARNING"
            # Fallback: criar arquivo com instruções
            $sshInstructions = @"
# Para gerar SSH key, execute no PowerShell (Admin):
# ssh-keygen -t rsa -b 4096 -f "$sshKeyPath" -N ""

# Ou use:
# wsl -e ssh-keygen -t rsa -b 4096 -f "$sshKeyPath" -N ""
"@
            $sshInstructions | Out-File "$certDir\SSH_SETUP_INSTRUCTIONS.txt"
            Write-Status "Instruções salvas em SSH_SETUP_INSTRUCTIONS.txt" "WARNING"
        }
    }
    
    # Gerar secrets
    $secretsContent = @"
# ============================================
# SECRETS GERADOS - MANTER SEGURO!
# ============================================
# Data Geração: $(Get-Date)
# NÃO COMMITAR ESTE ARQUIVO
# NÃO COMPARTILHAR ONLINE
# ============================================

# JWT Secrets
JWT_SECRET=TudoAqui_JWTSecret_2026_$(GenerateRandomString 32)
JWT_REFRESH_SECRET=TudoAqui_RefreshSecret_2026_$(GenerateRandomString 32)
JWT_EXPIRES_IN=24h
JWT_REFRESH_EXPIRES_IN=7d

# Webhook Secret
WEBHOOK_SECRET=TudoAqui_WebhookSecret_2026_$(GenerateRandomString 16)

# Database
DB_HOST=aws-rds-endpoint-aqui.amazonaws.com
DB_PORT=5432
DB_USER=tudoaqui_admin
DB_PASSWORD=Tr0p1cal_SecureDB_2026_$(GenerateRandomString 16)
DB_NAME=tudoaqui_prod
DB_SSL=true

# Redis
REDIS_HOST=aws-elasticache-endpoint-aqui.amazonaws.com
REDIS_PORT=6379
REDIS_PASSWORD=$(GenerateRandomString 20)
REDIS_CLUSTER_ENABLED=false

# AWS
AWS_REGION=us-east-1
AWS_ACCESS_KEY_ID=AKIA_YOUR_KEY_HERE
AWS_SECRET_ACCESS_KEY=YOUR_SECRET_KEY_HERE
AWS_S3_BUCKET=tudoaqui-media-prod
AWS_S3_REGION=us-east-1
AWS_CLOUDFRONT_DISTRIBUTION=dXXXXXXXXX.cloudfront.net

# External APIs
SENDGRID_API_KEY=SG.YOUR_SENDGRID_KEY
STRIPE_SECRET_KEY=sk_live_your_stripe_key
STRIPE_WEBHOOK_SECRET=$(GenerateRandomString 16)

# Firebase
FIREBASE_PROJECT_ID=tudoaqui-prod
FIREBASE_PRIVATE_KEY=YOUR_FIREBASE_PRIVATE_KEY
FIREBASE_CLIENT_EMAIL=firebase-adminsdk@tudoaqui-prod.iam.gserviceaccount.com

# Application
NODE_ENV=production
APP_PORT=3000
LOG_LEVEL=info
MONITORING_ENABLED=true

# Email
EMAIL_FROM=noreply@tudoaqui.ao
SMTP_HOST=smtp.sendgrid.net
SMTP_PORT=587
SMTP_USER=apikey

# SMS (Africa's Talking)
SMS_API_KEY=YOUR_AFRICAS_TALKING_KEY
SMS_USERNAME=YOUR_AFRICAS_TALKING_USERNAME

# Dados Sensíveis (Preenchimento Manual)
# =============================================
BI_OCR_API_KEY=YOUR_GOOGLE_VISION_API_KEY
FACIAL_RECOGNITION_API=YOUR_AMAZON_REKOGNITION_API
LIVENESS_DETECTION_API=YOUR_API_LIVENESS_KEY

# =============================================
# INSTRUÇÕES DE USO:
# 1. Preencher com seus valores reais
# 2. Copiar para .env (raiz do projeto)
# 3. NÃO COMMITAR .env para Git
# 4. Subir secrets para GitHub Actions
# =============================================
"@
    
    $secretsContent | Out-File "$certDir\secrets.txt" -Encoding UTF8
    Write-Status "Secrets gerados: $certDir\secrets.txt" "SUCCESS"
}

function GenerateRandomString {
    param([int]$Length = 32)
    $chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%"
    $random = New-Object System.Random
    $result = ""
    for ($i = 0; $i -lt $Length; $i++) {
        $result += $chars[$random.Next($chars.Length)]
    }
    return $result
}

# ============================================
# 2. CONFIGURAR TERRAFORM
# ============================================
function Setup-Terraform {
    Write-Status "====== CONFIGURANDO TERRAFORM ======" "INFO"
    
    # Verificar se Terraform está instalado
    $tfVersion = terraform version 2>$null
    if ($null -eq $tfVersion) {
        Write-Status "Terraform não encontrado. Instalando via Chocolatey..." "WARNING"
        choco install terraform -y
    }
    
    # Initializar Terraform
    cd .\infrastructure
    Write-Status "Inicializando Terraform..." "INFO"
    terraform init
    
    if ($LASTEXITCODE -eq 0) {
        Write-Status "Terraform inicializado com sucesso" "SUCCESS"
    } else {
        Write-Status "Erro ao inicializar Terraform" "ERROR"
        return $false
    }
    
    # Criar plan
    Write-Status "Gerando Terraform plan..." "INFO"
    terraform plan -out=tfplan
    
    cd ..
    return $true
}

# ============================================
# 3. SETUP DE GITHUB SECRETS
# ============================================
function Setup-GitHub-Secrets {
    Write-Status "====== SETUP GITHUB SECRETS ======" "INFO"
    
    $ghInstalled = gh version 2>$null
    if ($null -eq $ghInstalled) {
        Write-Status "GitHub CLI (gh) não instalado. Instalando..." "WARNING"
        choco install gh -y
    }
    
    Write-Status "Autenticando com GitHub..." "INFO"
    gh auth login
    
    # Ler secrets do arquivo
    $secrets = Get-Content ".\certificates\secrets.txt" | Select-String "^[A-Z_]+" | ForEach-Object {
        $line = $_.ToString()
        if ($line -match "^([A-Z_]+)=(.+)$") {
            @{
                Key = $matches[1]
                Value = $matches[2]
            }
        }
    }
    
    Write-Status "Adicionando $($secrets.Count) secrets ao GitHub..." "INFO"
    foreach ($secret in $secrets) {
        if ($secret.Key -and $secret.Value) {
            gh secret set $secret.Key --body $secret.Value
            Write-Status "Secret adicionado: $($secret.Key)" "SUCCESS"
        }
    }
}

# ============================================
# 4. CRIAR .env FILE
# ============================================
function Create-EnvFile {
    Write-Status "====== CRIANDO ARQUIVO .env ======" "INFO"
    
    if (Test-Path ".env") {
        Write-Status "Arquivo .env já existe. Fazendo backup..." "WARNING"
        Copy-Item ".env" ".env.backup.$(Get-Date -Format 'yyyyMMddHHmmss')"
    }
    
    # Copiar .env.example para .env
    if (Test-Path "backend\.env.example") {
        Copy-Item "backend\.env.example" ".env"
        Write-Status "Arquivo .env criado de .env.example" "SUCCESS"
    } else {
        Write-Status "Arquivo .env.example não encontrado" "ERROR"
    }
}

# ============================================
# MAIN EXECUTION
# ============================================
function Main {
    Write-Host ""
    Write-Status "╔════════════════════════════════════════╗" "INFO"
    Write-Status "║  SETUP PRODUÇÃO COMPLETO - TudoAqui  ║" "INFO"
    Write-Status "╚════════════════════════════════════════╝" "INFO"
    Write-Host ""
    
    # Menu
    Write-Host "Selecione as tarefas a executar:"
    Write-Host "1 - Gerar certificados e chaves"
    Write-Host "2 - Setup Terraform"
    Write-Host "3 - Configurar GitHub Secrets"
    Write-Host "4 - Criar arquivo .env"
    Write-Host "5 - EXECUTAR TUDO"
    Write-Host "0 - Sair"
    Write-Host ""
    
    $choice = Read-Host "Opção"
    
    switch ($choice) {
        "1" { Generate-Certificates }
        "2" { Setup-Terraform }
        "3" { Setup-GitHub-Secrets }
        "4" { Create-EnvFile }
        "5" {
            Generate-Certificates
            Create-EnvFile
            Setup-Terraform
            Write-Status "Próximo passo: Editar terraform.tfvars com suas credenciais AWS" "WARNING"
            Write-Status "Depois executar: cd infrastructure && terraform apply tfplan" "INFO"
        }
        "0" { Write-Status "Saindo..." "INFO"; exit }
        default { Write-Status "Opção inválida" "ERROR" }
    }
    
    Write-Host ""
    Write-Status "====== RESUMO DO SETUP ======" "INFO"
    Write-Status "✅ Certificados: ./certificates/secrets.txt" "SUCCESS"
    Write-Status "✅ Terraform: ./infrastructure/tfplan" "SUCCESS"
    Write-Status "⏳ GitHub Secrets: Configure manualmente em GitHub" "WARNING"
    Write-Status "⏳ AWS Credentials: Adicione em ./infrastructure/terraform.tfvars" "WARNING"
    Write-Host ""
}

# Executar
Main
