# TudoAqui Backend - Guia de Deploy e Configuração

## Inicialização Rápida (Local)

### Pré-requisitos
- Node.js 18+
- PostgreSQL 15+
- Redis 7+

### Setup

```bash
# Clonar repositório
git clone <repo>
cd TudoAqui_Backend

# Instalar dependências
npm install

# Configurar variáveis de ambiente
cp .env.example .env
# Editar .env com valores reais

# Criar database e rodar migrations
cd scripts
./run-migrations.sh  # ou run-migrations.bat no Windows

# Iniciar servidor (dev)
npm run start:dev

# Testes
npm test
npm run test:coverage
```

## Deployment com Docker Compose

### Pré-requisitos
- Docker Engine 20.10+
- Docker Compose 2.0+

### Deploy

```bash
# Criar arquivo .env com valores de produção
cp .env.example .env
# Editar .env com chaves reais (JWT, AWS, Stripe, etc.)

# Gerar certificados SSL (opcional, usar Let's Encrypt em produção)
mkdir -p ssl
# Colocar certificado.crt e chave.key em ./ssl/

# Build e subir containers
docker compose up -d --build

# Ver logs
docker compose logs -f app

# Testar
curl http://localhost/health
```

### Estrutura de Volumes
- `postgres_data`: Dados do PostgreSQL
- `redis_data`: Dados do Redis
- `./uploads`: Arquivos enviados pelos usuários (mapeado no container)
- `./ssl`: Certificados SSL/TLS

## Configuração de Produção

### 1. Segurança

**Nginx:**
- TLS 1.2+ obrigatório
- Rate limiting: 10r/s API, 5r/m Auth
- Security headers (HSTS, CSP, X-Frame-Options)
- Gzip compression ativado

**App:**
- JWT com expiração (7 dias padrão)
- Bcrypt 12 salt rounds
- Redis blacklist para logout
- Helmet.js + CORS configurados

**Banco:**
- PostgreSQL com user dedicado (não use postgres)
- Backup diário recomendado

### 2. Variáveis de Ambiente Críticas

```bash
# Gerar JWT_SECRET seguro (32+ chars)
node -e "console.log(require('crypto').randomBytes(32).toString('hex'))"

# Definir em .env:
JWT_SECRET=<gerado_acima>
DATABASE_PASSWORD=<senha_complexa>
REDIS_PASSWORD=<senha_complexa>
AWS_ACCESS_KEY_ID=<sua_chave_aws>
AWS_SECRET_ACCESS_KEY=<sua_chave_secreta_aws>
```

### 3. Backup e Restore

```bash
# Backup PostgreSQL
docker exec tudo_aqui_db pg_dump -U tudo_aqui_admin tudo_aqui_db > backup.sql

# Restore
docker exec -i tudo_aqui_db psql -U tudo_aqui_admin tudo_aqui_db < backup.sql

# Backup Redis
docker exec tudo_aqui_redis redis-cli --rdb /data/dump.rdb
docker cp tudo_aqui_redis:/data/dump.rdb ./redis_backup.rdb
```

### 4. Monitoramento

```bash
# Ver status dos containers
docker compose ps

# Logs de erro do app
docker compose logs --tail 100 app

# Inspecionar container
docker exec -it tudo_aqui_app /bin/sh

# Testar conectividade DB
docker exec tudo_aqui_app pg_isready -h postgres -p 5432

# Testar Redis
docker exec tudo_aqui_redis redis-cli ping
```

### 5. Escalabilidade

Para produção com alta demanda:

1. **Load Balancing**: Usar múltiplas instâncias do app atrás de Nginx/HAProxy
2. **Database**: Considerar replicação PostgreSQL (primary/standby)
3. **Cache**: Aumentar Redis ao múltiplos GBs conforme tráfego
4. **CDN**: Usar CloudFlare ou AWS CloudFront para assets
5. **Logs**: Integrar Sentry/DataDog para monitoramento centralizado

## Troubleshooting

### App não conecta ao Postgres
```bash
docker exec tudo_aqui_app pg_isready -h postgres -p 5432 -U tudo_aqui_admin
```

### Redis connection refused
```bash
docker exec tudo_aqui_redis redis-cli ping
# Verificar REDIS_PASSWORD no .env
```

### Health check failing
```bash
curl http://localhost:3000/health
# Verificar logs: docker compose logs app
```

### SSL certificate errors
```bash
# Renovar com Certbot (Let's Encrypt)
certbot certonly -d api.tudoaqui.ao
# Copiar para ./ssl/
```

## Migrações de Banco

As migrations rodam automaticamente no Docker via `docker-entrypoint-initdb.d/`.

Para migração manual:
```bash
cd scripts
./run-migrations.sh  # Linux/Mac
.\run-migrations.bat  # Windows
```

## Rollback de Deploy

```bash
# Ver versão anterior da imagem
docker images | grep tudoaqui

# Reverter para versão anterior
docker compose down
docker rmi tudoaqui_backend:<version_atual>
# Editar docker-compose.yml ou usar tag da imagem anterior
docker compose up -d
```

## Contato e Suporte

- Issues: GitHub Issues
- Email: devops@tudoaqui.ao
- Slack: #backend-infra
