# 🇦🇴 TudoAqui - Super App de Angola

![Version](https://img.shields.io/badge/version-1.0.0-blue.svg)
![Node](https://img.shields.io/badge/node-18.x-green.svg)
![PostgreSQL](https://img.shields.io/badge/postgresql-15.x-blue.svg)
![License](https://img.shields.io/badge/license-MIT-green.svg)

**Lema**: "Encontre tudo num só lugar"

Super app marketplace agregador para o mercado angolano que conecta utilizadores a produtos e serviços através de uma única plataforma integrada.

---

## 📑 Índice

- [Visão Geral](#visão-geral)
- [Arquitetura](#arquitetura)
- [Módulos Implementados](#módulos-implementados)
- [Stack Tecnológico](#stack-tecnológico)
- [Pré-requisitos](#pré-requisitos)
- [Instalação](#instalação)
- [Configuração](#configuração)
- [Executar Aplicação](#executar-aplicação)
- [Testes](#testes)
- [Documentação](#documentação)
- [Deploy](#deploy)
- [Monitoramento](#monitoramento)
- [Estrutura do Projeto](#estrutura-do-projeto)
- [Contribuição](#contribuição)
- [Licença](#licença)

---

## 🌟 Visão Geral

O **TudoAqui** é uma plataforma completa que integra os principais serviços do dia a dia angolano:

### Categorias Principais

| Categoria | Descrição | Status |
|-----------|-----------|--------|
| 🏠 **Imóveis** | Compra, venda e aluguel de imóveis | ✅ Implementado |
| 🚗 **Transporte** | Integração com Tupuca, Yango, Bolt, Heetch | ✅ Implementado |
| 🌴 **Turismo** | Pacotes turísticos e reservas | ✅ Implementado |
| 🎉 **Eventos** | Aluguel de salas para festas e eventos | ✅ Implementado |
| 🎫 **Ingressos** | Shows, teatros e eventos com QR code | ✅ Implementado |
| 🍽️ **Restaurantes** | Reservas de mesa e delivery | ✅ Implementado |
| 🏨 **Alojamentos** | Hotéis, pousadas e acomodações | ✅ Implementado |
| 💳 **Pagamentos** | Multicaixa, cartão, transferência | ✅ Implementado |

### Diferenciais

✅ **Segurança**: KYC com verificação de BI e reconhecimento facial (AWS Rekognition)  
✅ **Selo de Verificado**: Apenas vendedores aprovados podem publicar  
✅ **Integração Local**: Tupuca (logística), Multicaixa (pagamentos), Africa's Talking (SMS)  
✅ **Tempo Real**: Chat, rastreamento GPS, notificações push (Firebase)  
✅ **Escalabilidade**: Arquitetura de microserviços com Redis cache  
✅ **Observabilidade**: Prometheus + Grafana para métricas e monitoramento

---

## 🏗️ Arquitetura

### Diagrama de Alto Nível

```
┌─────────────────────────────────────────────────────────────────┐
│                        CLIENTES                                  │
│  Flutter App (iOS/Android/Web) | Postman | SDKs                 │
└────────────────────┬────────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────────┐
│                     LOAD BALANCER                                │
│                    Nginx (Reverse Proxy)                         │
└────────────────────┬────────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────────┐
│                   API GATEWAY (Express.js)                       │
│  • Rate Limiting  • CORS  • Security Headers  • JWT Validation  │
└────────────────────┬────────────────────────────────────────────┘
                     │
        ┌────────────┼────────────┐
        ▼            ▼             ▼
┌──────────────┬──────────────┬──────────────────┐
│   MÓDULOS    │   MÓDULOS    │    MÓDULOS       │
│              │              │                  │
│ • Imóveis    │ • Eventos    │ • Notificações   │
│ • Transporte │ • Ingressos  │ • Upload         │
│ • Turismo    │ • Restaur.   │ • Pagamentos     │
│ • Alojamentos│ • Auth       │                  │
└──────┬───────┴──────┬───────┴──────┬───────────┘
       │              │               │
       ▼              ▼               ▼
┌─────────────────────────────────────────────────────────────────┐
│                    CAMADA DE SERVIÇOS                            │
│  • KYC Service  • Payment Service  • Notification Service       │
│  • Upload Service  • Integration Service (Tupuca, Multicaixa)   │
└────────────────────┬────────────────────────────────────────────┘
                     │
        ┌────────────┼────────────────┐
        ▼            ▼                 ▼
┌──────────────┬──────────────┬────────────────┐
│  PostgreSQL  │    Redis     │   AWS S3       │
│  (Dados)     │ (Cache/GPS)  │ (Arquivos)     │
└──────────────┴──────────────┴────────────────┘
                     │
        ┌────────────┼────────────────┐
        ▼            ▼                 ▼
┌──────────────┬──────────────┬────────────────┐
│   Firebase   │   SendGrid   │ Africa's Talk  │
│ (Push Notif) │   (Email)    │     (SMS)      │
└──────────────┴──────────────┴────────────────┘
```

### Padrões Arquiteturais

- **Arquitetura em Camadas**: Controllers → Services → Repositories
- **Middleware Pipeline**: Security → Auth → Rate Limiting → Business Logic
- **Dependency Injection**: Services injetados via containers
- **Event-Driven**: Notificações via pub/sub pattern
- **CQRS**: Separação de leitura (cache) e escrita (DB)

---

## 📱 Módulos Implementados

### 1. Autenticação & Autorização
- ✅ Registro com verificação de telefone (SMS)
- ✅ Login com JWT (access + refresh tokens)
- ✅ Logout com blacklist Redis
- ✅ KYC: Upload de BI, selfie, liveness detection
- ✅ Tipos de usuário: Comprador, Vendedor, Prestador, Admin
- ✅ RBAC (Role-Based Access Control)

**Endpoints**: `POST /auth/register`, `POST /auth/login`, `POST /auth/logout`, `POST /kyc/upload-bi`

### 2. Imóveis
- ✅ CRUD de imóveis (casa, apartamento, terreno, escritório)
- ✅ Busca com filtros (preço, localização, tipo, quartos)
- ✅ Geolocalização (PostgreSQL + PostGIS)
- ✅ Upload de até 20 fotos (marca d'água automática)
- ✅ Chat interno comprador-vendedor
- ✅ Apenas vendedores verificados podem publicar

**Endpoints**: `POST /imoveis`, `GET /imoveis`, `GET /imoveis/:id`, `PUT /imoveis/:id`, `DELETE /imoveis/:id`

### 3. Transporte
- ✅ Integração com 4 fornecedores: Tupuca, Yango, Bolt, Heetch
- ✅ Comparação de preços em tempo real
- ✅ Rastreamento GPS (Redis para localização)
- ✅ Histórico de viagens
- ✅ Avaliação de motoristas

**Endpoints**: `POST /transporte/viagens`, `GET /transporte/viagens/:id`, `PUT /transporte/viagens/:id/confirmar`

### 4. Turismo
- ✅ Pacotes turísticos (praia, serra, safári, cultura)
- ✅ Reservas com voucher
- ✅ Check-in/check-out
- ✅ Política de cancelamento configurável
- ✅ Integração com operadoras locais

**Endpoints**: `GET /turismo/pacotes`, `POST /turismo/reservas`, `GET /turismo/reservas/:id`

### 5. Eventos (Salas)
- ✅ Aluguel de salas para eventos
- ✅ Calendário de disponibilidade
- ✅ Serviços extras (catering, decoração, som)
- ✅ Orçamento automático
- ✅ Contrato digital

**Endpoints**: `GET /eventos/salas`, `POST /eventos/reservas`, `GET /eventos/reservas/:id`

### 6. Ingressos
- ✅ Compra de ingressos (shows, teatro, esportes)
- ✅ Geração de QR code único por ingresso
- ✅ Validação na entrada (marca como usado)
- ✅ Transferência de ingresso
- ✅ Meia-entrada (estudante, idoso)
- ✅ Máximo 6 ingressos por compra

**Endpoints**: `POST /ingressos/comprar`, `POST /ingressos/validar`, `GET /ingressos/:codigo_qr`

### 7. Restaurantes
- ✅ Reserva de mesa
- ✅ Pedido de delivery
- ✅ Cardápio digital
- ✅ Rastreamento em tempo real
- ✅ Avaliação de restaurante e entregador

**Endpoints**: `GET /restaurantes`, `POST /restaurantes/:id/reservar`, `POST /restaurantes/:id/pedido`

### 8. Alojamentos
- ✅ Busca de hotéis, pousadas, apart-hotéis
- ✅ Filtros (preço, tipo, comodidades)
- ✅ Reservas com check-in/check-out
- ✅ Política de cancelamento
- ✅ Avaliações e fotos

**Endpoints**: `GET /alojamentos`, `POST /alojamentos/:id/reservar`, `GET /alojamentos/reservas/:id`

### 9. Upload de Arquivos
- ✅ Upload único e múltiplo
- ✅ AWS S3 para armazenamento
- ✅ Marca d'água automática (ImageMagick)
- ✅ Validação de tipo e tamanho
- ✅ URLs assinadas para download seguro

**Endpoints**: `POST /upload/single`, `POST /upload/multiple`, `DELETE /upload/:key`

### 10. Pagamentos
- ✅ **4 métodos**: Multicaixa Express, Cartão (Stripe), Transferência Bancária, Dinheiro
- ✅ Valor mínimo: 100 AOA
- ✅ Webhook para confirmação assíncrona
- ✅ Comprovativo em PDF
- ✅ Histórico de transações
- ✅ Sistema de reembolso

**Endpoints**: `POST /pagamentos/processar`, `GET /pagamentos/transacoes`, `POST /pagamentos/webhook`

### 11. Notificações
- ✅ **3 canais**: Push (Firebase), Email (SendGrid), SMS (Africa's Talking)
- ✅ Notificações transacionais (pagamento, reserva)
- ✅ Notificações promocionais (opt-in)
- ✅ Templates personalizáveis
- ✅ Histórico de notificações

**Endpoints**: `POST /notificacoes/enviar`, `GET /notificacoes`, `PUT /notificacoes/:id/marcar-lida`

---

## 🛠️ Stack Tecnológico

### Backend
| Tecnologia | Versão | Uso |
|------------|--------|-----|
| **Node.js** | 18.x | Runtime JavaScript |
| **Express** | 4.x | Framework web |
| **TypeScript** | 5.x | Linguagem (opcional) |
| **PostgreSQL** | 15.x | Banco de dados principal |
| **Redis** | 7.x | Cache e sessões |
| **AWS S3** | - | Armazenamento de arquivos |

### Autenticação & Segurança
| Tecnologia | Uso |
|------------|-----|
| **JWT** | Tokens de autenticação |
| **bcrypt** | Hash de senhas |
| **helmet** | Security headers |
| **express-rate-limit** | Rate limiting |
| **express-validator** | Validação de input |
| **AWS Rekognition** | KYC facial |

### Comunicação
| Serviço | Uso |
|---------|-----|
| **Firebase Cloud Messaging** | Push notifications |
| **SendGrid** | Email transacional |
| **Africa's Talking** | SMS (Angola) |

### Integrações
| Serviço | Uso |
|---------|-----|
| **Tupuca** | Logística e entregas |
| **Multicaixa Express** | Pagamentos móveis |
| **Stripe** | Pagamentos internacionais |
| **Yango/Bolt/Heetch** | Transporte/Táxi |

### DevOps & Monitoramento
| Tecnologia | Uso |
|------------|-----|
| **Docker** | Containerização |
| **Docker Compose** | Orquestração local |
| **Nginx** | Reverse proxy |
| **Prometheus** | Métricas |
| **Grafana** | Dashboards |
| **Winston** | Logging |

### Testing
| Tecnologia | Uso |
|------------|-----|
| **Jest** | Unit tests |
| **Supertest** | Integration tests |
| **Artillery** | Load testing |

---

## 📋 Pré-requisitos

### Obrigatórios
- **Node.js** 18.x ou superior
- **npm** 8.x ou superior
- **PostgreSQL** 15.x
- **Redis** 7.x
- **Git**

### Opcionais (para deploy)
- **Docker** 20.x
- **Docker Compose** 2.x
- **AWS CLI** (para S3)

### Verificar instalações

```bash
node --version    # v18.x.x
npm --version     # 8.x.x
psql --version    # 15.x
redis-cli --version  # 7.x
docker --version  # 20.x (opcional)
```

---

## 🚀 Instalação

### 1. Clonar Repositório

```bash
git clone https://github.com/tudoaqui/backend.git
cd backend
```

### 2. Instalar Dependências

```bash
npm install
```

**Principais dependências instaladas**:
```json
{
  "express": "^4.18.2",
  "pg": "^8.11.0",
  "redis": "^4.6.7",
  "jsonwebtoken": "^9.0.2",
  "bcrypt": "^5.1.1",
  "aws-sdk": "^2.1450.0",
  "firebase-admin": "^12.0.0",
  "stripe": "^14.0.0",
  "qrcode": "^1.5.3",
  "winston": "^3.11.0",
  "helmet": "^7.1.0",
  "cors": "^2.8.5",
  "dotenv": "^16.3.1"
}
```

### 3. Configurar Variáveis de Ambiente

Copie o arquivo de exemplo e edite:

```bash
cp .env.example .env
```

Edite `.env` com suas credenciais:

```env
# Aplicação
NODE_ENV=development
PORT=3000
API_BASE_URL=http://localhost:3000/api/v1

# PostgreSQL
DB_HOST=localhost
DB_PORT=5432
DB_NAME=tudoaqui
DB_USER=postgres
DB_PASSWORD=sua_senha_postgres

# Redis
REDIS_HOST=localhost
REDIS_PORT=6379
REDIS_PASSWORD=

# JWT
JWT_SECRET=seu_secret_key_muito_seguro_aqui
JWT_EXPIRES_IN=24h
JWT_REFRESH_SECRET=seu_refresh_secret_aqui
JWT_REFRESH_EXPIRES_IN=7d

# AWS S3
AWS_REGION=us-east-1
AWS_ACCESS_KEY_ID=sua_access_key
AWS_SECRET_ACCESS_KEY=sua_secret_key
AWS_S3_BUCKET=tudoaqui-uploads

# Firebase (Push Notifications)
FIREBASE_PROJECT_ID=seu_projeto_firebase
FIREBASE_CLIENT_EMAIL=firebase@seu-projeto.iam.gserviceaccount.com
FIREBASE_PRIVATE_KEY="-----BEGIN PRIVATE KEY-----\nSUA_CHAVE\n-----END PRIVATE KEY-----\n"

# SendGrid (Email)
SENDGRID_API_KEY=SG.sua_api_key_sendgrid
SENDGRID_FROM_EMAIL=noreply@tudoaqui.ao
SENDGRID_FROM_NAME=TudoAqui

# Africa's Talking (SMS)
AFRICAS_TALKING_USERNAME=seu_username
AFRICAS_TALKING_API_KEY=sua_api_key_africas_talking

# Stripe (Pagamentos)
STRIPE_SECRET_KEY=sk_test_sua_secret_key
STRIPE_WEBHOOK_SECRET=whsec_sua_webhook_secret

# Tupuca (Logística)
TUPUCA_API_KEY=sua_api_key_tupuca
TUPUCA_API_URL=https://api.tupuca.ao

# Multicaixa (Pagamentos)
MULTICAIXA_API_KEY=sua_api_key_multicaixa
MULTICAIXA_API_URL=https://api.multicaixa.ao

# Rate Limiting
RATE_LIMIT_WINDOW_MS=900000  # 15 minutos
RATE_LIMIT_MAX_REQUESTS=100
AUTH_RATE_LIMIT_MAX=5
```

### 4. Configurar Banco de Dados

#### Criar banco PostgreSQL

```bash
# Linux/Mac
psql -U postgres -c "CREATE DATABASE tudoaqui;"

# Windows
psql -U postgres
CREATE DATABASE tudoaqui;
\q
```

#### Rodar migrações

```bash
npm run migrate
```

**Migrations incluídas**:
- `001_create_users.sql` - Tabela de usuários
- `002_create_imoveis.sql` - Tabela de imóveis
- `003_create_transporte.sql` - Transporte e viagens
- `004_create_turismo.sql` - Pacotes e reservas turísticas
- `005_create_eventos.sql` - Salas e reservas de eventos
- `006_create_ingressos.sql` - Ingressos e validações
- `007_create_restaurantes.sql` - Restaurantes e pedidos
- `008_create_alojamentos.sql` - Alojamentos e reservas
- `009_create_pagamentos.sql` - Transações e métodos
- `010_create_notificacoes.sql` - Notificações

#### Popular banco com dados de teste (opcional)

```bash
npm run seed
```

### 5. Configurar Redis

#### Iniciar Redis

```bash
# Linux/Mac
redis-server

# Windows (se instalado via package manager)
redis-server.exe

# Ou use Docker
docker run -d -p 6379:6379 redis:7-alpine
```

#### Testar conexão

```bash
redis-cli ping
# Deve retornar: PONG
```

---

## ⚙️ Configuração

### AWS S3

1. Criar bucket no console AWS
2. Configurar CORS:

```json
[
  {
    "AllowedHeaders": ["*"],
    "AllowedMethods": ["GET", "PUT", "POST", "DELETE"],
    "AllowedOrigins": ["*"],
    "ExposeHeaders": ["ETag"]
  }
]
```

3. Criar IAM user com permissões S3
4. Copiar Access Key e Secret Key para `.env`

### Firebase

1. Criar projeto no [Firebase Console](https://console.firebase.google.com)
2. Gerar chave privada (Project Settings → Service Accounts)
3. Baixar JSON e copiar valores para `.env`

### SendGrid

1. Criar conta em [SendGrid](https://sendgrid.com)
2. Criar API Key (Settings → API Keys)
3. Verificar domínio (Sender Authentication)
4. Copiar API Key para `.env`

### Africa's Talking

1. Criar conta em [Africa's Talking](https://africastalking.com)
2. Criar API Key
3. Adicionar créditos para envio de SMS
4. Copiar credenciais para `.env`

### Stripe

1. Criar conta em [Stripe](https://stripe.com)
2. Copiar Secret Key (Developers → API Keys)
3. Configurar webhook:
   - URL: `https://api.tudoaqui.ao/api/v1/pagamentos/webhook`
   - Eventos: `payment_intent.succeeded`, `payment_intent.failed`
4. Copiar Webhook Secret para `.env`

---

## 🏃 Executar Aplicação

### Modo Desenvolvimento

```bash
npm run dev
```

Servidor inicia em: `http://localhost:3000`

**Features disponíveis em dev**:
- ✅ Hot reload (nodemon)
- ✅ Logs detalhados
- ✅ Swagger UI: `http://localhost:3000/api-docs`

### Modo Produção

```bash
npm start
```

### Com Docker

#### Desenvolvimento

```bash
docker-compose up -d
```

Serviços disponíveis:
- API: `http://localhost:3000`
- PostgreSQL: `localhost:5432`
- Redis: `localhost:6379`
- Prometheus: `http://localhost:9090`
- Grafana: `http://localhost:3001`

#### Produção

```bash
docker-compose -f docker-compose.prod.yml up -d
```

---

## 🧪 Testes

### Executar todos os testes

```bash
npm test
```

### Executar com coverage

```bash
npm run test:coverage
```

**Coverage atual**: 85%+ nos módulos principais

### Testes por módulo

```bash
# Auth
npm test -- auth.test.js

# Pagamentos
npm test -- pagamentos.test.js

# Imóveis
npm test -- imoveis.test.js
```

### Testes de integração

```bash
npm run test:integration
```

### Load testing

```bash
npm run test:load
```

**Resultados esperados**:
- 1000 req/s sem degradação
- P95 latency < 200ms
- 0% error rate

---

## 📚 Documentação

### Documentação Disponível

| Documento | Descrição | Localização |
|-----------|-----------|-------------|
| **API Documentation** | Referência completa da API | [API_DOCUMENTATION.md](docs/api/API_DOCUMENTATION.md) |
| **Swagger/OpenAPI** | Especificação OpenAPI 3.0 | [swagger.json](swagger.json) |
| **Swagger UI** | Documentação interativa | http://localhost:3000/api-docs |
| **Postman Collection** | Collection para testes | [TudoAqui.postman_collection.json](docs/api/TudoAqui.postman_collection.json) |
| **Postman Guide** | Guia de uso da collection | [POSTMAN_GUIDE.md](docs/api/POSTMAN_GUIDE.md) |
| **User Manual** | Manual do usuário final | [USER_MANUAL.md](docs/USER_MANUAL.md) |
| **Deployment Guide** | Guia de deployment | [DEPLOYMENT.md](docs/DEPLOYMENT.md) |
| **Architecture** | Decisões arquiteturais | [ARCHITECTURE.md](docs/ARCHITECTURE.md) |

### Swagger UI

Acesse documentação interativa:

```
http://localhost:3000/api-docs
```

**Features**:
- ✅ "Try it out" - Testar endpoints diretamente
- ✅ Schemas detalhados (request/response)
- ✅ Autenticação JWT integrada
- ✅ Exemplos de requisição
- ✅ Códigos HTTP documentados

### Postman Collection

1. Abra Postman
2. Import → File → Selecione `docs/api/TudoAqui.postman_collection.json`
3. Configure environment:
   - `base_url`: `http://localhost:3000/api/v1`
   - `auth_token`: (auto-preenchido após login)
4. Siga workflows em [POSTMAN_GUIDE.md](docs/api/POSTMAN_GUIDE.md)

---

## 🚢 Deploy

### Deploy Manual

#### 1. Preparar servidor

```bash
# Instalar dependências
sudo apt update
sudo apt install -y nodejs npm postgresql redis nginx

# Clonar repositório
git clone https://github.com/tudoaqui/backend.git /var/www/tudoaqui
cd /var/www/tudoaqui

# Instalar dependências
npm ci --production
```

#### 2. Configurar Nginx

Copie configuração:

```bash
sudo cp deploy/nginx.conf /etc/nginx/sites-available/tudoaqui
sudo ln -s /etc/nginx/sites-available/tudoaqui /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl restart nginx
```

#### 3. Configurar PM2

```bash
npm install -g pm2
pm2 start src/app.js --name tudoaqui-api
pm2 startup
pm2 save
```

#### 4. SSL com Let's Encrypt

```bash
sudo apt install certbot python3-certbot-nginx
sudo certbot --nginx -d api.tudoaqui.ao
```

### Deploy com Docker

#### Linux/Mac

```bash
chmod +x deploy/deploy.sh
./deploy/deploy.sh
```

#### Windows

```bash
deploy\deploy.bat
```

**Deploy script inclui**:
- ✅ Backup automático do banco
- ✅ Build de imagens Docker
- ✅ Pull de imagens atualizadas
- ✅ Restart de containers
- ✅ Health check
- ✅ Rollback automático se falhar
- ✅ Notificação Slack (opcional)

### Deploy para AWS

Veja guia completo: [docs/DEPLOYMENT.md](docs/DEPLOYMENT.md)

**Stack AWS sugerido**:
- **EC2** - Servidor de aplicação
- **RDS PostgreSQL** - Banco de dados
- **ElastiCache Redis** - Cache
- **S3** - Arquivos estáticos
- **CloudFront** - CDN
- **Route 53** - DNS
- **ALB** - Load balancer
- **Auto Scaling** - Escalabilidade

---

## 📊 Monitoramento

### Métricas Prometheus

Acesse: `http://localhost:9090`

**Métricas disponíveis**:
```
# Requests HTTP
http_requests_total{method="GET", route="/api/v1/imoveis", status="200"}
http_request_duration_seconds{method="POST", route="/api/v1/auth/login"}

# Database
db_query_duration_seconds{operation="SELECT", table="usuarios"}
db_connection_pool_size{status="active"}

# Redis
redis_commands_total{command="GET"}
redis_connection_errors_total

# Business Metrics
imoveis_criados_total
pagamentos_processados_total{metodo="multicaixa", status="confirmado"}
viagens_solicitadas_total{fornecedor="tupuca"}
```

### Dashboards Grafana

Acesse: `http://localhost:3001`

**Login padrão**:
- Username: `admin`
- Password: `admin`

**Dashboards incluídos**:
- 📈 **API Overview** - Requests, latência, errors
- 💾 **Database Performance** - Query time, connections
- 💳 **Payments** - Transações por método, taxa de sucesso
- 🚗 **Transport** - Viagens por fornecedor, tempo médio
- 👥 **Users** - Registros, logins, KYC aprovações

### Health Checks

#### Quick Health

```bash
curl http://localhost:3000/health
```

**Resposta**:
```json
{
  "status": "ok",
  "timestamp": "2026-01-22T10:30:00.000Z",
  "uptime": 3600
}
```

#### Detailed Health

```bash
curl http://localhost:3000/health/detailed
```

**Resposta**:
```json
{
  "status": "ok",
  "timestamp": "2026-01-22T10:30:00.000Z",
  "services": {
    "database": {
      "status": "ok",
      "responseTime": 5
    },
    "redis": {
      "status": "ok",
      "responseTime": 2
    },
    "s3": {
      "status": "ok",
      "responseTime": 150
    }
  }
}
```

### Logs

#### Visualizar logs em tempo real

```bash
# Docker
docker logs -f tudoaqui-api

# PM2
pm2 logs tudoaqui-api

# Arquivo direto
tail -f logs/app.log
```

#### Níveis de log

- `error` - Erros críticos (500, crashes)
- `warn` - Avisos (rate limit atingido, retry)
- `info` - Informações gerais (startup, requests)
- `debug` - Debugging detalhado (dev only)

#### Formato de log

```json
{
  "timestamp": "2026-01-22T10:30:00.000Z",
  "level": "info",
  "message": "POST /api/v1/pagamentos/processar",
  "meta": {
    "user_id": "123e4567-e89b-12d3-a456-426614174000",
    "ip": "192.168.1.100",
    "duration": 250,
    "status": 200
  }
}
```

---

## 📂 Estrutura do Projeto

```
TudoAqui_Backend/
├── src/
│   ├── app.js                    # Express app bootstrap
│   ├── server.js                 # HTTP server
│   ├── config/
│   │   ├── database.js           # PostgreSQL config
│   │   ├── redis.js              # Redis config
│   │   ├── aws.js                # AWS S3 config
│   │   ├── firebase.js           # Firebase config
│   │   └── swagger.js            # Swagger UI config
│   ├── modules/
│   │   ├── auth/
│   │   │   ├── auth.controller.js
│   │   │   ├── auth.service.js
│   │   │   ├── auth.routes.js
│   │   │   └── auth.validation.js
│   │   ├── imoveis/
│   │   ├── transporte/
│   │   ├── turismo/
│   │   ├── salas/
│   │   ├── ingressos/
│   │   ├── restaurantes/
│   │   ├── alojamentos/
│   │   ├── upload/
│   │   ├── notificacoes/
│   │   └── pagamentos/
│   ├── middleware/
│   │   ├── autenticar.js         # JWT verification
│   │   ├── autorizar.js          # RBAC
│   │   ├── rateLimiter.js        # Rate limiting
│   │   ├── sanitize.js           # Input sanitization
│   │   ├── errorHandler.js       # Global error handler
│   │   └── monitoring.js         # Prometheus metrics
│   ├── services/
│   │   ├── kycService.js         # KYC verification
│   │   ├── notificationService.js
│   │   ├── paymentService.js
│   │   ├── uploadService.js
│   │   └── integrations/
│   │       ├── tupuca.js
│   │       ├── multicaixa.js
│   │       ├── stripe.js
│   │       └── africasTalking.js
│   └── utils/
│       ├── constants.js
│       ├── helpers.js
│       └── validators.js
├── tests/
│   ├── unit/
│   ├── integration/
│   ├── fixtures/
│   └── __mocks__/
├── migrations/
│   ├── 001_create_users.sql
│   ├── 002_create_imoveis.sql
│   └── ...
├── seeds/
│   ├── 001_usuarios.js
│   └── 002_imoveis.js
├── docs/
│   ├── api/
│   │   ├── API_DOCUMENTATION.md
│   │   ├── TudoAqui.postman_collection.json
│   │   └── POSTMAN_GUIDE.md
│   ├── USER_MANUAL.md
│   ├── DEPLOYMENT.md
│   └── ARCHITECTURE.md
├── deploy/
│   ├── deploy.sh
│   ├── deploy.bat
│   ├── docker-compose.yml
│   ├── docker-compose.prod.yml
│   ├── Dockerfile
│   └── nginx.conf
├── logs/
│   ├── app.log
│   └── error.log
├── swagger.json                  # OpenAPI 3.0 spec
├── .env.example
├── .gitignore
├── package.json
├── jest.config.js
└── README.md
```

---

## 🤝 Contribuição

### Workflow de Contribuição

1. **Fork** o repositório
2. **Clone** seu fork: `git clone https://github.com/seu-usuario/backend.git`
3. **Crie branch**: `git checkout -b feature/nova-funcionalidade`
4. **Faça mudanças** e **commit**: `git commit -am 'Add nova funcionalidade'`
5. **Push**: `git push origin feature/nova-funcionalidade`
6. **Abra Pull Request**

### Padrões de Código

#### Commits (Conventional Commits)

```
feat: adiciona endpoint de busca de imóveis
fix: corrige validação de telefone angolano
docs: atualiza README com instruções de deploy
test: adiciona testes para módulo de pagamentos
refactor: reorganiza estrutura de controllers
chore: atualiza dependências
```

#### Nomenclatura

- **Arquivos**: `kebab-case.js` (ex: `auth.controller.js`)
- **Classes**: `PascalCase` (ex: `AuthController`)
- **Funções**: `camelCase` (ex: `autenticarUsuario()`)
- **Constantes**: `UPPER_SNAKE_CASE` (ex: `JWT_SECRET`)
- **Rotas**: `kebab-case` (ex: `/api/v1/meus-imoveis`)

#### Testes

- Todos os PRs devem incluir testes
- Coverage mínimo: 80%
- Testes devem seguir pattern AAA (Arrange, Act, Assert)

```javascript
describe('AuthController', () => {
  describe('POST /auth/register', () => {
    it('deve registrar novo usuário com sucesso', async () => {
      // Arrange
      const userData = { nome: 'João', email: 'joao@email.com', ... };
      
      // Act
      const response = await request(app)
        .post('/api/v1/auth/register')
        .send(userData);
      
      // Assert
      expect(response.status).toBe(201);
      expect(response.body.success).toBe(true);
    });
  });
});
```

### Code Review

PRs serão revisados quanto a:
- ✅ Funcionalidade correta
- ✅ Testes adequados
- ✅ Documentação atualizada
- ✅ Padrões de código seguidos
- ✅ Performance aceitável
- ✅ Segurança (sem vulnerabilidades)

---

## 🔒 Segurança

### Reportar Vulnerabilidade

**NÃO** abra issues públicas para vulnerabilidades de segurança.

Envie email para: **security@tudoaqui.ao**

Responderemos em **48 horas**.

### Práticas de Segurança

- ✅ Todas as senhas hasheadas com bcrypt (cost factor 12)
- ✅ JWT com expiração curta (24h) + refresh tokens (7d)
- ✅ Rate limiting por IP e por usuário
- ✅ Sanitização de inputs (XSS, SQL Injection)
- ✅ HTTPS obrigatório em produção
- ✅ Headers de segurança via Helmet
- ✅ CORS configurado
- ✅ Validação de schemas (Joi)
- ✅ Logs auditados

---

## 📄 Licença

Este projeto está licenciado sob a **MIT License**.

```
MIT License

Copyright (c) 2026 TudoAqui Angola

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
```

---

## 👥 Time

**TudoAqui Development Team**

- **Lead Developer**: [Nome]
- **Backend Team**: [Nomes]
- **DevOps**: [Nome]
- **QA**: [Nome]

---

## 📞 Contato

- **Website**: https://tudoaqui.ao
- **Email**: suporte@tudoaqui.ao
- **WhatsApp**: +244 923 123 456
- **GitHub**: https://github.com/tudoaqui
- **LinkedIn**: https://linkedin.com/company/tudoaqui

---

## 🎯 Roadmap

### Q1 2026 (Concluído ✅)
- [x] Backend completo com 8 módulos
- [x] Sistema de autenticação JWT
- [x] KYC com reconhecimento facial
- [x] 4 métodos de pagamento
- [x] Integrações (Tupuca, Multicaixa)
- [x] Documentação completa
- [x] Testes automatizados (85% coverage)
- [x] Docker e deploy scripts
- [x] Monitoramento (Prometheus + Grafana)

### Q2 2026 (Em andamento 🚧)
- [ ] Frontend Flutter (iOS/Android/Web)
- [ ] CI/CD pipeline (GitHub Actions)
- [ ] Kubernetes deployment
- [ ] Load balancing e auto-scaling
- [ ] CDN para assets
- [ ] Logs centralizados (ELK Stack)

### Q3 2026 (Planejado 📋)
- [ ] Machine Learning (recomendações)
- [ ] Chat em tempo real (Socket.io)
- [ ] Videochamadas (WebRTC)
- [ ] Marketplace de serviços
- [ ] Programa de fidelidade
- [ ] Multi-idioma (EN, PT)

### Q4 2026 (Futuro 🔮)
- [ ] Expansão para outros países africanos
- [ ] Integração com bancos locais
- [ ] Super app completo
- [ ] IPO preparation

---

## ⭐ Agradecimentos

Obrigado a todos que contribuíram para o TudoAqui:

- **Comunidade open-source** pelas ferramentas incríveis
- **Tupuca** pela parceria em logística
- **Multicaixa** pela integração de pagamentos
- **Investidores** pela confiança no projeto
- **Equipe TudoAqui** pela dedicação

---

**Feito com ❤️ em Angola 🇦🇴**

**TudoAqui - Encontre tudo num só lugar**
