@echo off
REM TudoAqui Backend - Deploy Script (Windows)
REM Script automatizado para deploy com backup, health checks e notificações
REM Uso: deploy.bat [production|staging]

setlocal enabledelayedexpansion

REM Configuração
set APP_NAME=tudo_aqui
set DOCKER_COMPOSE_FILE=docker-compose.yml
set DEPLOY_ENV=%1
if "%DEPLOY_ENV%"=="" set DEPLOY_ENV=production

for /f "tokens=2-4 delims=/ " %%a in ('date /t') do (set mydate=%%c%%a%%b)
for /f "tokens=1-2 delims=/:" %%a in ('time /t') do (set mytime=%%a%%b)
set DATE=%mydate%_%mytime%
set LOG_FILE=logs\tudo_aqui_deploy_%DATE%.log

if not exist logs mkdir logs

setlocal
call :log_info "🚀 Iniciando deploy do TudoAqui (Ambiente: %DEPLOY_ENV%)"
call :log_info "Log: %LOG_FILE%"

REM Verificar pré-requisitos
call :check_requirements

REM Backup do banco
call :backup_database

REM Validar compose
call :validate_compose

REM Pull das imagens
call :pull_images

REM Build da aplicação
call :build_application

REM Parar serviços
call :stop_services

REM Iniciar serviços
call :start_services

REM Aguardar inicialização
call :wait_for_startup

REM Health check
call :health_check

REM Limpar imagens
call :cleanup_images

call :log_success "Deploy concluído com sucesso!"
call :notify_slack success "Deploy do TudoAqui concluído com sucesso. Ambiente: %DEPLOY_ENV%"
endlocal
exit /b 0

:log_info
echo [INFO] %~1
echo [INFO] %~1 >> %LOG_FILE%
exit /b 0

:log_success
color 2
echo [SUCCESS] %~1
color 7
echo [SUCCESS] %~1 >> %LOG_FILE%
exit /b 0

:log_warning
color 6
echo [WARNING] %~1
color 7
echo [WARNING] %~1 >> %LOG_FILE%
exit /b 0

:log_error
color 4
echo [ERROR] %~1
color 7
echo [ERROR] %~1 >> %LOG_FILE%
exit /b 0

:check_requirements
call :log_info "Verificando pré-requisitos..."
where docker >nul 2>nul
if %ERRORLEVEL% NEQ 0 (
  call :log_error "Docker não está instalado"
  exit /b 1
)
where docker-compose >nul 2>nul
if %ERRORLEVEL% NEQ 0 (
  call :log_error "Docker Compose não está instalado"
  exit /b 1
)
call :log_success "Pré-requisitos OK"
exit /b 0

:backup_database
call :log_info "📦 Criando backup do banco de dados..."
docker ps | find "tudo_aqui_db" >nul 2>nul
if %ERRORLEVEL% NEQ 0 (
  call :log_warning "Container do banco não está rodando, pulando backup"
  exit /b 0
)
if not exist backups mkdir backups
docker exec tudo_aqui_db pg_dump -U tudo_aqui_admin tudo_aqui_db > backups\backup_%DATE%.sql
if %ERRORLEVEL% EQU 0 (
  call :log_success "Backup criado: backups\backup_%DATE%.sql"
) else (
  call :log_error "Falha ao fazer backup do banco"
  exit /b 1
)
exit /b 0

:validate_compose
call :log_info "Validando docker-compose.yml..."
docker-compose -f %DOCKER_COMPOSE_FILE% config >nul 2>nul
if %ERRORLEVEL% NEQ 0 (
  call :log_error "docker-compose.yml é inválido"
  exit /b 1
)
call :log_success "docker-compose.yml validado"
exit /b 0

:pull_images
call :log_info "📥 Atualizando imagens Docker..."
docker-compose -f %DOCKER_COMPOSE_FILE% pull
if %ERRORLEVEL% NEQ 0 (
  call :log_warning "Falha ao fazer pull, continuando com imagens locais"
)
call :log_success "Imagens atualizadas"
exit /b 0

:build_application
call :log_info "🔨 Build da aplicação..."
docker-compose -f %DOCKER_COMPOSE_FILE% build --no-cache
if %ERRORLEVEL% NEQ 0 (
  call :log_error "Falha no build da aplicação"
  exit /b 1
)
call :log_success "Build concluído"
exit /b 0

:stop_services
call :log_info "🛑 Parando serviços antigos..."
docker-compose -f %DOCKER_COMPOSE_FILE% down --remove-orphans
call :log_success "Serviços parados"
exit /b 0

:start_services
call :log_info "▶️  Iniciando novos serviços..."
docker-compose -f %DOCKER_COMPOSE_FILE% up -d
if %ERRORLEVEL% NEQ 0 (
  call :log_error "Falha ao iniciar serviços"
  exit /b 1
)
call :log_success "Serviços iniciados"
exit /b 0

:wait_for_startup
call :log_info "⏳ Aguardando serviços iniciarem (máximo 120 segundos)..."
setlocal
set /a attempt=1
:wait_loop
if !attempt! GTR 12 (
  call :log_error "Timeout aguardando inicialização dos serviços"
  exit /b 1
)
docker ps | find "tudo_aqui_app" >nul 2>nul
if %ERRORLEVEL% EQU 0 (
  call :log_success "Container da app está rodando"
  timeout /t 5 /nobreak
  exit /b 0
)
call :log_info "Tentativa !attempt!/12..."
timeout /t 10 /nobreak
set /a attempt=!attempt!+1
goto wait_loop
endlocal

:health_check
call :log_info "🏥 Realizando health check..."
setlocal
set /a attempt=1
:health_loop
if !attempt! GTR 10 (
  call :log_error "Health check falhou após 10 tentativas"
  exit /b 1
)
curl -sf http://127.0.0.1:3000/health >nul 2>nul
if %ERRORLEVEL% EQU 0 (
  call :log_success "Aplicação está saudável!"
  exit /b 0
)
call :log_info "Health check falhou, tentativa !attempt!/10..."
timeout /t 10 /nobreak
set /a attempt=!attempt!+1
goto health_loop
endlocal

:cleanup_images
call :log_info "🧹 Limpando imagens Docker antigas..."
docker image prune -f >nul 2>nul
docker volume prune -f >nul 2>nul
call :log_success "Limpeza concluída"
exit /b 0

:notify_slack
set STATUS=%~1
set MESSAGE=%~2
if "%SLACK_WEBHOOK_URL%"=="" (
  call :log_warning "SLACK_WEBHOOK_URL não configurado, pulando notificação"
  exit /b 0
)
REM Notificação via curl (requer JSON escrito em arquivo temporário)
(
  echo {
  echo   "text": "✅ TudoAqui Deploy Report",
  echo   "blocks": [
  echo     {
  echo       "type": "section",
  echo       "text": {
  echo         "type": "mrkdwn",
  echo         "text": "*Status:* %STATUS%\n*Ambiente:* %DEPLOY_ENV%\n*Data:* %date%"
  echo       }
  echo     }
  echo   ]
  echo }
) > slack_payload.json
curl -X POST -H "Content-Type: application/json" -d @slack_payload.json "%SLACK_WEBHOOK_URL%" >nul 2>nul
if %ERRORLEVEL% EQU 0 (
  call :log_success "Notificação Slack enviada"
  del slack_payload.json
) else (
  call :log_warning "Falha ao enviar notificação Slack"
)
exit /b 0
