#!/bin/bash

# TudoAqui Backend - Deploy Script
# Script automatizado para deploy com backup, health checks e notificações
# Uso: ./deploy.sh [production|staging]

set -e

# Cores para output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Configuração
APP_NAME="tudo_aqui"
DOCKER_COMPOSE_FILE="docker-compose.yml"
BACKUP_DIR="/backups/tudo_aqui"
DEPLOY_ENV="${1:-production}"
DATE=$(date +%Y%m%d_%H%M%S)
LOG_FILE="/var/log/tudo_aqui_deploy_$DATE.log"

# Funções
log_info() {
  echo -e "${BLUE}ℹ️  $1${NC}" | tee -a "$LOG_FILE"
}

log_success() {
  echo -e "${GREEN}✅ $1${NC}" | tee -a "$LOG_FILE"
}

log_warning() {
  echo -e "${YELLOW}⚠️  $1${NC}" | tee -a "$LOG_FILE"
}

log_error() {
  echo -e "${RED}❌ $1${NC}" | tee -a "$LOG_FILE"
}

error_exit() {
  log_error "$1"
  exit 1
}

# Verificar pré-requisitos
check_requirements() {
  log_info "Verificando pré-requisitos..."
  
  command -v docker >/dev/null 2>&1 || error_exit "Docker não está instalado"
  command -v docker-compose >/dev/null 2>&1 || error_exit "Docker Compose não está instalado"
  command -v curl >/dev/null 2>&1 || error_exit "curl não está instalado"
  
  log_success "Pré-requisitos OK"
}

# Criar diretório de backup
create_backup_dir() {
  log_info "Preparando diretório de backup..."
  mkdir -p "$BACKUP_DIR"
  log_success "Diretório de backup pronto: $BACKUP_DIR"
}

# Fazer backup do banco de dados
backup_database() {
  log_info "📦 Criando backup do banco de dados..."
  
  if ! docker ps | grep -q tudo_aqui_db; then
    log_warning "Container do banco não está rodando, pulando backup"
    return 0
  fi
  
  if docker exec tudo_aqui_db pg_dump -U tudo_aqui_admin tudo_aqui_db > "$BACKUP_DIR/backup_$DATE.sql"; then
    log_success "Backup criado: $BACKUP_DIR/backup_$DATE.sql"
    
    # Manter apenas os últimos 7 backups
    log_info "Limpando backups antigos (mantendo 7 últimos)..."
    ls -t "$BACKUP_DIR"/backup_*.sql 2>/dev/null | tail -n +8 | xargs -r rm
    log_success "Limpeza de backups completa"
  else
    error_exit "Falha ao fazer backup do banco"
  fi
}

# Validar docker-compose.yml
validate_compose() {
  log_info "Validando docker-compose.yml..."
  
  if ! docker-compose -f "$DOCKER_COMPOSE_FILE" config > /dev/null 2>&1; then
    error_exit "docker-compose.yml é inválido"
  fi
  
  log_success "docker-compose.yml validado"
}

# Pull das imagens
pull_images() {
  log_info "📥 Atualizando imagens Docker..."
  
  if ! docker-compose -f "$DOCKER_COMPOSE_FILE" pull; then
    log_warning "Falha ao fazer pull, continuando com imagens locais"
  else
    log_success "Imagens atualizadas"
  fi
}

# Build da aplicação
build_application() {
  log_info "🔨 Build da aplicação..."
  
  if ! docker-compose -f "$DOCKER_COMPOSE_FILE" build --no-cache; then
    error_exit "Falha no build da aplicação"
  fi
  
  log_success "Build concluído"
}

# Parar serviços antigos
stop_services() {
  log_info "🛑 Parando serviços antigos..."
  
  docker-compose -f "$DOCKER_COMPOSE_FILE" down --remove-orphans || true
  
  log_success "Serviços parados"
}

# Iniciar novos serviços
start_services() {
  log_info "▶️  Iniciando novos serviços..."
  
  if ! docker-compose -f "$DOCKER_COMPOSE_FILE" up -d; then
    error_exit "Falha ao iniciar serviços"
  fi
  
  log_success "Serviços iniciados"
}

# Aguardar inicialização
wait_for_startup() {
  log_info "⏳ Aguardando serviços iniciarem (máximo 120 segundos)..."
  
  local max_attempts=12
  local attempt=1
  
  while [ $attempt -le $max_attempts ]; do
    if docker ps | grep -q tudo_aqui_app; then
      log_success "Container da app está rodando"
      sleep 5 # Aguardar app completar boot
      return 0
    fi
    
    log_info "Tentativa $attempt/$max_attempts..."
    sleep 10
    ((attempt++))
  done
  
  error_exit "Timeout aguardando inicialização dos serviços"
}

# Health check
health_check() {
  log_info "🏥 Realizando health check..."
  
  local max_attempts=10
  local attempt=1
  local health_endpoint="http://127.0.0.1:3000/health"
  
  while [ $attempt -le $max_attempts ]; do
    if curl -sf "$health_endpoint" > /dev/null 2>&1; then
      log_success "Aplicação está saudável!"
      return 0
    fi
    
    log_info "Health check falhou, tentativa $attempt/$max_attempts..."
    sleep 10
    ((attempt++))
  done
  
  error_exit "Health check falhou após $max_attempts tentativas"
}

# Verificar logs de erro
check_logs() {
  log_info "📋 Verificando logs da aplicação..."
  
  echo "" >> "$LOG_FILE"
  echo "=== Últimas 50 linhas dos logs ===" >> "$LOG_FILE"
  docker-compose -f "$DOCKER_COMPOSE_FILE" logs --tail=50 app >> "$LOG_FILE" 2>&1 || true
  
  log_info "Logs adicionados ao arquivo de log"
}

# Limpar imagens antigas
cleanup_images() {
  log_info "🧹 Limpando imagens Docker antigas..."
  
  docker image prune -f > /dev/null 2>&1 || true
  docker volume prune -f > /dev/null 2>&1 || true
  
  log_success "Limpeza concluída"
}

# Notificar Slack
notify_slack() {
  local status=$1
  local message=$2
  
  if [ -z "$SLACK_WEBHOOK_URL" ]; then
    log_warning "SLACK_WEBHOOK_URL não configurado, pulando notificação"
    return 0
  fi
  
  local emoji="✅"
  [ "$status" = "failure" ] && emoji="❌"
  
  local slack_message=$(cat <<EOF
{
  "text": "$emoji TudoAqui Deploy Report",
  "blocks": [
    {
      "type": "section",
      "text": {
        "type": "mrkdwn",
        "text": "*Status:* $status\n*Ambiente:* $DEPLOY_ENV\n*Data:* $(date)\n*Mensagem:* $message"
      }
    }
  ]
}
EOF
)
  
  if curl -X POST -H 'Content-Type: application/json' -d "$slack_message" "$SLACK_WEBHOOK_URL" > /dev/null 2>&1; then
    log_success "Notificação Slack enviada"
  else
    log_warning "Falha ao enviar notificação Slack"
  fi
}

# Função principal
main() {
  log_info "🚀 Iniciando deploy do TudoAqui (Ambiente: $DEPLOY_ENV)..."
  log_info "Log: $LOG_FILE"
  
  check_requirements
  create_backup_dir
  backup_database
  validate_compose
  pull_images
  build_application
  stop_services
  start_services
  wait_for_startup
  health_check
  check_logs
  cleanup_images
  
  log_success "Deploy concluído com sucesso!"
  log_info "Backup: $BACKUP_DIR/backup_$DATE.sql"
  log_info "Log completo: $LOG_FILE"
  
  notify_slack "success" "Deploy do TudoAqui concluído com sucesso. Ambiente: $DEPLOY_ENV"
}

# Tratamento de erros
trap 'log_error "Deploy falhou na linha $LINENO"; notify_slack "failure" "Deploy falhou na linha $LINENO"; exit 1' ERR

# Executar
main "$@"
