-- Migração: Criar todas as tabelas dos módulos do marketplace TudoAqui
-- Data: 2026-01-22
-- Descrição: Estrutura completa de dados para imobiliário, transporte, turismo, eventos, restaurantes e alojamentos

-- EXTENSÕES (Se não criadas anteriormente)
CREATE EXTENSION IF NOT EXISTS "uuid-ossp";
CREATE EXTENSION IF NOT EXISTS "postgis";
CREATE EXTENSION IF NOT EXISTS "pg_trgm";

-- ===========================
-- MÓDULO IMOBILIÁRIO
-- ===========================
CREATE TABLE IF NOT EXISTS imoveis (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    proprietario_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Informações básicas
    titulo VARCHAR(255) NOT NULL,
    descricao TEXT,
    tipo VARCHAR(50) NOT NULL, -- casa, apartamento, terreno, escritorio
    finalidade VARCHAR(50) NOT NULL, -- venda, aluguel, temporada
    
    -- Preços
    preco DECIMAL(12,2) NOT NULL,
    preco_condominio DECIMAL(8,2),
    iptu DECIMAL(8,2),
    
    -- Localização
    localizacao VARCHAR(255),
    latitude DECIMAL(10,8),
    longitude DECIMAL(11,8),
    endereco_completo JSONB,
    
    -- Características
    area_total DECIMAL(8,2),
    area_util DECIMAL(8,2),
    quartos INTEGER,
    banheiros INTEGER,
    suites INTEGER,
    vagas_garagem INTEGER,
    andar INTEGER,
    apartamento_andar INTEGER,
    idade_imovel INTEGER,
    
    -- Detalhes
    comodidades TEXT[],
    fotos TEXT[],
    video_url TEXT,
    matricula_imovel VARCHAR(50),
    
    -- Status e métricas
    status VARCHAR(50) DEFAULT 'ativo', -- ativo, vendido, alugado, inativo
    destaque BOOLEAN DEFAULT false,
    visualizacoes INTEGER DEFAULT 0,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX idx_imoveis_proprietario ON imoveis(proprietario_id);
CREATE INDEX idx_imoveis_tipo_finalidade ON imoveis(tipo, finalidade);
CREATE INDEX idx_imoveis_preco ON imoveis(preco);
CREATE INDEX idx_imoveis_status ON imoveis(status);
CREATE INDEX idx_imoveis_criado ON imoveis(created_at DESC);

-- ===========================
-- MÓDULO TRANSPORTE
-- ===========================
CREATE TABLE IF NOT EXISTS viagens (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    usuario_id UUID REFERENCES usuarios(id) ON DELETE SET NULL,
    motorista_id UUID REFERENCES usuarios(id) ON DELETE SET NULL,
    
    -- Rotas
    origem JSONB NOT NULL,
    destino JSONB NOT NULL,
    distancia_km DECIMAL(8,2),
    duracao_minutos INTEGER,
    
    -- Serviço
    tipo_servico VARCHAR(50) NOT NULL, -- normal, exec, mulher, acessivel
    servico_empresa VARCHAR(50), -- tupuca, yango, bolt, heetch
    servico_id_externo VARCHAR(100),
    
    -- Preços
    preco_estimado DECIMAL(8,2),
    preco_final DECIMAL(8,2),
    forma_pagamento VARCHAR(50),
    
    -- Status e datas
    status VARCHAR(50) DEFAULT 'solicitada', -- solicitada, confirmada, em_curso, concluida, cancelada
    data_solicitacao TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    data_confirmacao TIMESTAMP,
    data_inicio TIMESTAMP,
    data_fim TIMESTAMP,
    
    -- Avaliações
    avaliacao_motorista INTEGER CHECK (avaliacao_motorista IS NULL OR (avaliacao_motorista >= 1 AND avaliacao_motorista <= 5)),
    avaliacao_passageiro INTEGER CHECK (avaliacao_passageiro IS NULL OR (avaliacao_passageiro >= 1 AND avaliacao_passageiro <= 5))
);

CREATE INDEX idx_viagens_usuario_status ON viagens(usuario_id, status);
CREATE INDEX idx_viagens_motorista ON viagens(motorista_id);
CREATE INDEX idx_viagens_data ON viagens(data_solicitacao DESC);

-- ===========================
-- MÓDULO TURISMO
-- ===========================
CREATE TABLE IF NOT EXISTS pacotes_turisticos (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    fornecedor_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Informações básicas
    titulo VARCHAR(255) NOT NULL,
    descricao TEXT,
    categoria VARCHAR(50), -- aventura, cultural, praia, safari
    localizacao VARCHAR(255),
    
    -- Duração
    duracao_dias INTEGER,
    duracao_horas INTEGER,
    
    -- Preços
    preco DECIMAL(8,2) NOT NULL,
    preco_crianca DECIMAL(8,2),
    
    -- Capacidade
    capacidade_maxima INTEGER DEFAULT 20,
    
    -- Detalhes
    inclusos TEXT[],
    nao_inclusos TEXT[],
    requisitos TEXT[],
    
    -- Mídia
    fotos TEXT[],
    video_url TEXT,
    
    -- Status
    ativo BOOLEAN DEFAULT true,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE IF NOT EXISTS reservas_turisticas (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    usuario_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    pacote_id UUID REFERENCES pacotes_turisticos(id) ON DELETE CASCADE,
    
    -- Detalhes da reserva
    data_reserva DATE NOT NULL,
    numero_adultos INTEGER DEFAULT 1,
    numero_criancas INTEGER DEFAULT 0,
    observacoes TEXT,
    
    -- Valores
    valor_total DECIMAL(8,2),
    
    -- Status
    status VARCHAR(50) DEFAULT 'pendente', -- pendente, confirmada, concluida, cancelada
    codigo_reserva VARCHAR(20) UNIQUE,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX idx_pacotes_fornecedor ON pacotes_turisticos(fornecedor_id);
CREATE INDEX idx_pacotes_categoria ON pacotes_turisticos(categoria);
CREATE INDEX idx_reservas_turisticas_usuario ON reservas_turisticas(usuario_id);
CREATE INDEX idx_reservas_turisticas_status ON reservas_turisticas(status);

-- ===========================
-- MÓDULO EVENTOS E SALAS
-- ===========================
CREATE TABLE IF NOT EXISTS salas_eventos (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    proprietario_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Informações básicas
    nome VARCHAR(255) NOT NULL,
    descricao TEXT,
    
    -- Localização
    localizacao VARCHAR(255),
    endereco_completo JSONB,
    
    -- Capacidade e características
    capacidade INTEGER NOT NULL,
    area_m2 DECIMAL(8,2),
    
    -- Preços
    preco_hora DECIMAL(8,2) NOT NULL,
    preco_diaria DECIMAL(10,2),
    
    -- Políticas
    tipo_eventos_permitidos TEXT[],
    comodidades TEXT[],
    regras TEXT[],
    
    -- Mídia
    fotos TEXT[],
    video_url TEXT,
    
    -- Disponibilidade
    disponibilidade JSONB,
    ativa BOOLEAN DEFAULT true,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE IF NOT EXISTS reservas_salas (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    usuario_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    sala_id UUID REFERENCES salas_eventos(id) ON DELETE CASCADE,
    
    -- Data e hora
    data_evento DATE NOT NULL,
    hora_inicio TIME NOT NULL,
    hora_fim TIME NOT NULL,
    numero_convidados INTEGER NOT NULL,
    
    -- Detalhes
    tipo_evento VARCHAR(50),
    observacoes TEXT,
    
    -- Status e valores
    status VARCHAR(50) DEFAULT 'pendente',
    valor_total DECIMAL(8,2),
    
    -- Check-in/out
    qr_code_checkin TEXT,
    data_checkin TIMESTAMP,
    data_checkout TIMESTAMP,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX idx_salas_proprietario ON salas_eventos(proprietario_id);
CREATE INDEX idx_reservas_salas_usuario ON reservas_salas(usuario_id);
CREATE INDEX idx_reservas_salas_sala ON reservas_salas(sala_id);
CREATE INDEX idx_reservas_salas_data ON reservas_salas(data_evento);

-- ===========================
-- MÓDULO INGRESSOS
-- ===========================
CREATE TABLE IF NOT EXISTS eventos_ingressos (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    organizador_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Informações básicas
    titulo VARCHAR(255) NOT NULL,
    descricao TEXT,
    categoria VARCHAR(50),
    
    -- Localização
    localizacao VARCHAR(255),
    endereco_completo JSONB,
    
    -- Data e hora
    data_hora TIMESTAMP NOT NULL,
    duracao_horas INTEGER,
    
    -- Capacidade
    capacidade_total INTEGER NOT NULL,
    lotacao_por_setor JSONB,
    
    -- Preços
    preco_ingresso DECIMAL(8,2),
    taxa_servico DECIMAL(5,2),
    
    -- Mídia
    foto_cartaz TEXT,
    
    -- Status
    status VARCHAR(50) DEFAULT 'ativo', -- ativo, cancelado, finalizado
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE IF NOT EXISTS ingressos_vendidos (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    evento_id UUID REFERENCES eventos_ingressos(id) ON DELETE CASCADE,
    comprador_id UUID REFERENCES usuarios(id) ON DELETE SET NULL,
    
    -- Identificação
    codigo_qr VARCHAR(255) UNIQUE NOT NULL,
    codigo_barras VARCHAR(50),
    
    -- Características
    setor VARCHAR(50),
    tipo VARCHAR(50), -- normal, vip, estudante, idoso
    
    -- Status
    status VARCHAR(50) DEFAULT 'ativo', -- ativo, usado, cancelado
    preco DECIMAL(8,2),
    data_uso TIMESTAMP,
    
    data_compra TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX idx_eventos_organizador ON eventos_ingressos(organizador_id);
CREATE INDEX idx_eventos_data ON eventos_ingressos(data_hora);
CREATE INDEX idx_ingressos_evento ON ingressos_vendidos(evento_id);
CREATE INDEX idx_ingressos_comprador ON ingressos_vendidos(comprador_id);
CREATE INDEX idx_ingressos_codigo_qr ON ingressos_vendidos(codigo_qr);

-- ===========================
-- MÓDULO RESTAURANTES
-- ===========================
CREATE TABLE IF NOT EXISTS restaurantes (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    proprietario_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Informações básicas
    nome VARCHAR(255) NOT NULL,
    descricao TEXT,
    tipo_cozinha VARCHAR(50),
    
    -- Localização
    localizacao VARCHAR(255),
    endereco_completo JSONB,
    
    -- Informações gerais
    faixa_preco INTEGER, -- 1-4 (R$ a R$$$$)
    telefone VARCHAR(20),
    whatsapp VARCHAR(20),
    email_contato VARCHAR(255),
    
    -- Horários
    horario_funcionamento JSONB,
    dias_funcionamento TEXT[],
    
    -- Comodidades e pagamento
    comodidades TEXT[],
    formas_pagamento TEXT[],
    
    -- Entrega
    faz_entrega BOOLEAN DEFAULT false,
    raio_entrega_km INTEGER,
    taxa_entrega DECIMAL(6,2),
    tempo_medio_preparo INTEGER,
    
    -- Mídia
    foto_perfil TEXT,
    fotos_ambiente TEXT[],
    
    -- Avaliações
    avaliacao_media DECIMAL(3,2),
    total_avaliacoes INTEGER DEFAULT 0,
    
    -- Status
    ativo BOOLEAN DEFAULT true,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE IF NOT EXISTS itens_cardapio (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    restaurante_id UUID REFERENCES restaurantes(id) ON DELETE CASCADE,
    
    -- Informações básicas
    nome VARCHAR(255) NOT NULL,
    descricao TEXT,
    categoria VARCHAR(50), -- entrada, principal, sobremesa, bebida
    
    -- Preço e disponibilidade
    preco DECIMAL(8,2) NOT NULL,
    disponivel BOOLEAN DEFAULT true,
    
    -- Mídia
    foto_url TEXT,
    
    -- Tempo
    tempo_preparo_estimado INTEGER,
    
    -- Informações nutricionais
    ingredientes TEXT[],
    alergenicos TEXT[],
    informacoes_nutricionais JSONB,
    
    -- Status
    destaque BOOLEAN DEFAULT false,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE IF NOT EXISTS reservas_restaurantes (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    usuario_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    restaurante_id UUID REFERENCES restaurantes(id) ON DELETE CASCADE,
    
    -- Data e hora
    data DATE NOT NULL,
    hora TIME NOT NULL,
    numero_pessoas INTEGER NOT NULL,
    
    -- Preferências
    area_preferencial VARCHAR(50),
    observacoes TEXT,
    
    -- Status e avaliação
    status VARCHAR(50) DEFAULT 'confirmada', -- confirmada, concluida, cancelada, no_show
    valor_gasto DECIMAL(8,2),
    avaliacao INTEGER CHECK (avaliacao IS NULL OR (avaliacao >= 1 AND avaliacao <= 5)),
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE IF NOT EXISTS pedidos_restaurantes (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    usuario_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    restaurante_id UUID REFERENCES restaurantes(id) ON DELETE CASCADE,
    
    -- Tipo e local de entrega
    tipo_entrega VARCHAR(50), -- retirada, delivery, mesa
    endereco_entrega JSONB,
    
    -- Detalhes do pedido
    observacoes TEXT,
    
    -- Valores
    valor_total DECIMAL(8,2),
    taxa_entrega DECIMAL(6,2),
    desconto DECIMAL(6,2),
    
    -- Tempo
    tempo_estimado INTEGER,
    
    -- Status
    status VARCHAR(50) DEFAULT 'recebido', -- recebido, preparando, pronto, entregando, entregue, cancelado
    codigo_retirada VARCHAR(10),
    
    -- Avaliação
    avaliacao INTEGER CHECK (avaliacao IS NULL OR (avaliacao >= 1 AND avaliacao <= 5)),
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX idx_restaurantes_proprietario ON restaurantes(proprietario_id);
CREATE INDEX idx_restaurantes_ativo ON restaurantes(ativo);
CREATE INDEX idx_cardapio_restaurante ON itens_cardapio(restaurante_id);
CREATE INDEX idx_cardapio_categoria ON itens_cardapio(categoria);
CREATE INDEX idx_reservas_restaurantes_usuario ON reservas_restaurantes(usuario_id);
CREATE INDEX idx_reservas_restaurantes_data ON reservas_restaurantes(data);
CREATE INDEX idx_pedidos_restaurantes_usuario ON pedidos_restaurantes(usuario_id);
CREATE INDEX idx_pedidos_restaurantes_status ON pedidos_restaurantes(status);

-- ===========================
-- MÓDULO ALOJAMENTOS
-- ===========================
CREATE TABLE IF NOT EXISTS alojamentos (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    anfitriao_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Informações básicas
    titulo VARCHAR(255) NOT NULL,
    descricao TEXT,
    tipo_alojamento VARCHAR(50), -- casa, apartamento, quarto, hotel, pousada
    
    -- Localização
    localizacao VARCHAR(255),
    endereco_completo JSONB,
    
    -- Capacidade
    capacidade_maxima INTEGER NOT NULL,
    numero_quartos INTEGER,
    numero_banheiros INTEGER,
    area_m2 DECIMAL(8,2),
    
    -- Preços
    preco_noite DECIMAL(8,2) NOT NULL,
    taxa_hospede_extra DECIMAL(6,2),
    taxa_limpeza DECIMAL(8,2),
    taxa_servico DECIMAL(6,2),
    desconto_semana DECIMAL(5,2),
    desconto_mes DECIMAL(5,2),
    
    -- Políticas
    comodidades TEXT[],
    regras_casa TEXT[],
    politica_cancelamento JSONB,
    tempo_minimo_permanencia INTEGER DEFAULT 1,
    tempo_maximo_permanencia INTEGER,
    
    -- Mídia
    fotos TEXT[],
    video_url TEXT,
    
    -- Disponibilidade
    disponibilidade JSONB,
    
    -- Avaliações
    avaliacao_media DECIMAL(3,2),
    total_avaliacoes INTEGER DEFAULT 0,
    
    -- Status
    ativo BOOLEAN DEFAULT true,
    verificado BOOLEAN DEFAULT false,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE IF NOT EXISTS reservas_alojamentos (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    usuario_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    alojamento_id UUID REFERENCES alojamentos(id) ON DELETE CASCADE,
    
    -- Datas
    data_checkin DATE NOT NULL,
    data_checkout DATE NOT NULL,
    
    -- Hóspedes
    numero_hospedes INTEGER NOT NULL,
    dados_hospedes JSONB,
    
    -- Valores
    valor_total DECIMAL(8,2),
    valor_diarias DECIMAL(8,2),
    valor_taxas DECIMAL(8,2),
    valor_desconto DECIMAL(8,2),
    
    -- Pagamento
    metodo_pagamento VARCHAR(50),
    
    -- Status
    status VARCHAR(50) DEFAULT 'confirmada', -- confirmada, checkin, checkout, cancelada, concluida
    observacoes TEXT,
    motivo_cancelamento TEXT,
    data_cancelamento TIMESTAMP,
    
    -- Código
    codigo_reserva VARCHAR(20) UNIQUE,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX idx_alojamentos_anfitriao ON alojamentos(anfitriao_id);
CREATE INDEX idx_alojamentos_tipo ON alojamentos(tipo_alojamento);
CREATE INDEX idx_alojamentos_ativo ON alojamentos(ativo);
CREATE INDEX idx_reservas_alojamentos_usuario ON reservas_alojamentos(usuario_id);
CREATE INDEX idx_reservas_alojamentos_alojamento ON reservas_alojamentos(alojamento_id);
CREATE INDEX idx_reservas_alojamentos_datas ON reservas_alojamentos(data_checkin, data_checkout);

-- ===========================
-- SISTEMA UNIVERSAL DE AVALIAÇÕES
-- ===========================
CREATE TABLE IF NOT EXISTS avaliacoes (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    usuario_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Tipo e referência
    tipo_avaliacao VARCHAR(50), -- imovel, viagem, pacote, sala, evento, restaurante, alojamento
    item_id UUID,
    
    -- Avaliação
    nota INTEGER NOT NULL CHECK (nota >= 1 AND nota <= 5),
    titulo VARCHAR(255),
    comentario TEXT,
    fotos TEXT[],
    
    -- Data
    data_experiencia DATE,
    
    -- Interação
    util INTEGER DEFAULT 0,
    status VARCHAR(50) DEFAULT 'ativo',
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX idx_avaliacoes_usuario ON avaliacoes(usuario_id);
CREATE INDEX idx_avaliacoes_tipo_item ON avaliacoes(tipo_avaliacao, item_id);
CREATE INDEX idx_avaliacoes_nota ON avaliacoes(nota);

-- ===========================
-- SISTEMA DE PAGAMENTOS
-- ===========================
CREATE TABLE IF NOT EXISTS transacoes (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    usuario_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Tipo
    tipo_transacao VARCHAR(50), -- pagamento, reembolso, comissao
    modulo VARCHAR(50), -- imobiliario, transporte, turismo, etc
    item_id UUID,
    
    -- Valores
    valor DECIMAL(10,2) NOT NULL,
    moeda VARCHAR(3) DEFAULT 'AOA',
    
    -- Pagamento
    metodo_pagamento VARCHAR(50), -- multicaixa, cartao, transferencia, dinheiro
    status_transacao VARCHAR(50), -- pendente, processando, confirmada, cancelada, estornada
    dados_pagamento JSONB,
    codigo_transacao VARCHAR(100),
    
    -- Datas
    data_confirmacao TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX idx_transacoes_usuario ON transacoes(usuario_id);
CREATE INDEX idx_transacoes_status ON transacoes(status_transacao);
CREATE INDEX idx_transacoes_data ON transacoes(created_at DESC);

-- ===========================
-- SISTEMA DE NOTIFICAÇÕES
-- ===========================
CREATE TABLE IF NOT EXISTS notificacoes (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    usuario_id UUID REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Conteúdo
    tipo_notificacao VARCHAR(50),
    titulo VARCHAR(255),
    mensagem TEXT,
    dados_adicionais JSONB,
    
    -- Status
    lida BOOLEAN DEFAULT false,
    data_leitura TIMESTAMP,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX idx_notificacoes_usuario ON notificacoes(usuario_id);
CREATE INDEX idx_notificacoes_usuario_lida ON notificacoes(usuario_id, lida);
CREATE INDEX idx_notificacoes_data ON notificacoes(created_at DESC);

-- ===========================
-- COMENTÁRIOS DE DOCUMENTAÇÃO
-- ===========================
COMMENT ON TABLE imoveis IS 'Listagens de imobiliário (casas, apartamentos, terrenos, escritórios)';
COMMENT ON TABLE viagens IS 'Histórico de viagens e integração com APIs de transporte (Tupuca, Yango, Bolt, Heetch)';
COMMENT ON TABLE pacotes_turisticos IS 'Pacotes turísticos oferecidos por fornecedores';
COMMENT ON TABLE salas_eventos IS 'Salas disponíveis para aluguel para eventos';
COMMENT ON TABLE eventos_ingressos IS 'Eventos com sistema de vendas de ingressos e QR codes';
COMMENT ON TABLE restaurantes IS 'Perfis de restaurantes com cardápio e reservas';
COMMENT ON TABLE alojamentos IS 'Hospedagens (casas, apartamentos, hotéis, pousadas)';
COMMENT ON TABLE avaliacoes IS 'Sistema universal de avaliações para todos os módulos';
COMMENT ON TABLE transacoes IS 'Histórico de todas as transações financeiras (Multicaixa, cartão, etc)';
COMMENT ON TABLE notificacoes IS 'Sistema de notificações push via Firebase Cloud Messaging';
