# Guia de Migrações - TudoAqui Backend

## Visão Geral

Este documento descreve o sistema de migrações de banco de dados para o TudoAqui Backend. As migrações são arquivos SQL que definem o esquema do PostgreSQL de forma versionada e controlada.

## Estrutura de Migrações

```
scripts/migrations/
├── 001_criar_usuarios.sql          # Tabela base: usuários, auditoria, KYC
├── 002_criar_modulos_marketplace.sql # Todos os módulos de negócio
├── run-migrations.sh                # Script Linux/Mac
├── run-migrations.bat               # Script Windows
└── README.md                        # Este arquivo
```

## Migrações Disponíveis

### 001_criar_usuarios.sql
**Descrição:** Cria tabelas base de usuários e autenticação

**Tabelas criadas:**
- `usuarios` - Perfil completo do usuário com suporte a KYC
- `auditoria_usuarios` - Auditoria de ações (logins, alterações)
- `verificacao_telefone` - Armazenamento de códigos SMS para 2FA
- Índices e triggers para performance e automação

**Função:**
- `atualizar_updated_at()` - Trigger automático para atualizar timestamps

**Extensões criadas:**
- `uuid-ossp` - Geração de UUIDs
- `postgis` - Suporte geoespacial para "perto de mim"

### 002_criar_modulos_marketplace.sql
**Descrição:** Cria todas as tabelas dos módulos de negócio

**Módulos (tabelas e relacionamentos):**

1. **IMOBILIÁRIO** (`imoveis`)
   - Listagens de imóveis (casa, apartamento, terreno, escritório)
   - Filtros por tipo, finalidade, preço, localização
   - Suporte a múltiplas fotos e vídeo

2. **TRANSPORTE** (`viagens`)
   - Histórico de viagens
   - Integração com APIs externas (Tupuca, Yango, Bolt, Heetch)
   - Rastreamento de status e avaliações

3. **TURISMO** (`pacotes_turisticos`, `reservas_turisticas`)
   - Catálogo de pacotes (aventura, cultural, praia, safari)
   - Sistema de reservas com código único
   - Diferenciais de preço (adulto/criança)

4. **EVENTOS E SALAS** (`salas_eventos`, `reservas_salas`)
   - Cadastro de salas para aluguel
   - Reservas com check-in/out via QR code
   - Gerenciamento de capacidade

5. **INGRESSOS** (`eventos_ingressos`, `ingressos_vendidos`)
   - Criação de eventos com setores
   - Vendas de ingressos com código QR e código de barras
   - Controle de acesso e uso

6. **RESTAURANTES** (`restaurantes`, `itens_cardapio`, `reservas_restaurantes`, `pedidos_restaurantes`)
   - Perfil do restaurante com avaliações
   - Cardápio com categorias e informações nutricionais
   - Sistema de reservas e pedidos (delivery/retirada/mesa)

7. **ALOJAMENTOS** (`alojamentos`, `reservas_alojamentos`)
   - Cadastro de hospedagens (casas, apartamentos, hotéis, pousadas)
   - Gerenciamento de disponibilidade e preços
   - Sistema de check-in/checkout

**Tabelas Universais:**
- `avaliacoes` - Sistema de avaliações para todos os módulos (1-5 estrelas)
- `transacoes` - Histórico de pagamentos e comissões
- `notificacoes` - Sistema de notificações push (Firebase)

## Como Executar Migrações

### Prerequisitos

**PostgreSQL 13+**
```bash
# Linux/Mac
brew install postgresql

# Windows (usar instalador: https://www.postgresql.org/download/windows/)
```

**Variáveis de Ambiente (.env)**
```env
DB_HOST=localhost
DB_PORT=5432
DB_NAME=tudo_aqui_db
DB_USER=tudo_aqui_admin
DB_PASSWORD=senha_forte_2024
```

### Opção 1: Usar Scripts Automatizados

**Windows:**
```bash
cd scripts
.\run-migrations.bat
```

**Linux/Mac:**
```bash
cd scripts
chmod +x run-migrations.sh
./run-migrations.sh
```

### Opção 2: Executar Manualmente com psql

**Criar o banco de dados (primeira vez):**
```bash
psql -U postgres -c "CREATE DATABASE tudo_aqui_db;"
psql -U postgres -d tudo_aqui_db -c "CREATE USER tudo_aqui_admin WITH PASSWORD 'senha_forte_2024';"
psql -U postgres -d tudo_aqui_db -c "ALTER DEFAULT PRIVILEGES IN SCHEMA public GRANT ALL ON TABLES TO tudo_aqui_admin;"
psql -U postgres -d tudo_aqui_db -c "GRANT ALL PRIVILEGES ON DATABASE tudo_aqui_db TO tudo_aqui_admin;"
```

**Executar migrações:**
```bash
# Migração 1 (usuários)
psql -h localhost -U tudo_aqui_admin -d tudo_aqui_db -f scripts/migrations/001_criar_usuarios.sql

# Migração 2 (módulos)
psql -h localhost -U tudo_aqui_admin -d tudo_aqui_db -f scripts/migrations/002_criar_modulos_marketplace.sql
```

### Opção 3: Usar Node.js (Recomendado para Produção)

```bash
# Instalar cliente PostgreSQL Node
npm install pg pg-migrate

# Executar migrações com pg-migrate (futuro)
npm run migrate:up
```

## Verificar Migrações

**Listar tabelas criadas:**
```sql
SELECT table_name 
FROM information_schema.tables 
WHERE table_schema = 'public'
ORDER BY table_name;
```

**Ver estrutura de uma tabela:**
```sql
\d usuarios
\d imoveis
-- etc
```

**Ver índices:**
```sql
SELECT indexname, tablename 
FROM pg_indexes 
WHERE schemaname = 'public'
ORDER BY tablename;
```

## Rollback (Desfazer Migrações)

**⚠️ CUIDADO:** Este comando apaga TODOS os dados!

```sql
-- Dropar tabelas (em ordem inversa de dependência)
DROP TABLE IF EXISTS notificacoes CASCADE;
DROP TABLE IF EXISTS transacoes CASCADE;
DROP TABLE IF EXISTS avaliacoes CASCADE;
DROP TABLE IF EXISTS ingressos_vendidos CASCADE;
DROP TABLE IF EXISTS eventos_ingressos CASCADE;
DROP TABLE IF EXISTS pedidos_restaurantes CASCADE;
DROP TABLE IF EXISTS reservas_restaurantes CASCADE;
DROP TABLE IF EXISTS itens_cardapio CASCADE;
DROP TABLE IF EXISTS restaurantes CASCADE;
DROP TABLE IF EXISTS reservas_alojamentos CASCADE;
DROP TABLE IF EXISTS alojamentos CASCADE;
DROP TABLE IF EXISTS reservas_salas CASCADE;
DROP TABLE IF EXISTS salas_eventos CASCADE;
DROP TABLE IF EXISTS reservas_turisticas CASCADE;
DROP TABLE IF EXISTS pacotes_turisticos CASCADE;
DROP TABLE IF EXISTS viagens CASCADE;
DROP TABLE IF EXISTS imoveis CASCADE;
DROP TABLE IF EXISTS verificacao_telefone CASCADE;
DROP TABLE IF EXISTS auditoria_usuarios CASCADE;
DROP TABLE IF EXISTS usuarios CASCADE;

-- Dropar funções
DROP FUNCTION IF EXISTS atualizar_updated_at() CASCADE;
```

## Estrutura de Dados

### Fluxo de Usuário Típico

```
1. REGISTRO (KYC)
   usuarios.kyc_status = 'pendente'
   ↓
   (Análise de documentos)
   ↓
   usuarios.kyc_status = 'aprovado'
   usuarios.selo_verificado = true

2. CRIAÇÃO DE ANÚNCIOS
   (Apenas com selo_verificado = true)
   imoveis / restaurantes / alojamentos / etc.

3. TRANSAÇÕES
   transacoes (pagamento, comissão)
   ↓
   usuarios (saldo atualizado)

4. FEEDBACK
   avaliacoes (1-5 estrelas)
   ↓
   Média calculada (restaurantes.avaliacao_media, etc)
```

### Relacionamentos Principais

```
usuarios (1) ──→ (N) imoveis
usuarios (1) ──→ (N) viagens
usuarios (1) ──→ (N) restaurantes
usuarios (1) ──→ (N) alojamentos
usuarios (1) ──→ (N) transacoes

restaurantes (1) ──→ (N) itens_cardapio
restaurantes (1) ──→ (N) pedidos_restaurantes
restaurantes (1) ──→ (N) reservas_restaurantes

eventos_ingressos (1) ──→ (N) ingressos_vendidos
salas_eventos (1) ──→ (N) reservas_salas
pacotes_turisticos (1) ──→ (N) reservas_turisticas
alojamentos (1) ──→ (N) reservas_alojamentos
```

## Performance e Índices

Todas as migrações incluem índices otimizados para queries comuns:

**Ejemplos de índices criados:**
- `idx_usuarios_email` - Busca por email (login)
- `idx_imoveis_tipo_finalidade` - Filtro de tipo/finalidade
- `idx_imoveis_preco` - Ordenação por preço
- `idx_viagens_usuario_status` - Histórico de viagens
- `idx_restaurantes_ativo` - Filtro de restaurantes ativos
- `idx_notificacoes_usuario_lida` - Notificações não lidas

## Dicas de Segurança

1. **Sempre fazer backup antes de executar migrações em produção**
   ```bash
   pg_dump tudo_aqui_db > backup_$(date +%Y%m%d_%H%M%S).sql
   ```

2. **Usar transações para migrações personalizadas**
   ```sql
   BEGIN;
   -- suas alterações
   COMMIT; -- ou ROLLBACK;
   ```

3. **Testar em ambiente de desenvolvimento primeiro**

4. **Documentar migrações personalizadas com comentários**

## Próximos Passos

1. ✅ Executar migrações (001 e 002)
2. ✅ Verificar esquema com `\d` no psql
3. 🔄 Implementar seeds (dados iniciais) em 003_seeds.sql
4. 🔄 Adicionar políticas de privacidade Row Level Security (RLS)
5. 🔄 Implementar backups automáticos com pg_dump

## Troubleshooting

**Erro: "database does not exist"**
```bash
# Criar banco de dados
createdb -U postgres tudo_aqui_db
```

**Erro: "permission denied"**
```bash
# Verificar permissões do usuário
psql -U postgres -d tudo_aqui_db -c "GRANT ALL PRIVILEGES ON DATABASE tudo_aqui_db TO tudo_aqui_admin;"
```

**Erro: "relation already exists"**
```bash
# Migração já foi executada (seguro ignorar)
# Ou dropar e re-executar
DROP TABLE IF EXISTS usuarios CASCADE;
```

## Suporte

Para dúvidas ou problemas:
1. Verificar logs de erro: `psql ... 2>&1 | tail -50`
2. Consultar documentação PostgreSQL: https://www.postgresql.org/docs/13/
3. Revisar comentários nas migrações SQL

---

**Última atualização:** 2026-01-22
**Versão:** 1.0
**Autor:** TudoAqui Dev Team
