#!/bin/bash

# Script de Execução de Migrações - TudoAqui Backend
# Este script executa todas as migrações SQL em ordem sequencial

set -e  # Exit on error

# Cores para output
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Configurações
DB_HOST="${DB_HOST:=localhost}"
DB_PORT="${DB_PORT:=5432}"
DB_NAME="${DB_NAME:=tudo_aqui_db}"
DB_USER="${DB_USER:=tudo_aqui_admin}"
DB_PASSWORD="${DB_PASSWORD:=senha_forte_2024}"

MIGRATIONS_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)/migrations"

echo -e "${YELLOW}=== TudoAqui - Executor de Migrações ===${NC}"
echo ""
echo "Configurações:"
echo "  Host: $DB_HOST"
echo "  Port: $DB_PORT"
echo "  Database: $DB_NAME"
echo "  User: $DB_USER"
echo ""

# Verificar se o diretório de migrações existe
if [ ! -d "$MIGRATIONS_DIR" ]; then
    echo -e "${RED}✗ Diretório de migrações não encontrado: $MIGRATIONS_DIR${NC}"
    exit 1
fi

# Contar migrações
MIGRATION_COUNT=$(find "$MIGRATIONS_DIR" -name "*.sql" | wc -l)
if [ $MIGRATION_COUNT -eq 0 ]; then
    echo -e "${RED}✗ Nenhum arquivo de migração encontrado em $MIGRATIONS_DIR${NC}"
    exit 1
fi

echo -e "${GREEN}✓ Encontradas $MIGRATION_COUNT migrações${NC}"
echo ""

# Executar migrações em ordem
export PGPASSWORD="$DB_PASSWORD"
MIGRATION_COUNT=0
ERROR_COUNT=0

for migration_file in $(ls -1 "$MIGRATIONS_DIR"/*.sql | sort); do
    MIGRATION_NAME=$(basename "$migration_file")
    MIGRATION_COUNT=$((MIGRATION_COUNT + 1))
    
    echo -n "[$MIGRATION_COUNT] Executando $MIGRATION_NAME ... "
    
    if psql -h "$DB_HOST" -p "$DB_PORT" -d "$DB_NAME" -U "$DB_USER" -f "$migration_file" > /dev/null 2>&1; then
        echo -e "${GREEN}✓ OK${NC}"
    else
        echo -e "${RED}✗ ERRO${NC}"
        ERROR_COUNT=$((ERROR_COUNT + 1))
        
        # Tentar executar novamente com output de erro
        echo "  Detalhes do erro:"
        psql -h "$DB_HOST" -p "$DB_PORT" -d "$DB_NAME" -U "$DB_USER" -f "$migration_file" 2>&1 | sed 's/^/    /'
    fi
done

unset PGPASSWORD

echo ""
echo -e "${YELLOW}=== Resumo ===${NC}"
echo "Total de migrações: $MIGRATION_COUNT"
echo "Sucesso: $((MIGRATION_COUNT - ERROR_COUNT))"

if [ $ERROR_COUNT -gt 0 ]; then
    echo -e "${RED}Erros: $ERROR_COUNT${NC}"
    exit 1
else
    echo -e "${GREEN}✓ Todas as migrações executadas com sucesso!${NC}"
    exit 0
fi
