const express = require('express');
const cors = require('cors');
const compression = require('compression');
const helmet = require('helmet');
const { authLimiter, apiLimiter, securityMiddleware } = require('./middleware/security.middleware');
const { metricsMiddleware } = require('./middleware/monitoring.middleware');
const swaggerSetup = require('./config/swagger');

// Rotas
const authRoutes = require('./modules/auth/auth.routes');
const imoveisRoutes = require('./modules/imoveis/imoveis.routes');
const pagamentosRoutes = require('./modules/pagamentos/pagamentos.routes');
const monitoringRoutes = require('./routes/monitoring.routes');

const app = express();

// Middlewares básicos
app.use(helmet({ crossOriginResourcePolicy: false }));
app.use(cors({ origin: '*', methods: ['GET', 'POST', 'PUT', 'PATCH', 'DELETE'], allowedHeaders: ['Content-Type', 'Authorization'] }));
app.use(compression());
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Documentação da API (Swagger/OpenAPI)
swaggerSetup(app);

// Segurança (sanitização, XSS, HPP, headers)
securityMiddleware(app);

// Monitoramento e métricas (aplicar antes das rotas de negócio)
app.use(metricsMiddleware);

// Rotas de observabilidade (health, metrics)
app.use('/', monitoringRoutes);

// Rotas versionadas
app.use('/api/v1/auth', authLimiter, authRoutes);
app.use('/api/v1/imoveis', apiLimiter, imoveisRoutes);
app.use('/api/v1/pagamentos', apiLimiter, pagamentosRoutes);

// 404 padrão
app.use((req, res) => {
  res.status(404).json({ success: false, error: 'Rota não encontrada' });
});

module.exports = app;
