const jwt = require('jsonwebtoken');
const { pool, redis } = require('../config/database');

/**
 * Middleware de autenticação via JWT
 * Valida o token e o adiciona ao objeto req.user
 */
const autenticar = async (req, res, next) => {
  try {
    const token = req.headers.authorization?.split(' ')[1];

    if (!token) {
      return res.status(401).json({
        success: false,
        error: 'Token não fornecido'
      });
    }

    // Verificar se o token está na lista negra (logout)
    const isBlacklisted = await redis.exists(`blacklist:${token}`);
    if (isBlacklisted) {
      return res.status(401).json({
        success: false,
        error: 'Token inválido ou expirado'
      });
    }

    // Verificar token e garantir que o usuário continua ativo
    const decoded = jwt.verify(token, process.env.JWT_SECRET);

    const query = 'SELECT id, nome, email, telefone, tipo_usuario, status, email_verificado, telefone_verificado, selo_verificado FROM usuarios WHERE id = $1';
    const result = await pool.query(query, [decoded.userId]);

    if (result.rows.length === 0 || result.rows[0].status !== 'ativo') {
      return res.status(401).json({
        success: false,
        error: 'Usuário não encontrado ou inativo'
      });
    }

    // req.user contém o usuário do banco; req.auth preserva o payload do token
    req.auth = decoded;
    req.user = { ...result.rows[0], userId: result.rows[0].id };
    req.token = token;

    next();

  } catch (error) {
    if (error.name === 'TokenExpiredError') {
      return res.status(401).json({
        success: false,
        error: 'Token expirado',
        code: 'TOKEN_EXPIRED'
      });
    }

    if (error.name === 'JsonWebTokenError') {
      return res.status(401).json({
        success: false,
        error: 'Token inválido'
      });
    }

    res.status(500).json({
      success: false,
      error: 'Erro na autenticação'
    });
  }
};

/**
 * Middleware para verificar se o usuário é vendedor
 * Deve ser usado após o middleware 'autenticar'
 */
const verificarVendedor = async (req, res, next) => {
  try {
    const query = 'SELECT tipo_usuario FROM usuarios WHERE id = $1 AND status = $2';
    const result = await pool.query(query, [req.user.id, 'ativo']);

    if (result.rows.length === 0 || result.rows[0].tipo_usuario !== 'vendedor') {
      return res.status(403).json({
        success: false,
        error: 'Acesso negado: apenas vendedores podem acessar este recurso'
      });
    }

    next();

  } catch (error) {
    console.error('Erro ao verificar vendedor:', error);
    res.status(500).json({
      success: false,
      error: 'Erro na autorização'
    });
  }
};

/**
 * Middleware para verificar se o usuário tem Selo de Verificado (KYC aprovado)
 * Deve ser usado após o middleware 'autenticar'
 */
const verificarSeloVerificado = async (req, res, next) => {
  try {
    const query = 'SELECT selo_verificado FROM usuarios WHERE id = $1 AND status = $2';
    const result = await pool.query(query, [req.user.id, 'ativo']);

    if (result.rows.length === 0 || !result.rows[0].selo_verificado) {
      return res.status(403).json({
        success: false,
        error: 'Acesso negado: é necessário passar por verificação de identidade (KYC) para acessar este recurso',
        code: 'KYC_REQUIRED'
      });
    }

    next();

  } catch (error) {
    console.error('Erro ao verificar KYC:', error);
    res.status(500).json({
      success: false,
      error: 'Erro na autorização'
    });
  }
};

/**
 * Middleware para verificar se o usuário é administrador
 * Deve ser usado após o middleware 'autenticar'
 */
const verificarAdmin = async (req, res, next) => {
  try {
    const query = 'SELECT is_admin FROM usuarios WHERE id = $1 AND status = $2';
    const result = await pool.query(query, [req.user.id, 'ativo']);

    if (result.rows.length === 0 || !result.rows[0].is_admin) {
      return res.status(403).json({
        success: false,
        error: 'Acesso negado: permissões insuficientes'
      });
    }

    next();

  } catch (error) {
    console.error('Erro ao verificar admin:', error);
    res.status(500).json({
      success: false,
      error: 'Erro na autorização'
    });
  }
};

// Middleware genérico para checar tipos de usuário permitidos
const autorizar = (...tipos) => (req, res, next) => {
  if (!tipos.includes(req.user.tipo_usuario)) {
    return res.status(403).json({
      success: false,
      error: 'Acesso negado. Permissão insuficiente.'
    });
  }
  next();
};

module.exports = {
  autenticar,
   autorizar,
  verificarVendedor,
  verificarSeloVerificado,
  verificarAdmin
};
