const { redis } = require('../config/database');

/**
 * Cache de respostas em Redis para endpoints idempotentes (GET)
 * Uso: router.get('/rota', cache(300), controller)
 */
const cache = (duration = 300) => {
  return async (req, res, next) => {
    try {
      // Só cacheia requisições GET
      if (req.method !== 'GET') return next();

      const key = `cache:${req.method}:${req.originalUrl}:${JSON.stringify(req.query || {})}`;
      const cached = await redis.get(key);

      if (cached) {
        return res.json(JSON.parse(cached));
      }

      // Intercepta res.json para salvar resposta bem-sucedida
      const originalJson = res.json.bind(res);
      res.json = async (data) => {
        try {
          // Apenas cacheia respostas 2xx
          if (res.statusCode >= 200 && res.statusCode < 300) {
            await redis.setex(key, duration, JSON.stringify(data));
          }
        } catch (err) {
          console.error('Erro ao gravar cache:', err.message);
        }
        return originalJson(data);
      };

      return next();
    } catch (error) {
      console.error('Erro no middleware de cache:', error.message);
      return next();
    }
  };
};

/**
 * Remove entradas de cache que combinem com um padrão
 * Uso: router.post('/rota', clearCache('imoveis'), controller)
 */
const clearCache = (pattern = '') => {
  return async (req, res, next) => {
    try {
      const stream = redis.scanStream({ match: `cache:*${pattern}*` });
      const keys = [];

      stream.on('data', (resultKeys) => {
        for (const k of resultKeys) keys.push(k);
      });

      stream.on('end', async () => {
        if (keys.length > 0) {
          await redis.del(keys);
        }
      });

      return next();
    } catch (error) {
      console.error('Erro ao limpar cache:', error.message);
      return next();
    }
  };
};

module.exports = { cache, clearCache };
