const prometheus = require('prom-client');
const { pool, redis } = require('../config/database');

/**
 * Monitoramento com Prometheus
 * Coleta métricas de requisições HTTP, performance e health dos serviços
 */

// Coletor padrão do Prometheus (memoria, CPU, etc.)
prometheus.collectDefaultMetrics();

// ========== MÉTRICAS HTTP ==========

const httpRequestDuration = new prometheus.Histogram({
  name: 'http_request_duration_seconds',
  help: 'Duração das requisições HTTP em segundos',
  labelNames: ['method', 'route', 'status_code'],
  buckets: [0.1, 0.5, 1, 2, 5, 10] // em segundos
});

const httpRequestTotal = new prometheus.Counter({
  name: 'http_requests_total',
  help: 'Número total de requisições HTTP',
  labelNames: ['method', 'route', 'status_code']
});

const httpRequestSize = new prometheus.Histogram({
  name: 'http_request_size_bytes',
  help: 'Tamanho das requisições HTTP em bytes',
  labelNames: ['method', 'route'],
  buckets: [1000, 5000, 10000, 50000, 100000, 500000]
});

const httpResponseSize = new prometheus.Histogram({
  name: 'http_response_size_bytes',
  help: 'Tamanho das respostas HTTP em bytes',
  labelNames: ['method', 'route', 'status_code'],
  buckets: [1000, 5000, 10000, 50000, 100000, 500000]
});

// ========== MÉTRICAS DE NEGÓCIO ==========

const activeUsers = new prometheus.Gauge({
  name: 'active_users_total',
  help: 'Número de usuários ativos no momento'
});

const transacoesProcessadas = new prometheus.Counter({
  name: 'transacoes_processadas_total',
  help: 'Total de transações processadas',
  labelNames: ['status', 'metodo_pagamento']
});

const imoveisListados = new prometheus.Gauge({
  name: 'imoveis_listados_total',
  help: 'Número total de imóveis listados'
});

const usuariosVerificados = new prometheus.Gauge({
  name: 'usuarios_verificados_total',
  help: 'Número de usuários com KYC aprovado'
});

// ========== MÉTRICAS DE BANCO DE DADOS ==========

const dbConnectionPoolSize = new prometheus.Gauge({
  name: 'db_connection_pool_size',
  help: 'Tamanho do pool de conexões PostgreSQL'
});

const dbQueryDuration = new prometheus.Histogram({
  name: 'db_query_duration_seconds',
  help: 'Duração das queries PostgreSQL em segundos',
  labelNames: ['query_type'],
  buckets: [0.01, 0.05, 0.1, 0.5, 1, 5]
});

const dbErrors = new prometheus.Counter({
  name: 'db_errors_total',
  help: 'Total de erros de banco de dados',
  labelNames: ['error_type']
});

// ========== MÉTRICAS DE CACHE ==========

const redisOperationDuration = new prometheus.Histogram({
  name: 'redis_operation_duration_seconds',
  help: 'Duração das operações Redis em segundos',
  labelNames: ['operation'],
  buckets: [0.001, 0.01, 0.05, 0.1, 0.5]
});

const redisErrors = new prometheus.Counter({
  name: 'redis_errors_total',
  help: 'Total de erros de Redis',
  labelNames: ['operation']
});

// ========== MIDDLEWARE DE COLETA DE MÉTRICAS ==========

const metricsMiddleware = (req, res, next) => {
  const start = Date.now();
  const route = req.route ? req.route.path : req.path;

  // Tamanho da requisição
  const requestSize = parseInt(req.headers['content-length'] || 0, 10);
  if (requestSize > 0) {
    httpRequestSize.labels(req.method, route).observe(requestSize);
  }

  // Interceptar res.json para capturar tamanho de resposta
  const originalJson = res.json.bind(res);
  res.json = function(data) {
    const responseSize = JSON.stringify(data).length;
    httpResponseSize.labels(req.method, route, res.statusCode).observe(responseSize);
    return originalJson(data);
  };

  res.on('finish', () => {
    const duration = (Date.now() - start) / 1000; // em segundos
    const statusCode = res.statusCode;

    httpRequestDuration
      .labels(req.method, route, statusCode)
      .observe(duration);

    httpRequestTotal
      .labels(req.method, route, statusCode)
      .inc();
  });

  next();
};

// ========== HEALTH CHECKS ==========

async function checkDatabase() {
  try {
    const startTime = Date.now();
    await pool.query('SELECT 1');
    const duration = (Date.now() - startTime) / 1000;

    dbConnectionPoolSize.set(pool.totalCount || 20);

    return {
      status: 'healthy',
      message: 'Banco de dados responsivo',
      response_time: `${(duration * 1000).toFixed(2)}ms`,
      timestamp: new Date().toISOString()
    };
  } catch (error) {
    dbErrors.labels('health_check').inc();
    return {
      status: 'unhealthy',
      message: `Falha na conexão com banco: ${error.message}`,
      timestamp: new Date().toISOString()
    };
  }
}

async function checkRedis() {
  try {
    const startTime = Date.now();
    await redis.ping();
    const duration = (Date.now() - startTime) / 1000;

    return {
      status: 'healthy',
      message: 'Redis responsivo',
      response_time: `${(duration * 1000).toFixed(2)}ms`,
      timestamp: new Date().toISOString()
    };
  } catch (error) {
    redisErrors.labels('health_check').inc();
    return {
      status: 'unhealthy',
      message: `Falha na conexão com Redis: ${error.message}`,
      timestamp: new Date().toISOString()
    };
  }
}

async function checkStorage() {
  try {
    // Verificar se diretório de uploads existe e tem espaço (simplificado)
    const fs = require('fs').promises;
    const os = require('os');
    const diskSpace = os.freemem() / (1024 * 1024 * 1024); // em GB

    return {
      status: diskSpace > 1 ? 'healthy' : 'unhealthy',
      message: diskSpace > 1 ? 'Storage acessível' : 'Espaço em disco baixo',
      available_gb: diskSpace.toFixed(2),
      timestamp: new Date().toISOString()
    };
  } catch (error) {
    return {
      status: 'unhealthy',
      message: `Falha ao verificar storage: ${error.message}`,
      timestamp: new Date().toISOString()
    };
  }
}

/**
 * Health check detalhado para orquestração
 * Retorna status 503 se algum serviço crítico falhar
 */
const healthCheck = async (req, res) => {
  try {
    const checks = {
      database: await checkDatabase(),
      redis: await checkRedis(),
      storage: await checkStorage()
    };

    // Serviços críticos devem estar saudáveis
    const allHealthy = checks.database.status === 'healthy' && checks.redis.status === 'healthy';

    const response = {
      status: allHealthy ? 'healthy' : 'degraded',
      timestamp: new Date().toISOString(),
      uptime_seconds: process.uptime(),
      checks: checks
    };

    res.status(allHealthy ? 200 : 503).json(response);
  } catch (error) {
    res.status(500).json({
      status: 'error',
      message: `Erro ao executar health check: ${error.message}`,
      timestamp: new Date().toISOString()
    });
  }
};

/**
 * Health check leve (apenas ping)
 * Usado por load balancers que precisam de resposta rápida
 */
const quickHealthCheck = (req, res) => {
  res.status(200).json({
    status: 'ok',
    timestamp: new Date().toISOString()
  });
};

// ========== FUNÇÕES AUXILIARES PARA ATUALIZAR MÉTRICAS ==========

const updateMetrics = {
  /**
   * Incrementar contador de transações
   */
  recordTransaction: (status, metodo) => {
    transacoesProcessadas.labels(status, metodo).inc();
  },

  /**
   * Atualizar número de imóveis listados
   */
  async updateImoveisCount() {
    try {
      const result = await pool.query('SELECT COUNT(*) FROM imoveis WHERE status = $1', ['ativo']);
      imoveisListados.set(parseInt(result.rows[0].count, 10));
    } catch (error) {
      console.error('Erro ao contar imóveis:', error.message);
    }
  },

  /**
   * Atualizar número de usuários verificados
   */
  async updateUsuariosVerificadosCount() {
    try {
      const result = await pool.query('SELECT COUNT(*) FROM usuarios WHERE selo_verificado = $1', [true]);
      usuariosVerificados.set(parseInt(result.rows[0].count, 10));
    } catch (error) {
      console.error('Erro ao contar usuários verificados:', error.message);
    }
  },

  /**
   * Registrar duração de query
   */
  recordQueryDuration: (type, duration) => {
    dbQueryDuration.labels(type).observe(duration);
  },

  /**
   * Registrar erro de banco
   */
  recordDBError: (type) => {
    dbErrors.labels(type).inc();
  },

  /**
   * Registrar duração de operação Redis
   */
  recordRedisOperation: (operation, duration) => {
    redisOperationDuration.labels(operation).observe(duration);
  },

  /**
   * Registrar erro de Redis
   */
  recordRedisError: (operation) => {
    redisErrors.labels(operation).inc();
  }
};

module.exports = {
  metricsMiddleware,
  healthCheck,
  quickHealthCheck,
  updateMetrics,
  register: prometheus.register,
  // Exportar métricas para testes/debug
  metrics: {
    httpRequestDuration,
    httpRequestTotal,
    httpRequestSize,
    httpResponseSize,
    activeUsers,
    transacoesProcessadas,
    imoveisListados,
    usuariosVerificados,
    dbConnectionPoolSize,
    dbQueryDuration,
    dbErrors,
    redisOperationDuration,
    redisErrors
  }
};
