const rateLimit = require('express-rate-limit');
const mongoSanitize = require('express-mongo-sanitize');
const xss = require('xss-clean');
const hpp = require('hpp');

// Rate limiting específico para autenticação (tenta bloquear brute-force)
const authLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutos
  max: parseInt(process.env.RATE_LIMIT_AUTH_MAX || '5', 10),
  standardHeaders: true,
  legacyHeaders: false,
  message: 'Muitas tentativas de autenticação, tente novamente mais tarde'
});

// Rate limiting genérico da API
const apiLimiter = rateLimit({
  windowMs: parseInt(process.env.RATE_LIMIT_WINDOW_MS || String(15 * 60 * 1000), 10),
  max: parseInt(process.env.RATE_LIMIT_MAX_REQUESTS || '100', 10),
  standardHeaders: true,
  legacyHeaders: false,
  message: 'Muitas requisições, tente novamente mais tarde'
});

// Middleware de segurança comum
const securityMiddleware = (app) => {
  // Sanitiza payload contra NoSQL injection (mantém seguro caso exista integração futura com Mongo)
  app.use(mongoSanitize());

  // Proteção contra XSS
  app.use(xss());

  // Previni HTTP Parameter Pollution em queries
  app.use(hpp({
    whitelist: ['preco_min', 'preco_max', 'quartos', 'banheiros', 'area_min', 'area_max']
  }));

  // Headers de segurança
  app.use((req, res, next) => {
    res.setHeader('X-Content-Type-Options', 'nosniff');
    res.setHeader('X-Frame-Options', 'deny');
    res.setHeader('X-XSS-Protection', '1; mode=block');
    res.setHeader('Strict-Transport-Security', 'max-age=31536000; includeSubDomains');
    next();
  });
};

module.exports = { authLimiter, apiLimiter, securityMiddleware };
