const crypto = require('crypto');
const { pool } = require('../../config/database');

class AlojamentosController {
  async listarAlojamentos(req, res) {
    try {
      const {
        localizacao,
        tipo_alojamento,
        capacidade,
        preco_min,
        preco_max,
        data_checkin,
        data_checkout,
        pagina = '1',
        limite = '20',
        ordenar = 'created_at',
        ordem = 'DESC'
      } = req.query;

      const page = Math.max(parseInt(pagina, 10) || 1, 1);
      const perPage = Math.min(Math.max(parseInt(limite, 10) || 20, 1), 100);
      const offset = (page - 1) * perPage;

      const where = ['a.ativo = true'];
      const params = [];

      if (tipo_alojamento) {
        params.push(tipo_alojamento);
        where.push(`a.tipo_alojamento = $${params.length}`);
      }

      if (localizacao) {
        params.push(`%${localizacao}%`);
        where.push(`(a.localizacao ILIKE $${params.length} OR a.endereco_completo->>'cidade' ILIKE $${params.length})`);
      }

      if (capacidade !== undefined) {
        params.push(Number(capacidade));
        where.push(`a.capacidade_maxima >= $${params.length}`);
      }

      if (preco_min !== undefined) {
        params.push(Number(preco_min));
        where.push(`a.preco_noite >= $${params.length}`);
      }

      if (preco_max !== undefined) {
        params.push(Number(preco_max));
        where.push(`a.preco_noite <= $${params.length}`);
      }

      // Filtro de disponibilidade por datas
      if (data_checkin && data_checkout) {
        params.push(data_checkin, data_checkout);
        const checkInIdx = params.length - 1;
        const checkOutIdx = params.length;
        where.push(`NOT EXISTS (
          SELECT 1 FROM reservas_alojamentos ra
          WHERE ra.alojamento_id = a.id
          AND ra.status IN ('confirmada', 'checkin')
          AND ra.data_checkin < $${checkOutIdx}::date
          AND ra.data_checkout > $${checkInIdx}::date
        )`);
      }

      const orderMap = {
        preco: 'a.preco_noite',
        capacidade: 'a.capacidade_maxima',
        avaliacao: 'a.avaliacao_media',
        created_at: 'a.created_at'
      };
      const orderField = orderMap[ordenar] || 'a.created_at';
      const orderDirection = ordem && ordem.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';

      const query = `
        SELECT a.*, u.nome AS anfitriao_nome,
               COALESCE(a.avaliacao_media, 0) AS avaliacao_media,
               COALESCE(a.total_avaliacoes, 0) AS total_avaliacoes
        FROM alojamentos a
        JOIN usuarios u ON a.anfitriao_id = u.id
        WHERE ${where.join(' AND ')}
        ORDER BY ${orderField} ${orderDirection}
        LIMIT $${params.length + 1} OFFSET $${params.length + 2}
      `;

      const result = await pool.query(query, [...params, perPage, offset]);

      const countQuery = `
        SELECT COUNT(*) AS total
        FROM alojamentos a
        WHERE ${where.join(' AND ')}
      `;
      const countResult = await pool.query(countQuery, params);
      const total = Number(countResult.rows[0].total) || 0;

      res.json({
        success: true,
        alojamentos: result.rows,
        paginacao: {
          pagina: page,
          limite: perPage,
          total,
          total_paginas: Math.ceil(total / perPage)
        }
      });
    } catch (error) {
      console.error('Erro ao listar alojamentos:', error);
      res.status(500).json({ success: false, error: 'Erro ao listar alojamentos' });
    }
  }

  async buscarAlojamentoPorId(req, res) {
    try {
      const { id } = req.params;

      const query = `
        SELECT a.*, u.nome AS anfitriao_nome, u.foto_url AS anfitriao_foto,
               COALESCE(a.avaliacao_media, 0) AS avaliacao_media,
               COALESCE(a.total_avaliacoes, 0) AS total_avaliacoes
        FROM alojamentos a
        JOIN usuarios u ON a.anfitriao_id = u.id
        WHERE a.id = $1 AND a.ativo = true
      `;

      const result = await pool.query(query, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Alojamento não encontrado' });
      }

      res.json({
        success: true,
        alojamento: result.rows[0]
      });
    } catch (error) {
      console.error('Erro ao buscar alojamento:', error);
      res.status(500).json({ success: false, error: 'Erro ao buscar alojamento' });
    }
  }

  async verificarDisponibilidade(req, res) {
    try {
      const { alojamento_id, data_checkin, data_checkout } = req.query;

      if (!alojamento_id || !data_checkin || !data_checkout) {
        return res.status(400).json({
          success: false,
          error: 'Parâmetros obrigatórios: alojamento_id, data_checkin, data_checkout'
        });
      }

      const resultado = await this._verificarDisponibilidade(
        alojamento_id,
        data_checkin,
        data_checkout
      );

      res.json({
        success: true,
        disponivel: resultado.disponivel,
        mensagem: resultado.disponivel
          ? 'Alojamento disponível nas datas selecionadas'
          : 'Alojamento não disponível',
        datas_ocupadas: resultado.datas_ocupadas || [],
        datas_alternativas: resultado.datas_alternativas || []
      });
    } catch (error) {
      console.error('Erro ao verificar disponibilidade:', error);
      res.status(500).json({ success: false, error: 'Erro ao verificar disponibilidade' });
    }
  }

  async _verificarDisponibilidade(alojamentoId, dataCheckin, dataCheckout) {
    const query = `
      SELECT COUNT(*) AS conflitos
      FROM reservas_alojamentos
      WHERE alojamento_id = $1
      AND status IN ('confirmada', 'checkin')
      AND data_checkin < $3::date
      AND data_checkout > $2::date
    `;

    const result = await pool.query(query, [alojamentoId, dataCheckin, dataCheckout]);
    const conflitos = Number(result.rows[0].conflitos) || 0;

    if (conflitos > 0) {
      // Buscar datas ocupadas
      const ocupadasQuery = `
        SELECT data_checkin, data_checkout
        FROM reservas_alojamentos
        WHERE alojamento_id = $1
        AND status IN ('confirmada', 'checkin')
        AND data_checkout > CURRENT_DATE
        ORDER BY data_checkin
      `;
      const ocupadas = await pool.query(ocupadasQuery, [alojamentoId]);

      return {
        disponivel: false,
        datas_ocupadas: ocupadas.rows,
        datas_alternativas: []
      };
    }

    return { disponivel: true };
  }

  async calcularValorTotal(req, res) {
    try {
      const { alojamento_id, data_checkin, data_checkout, numero_hospedes } = req.query;

      if (!alojamento_id || !data_checkin || !data_checkout) {
        return res.status(400).json({
          success: false,
          error: 'Parâmetros obrigatórios: alojamento_id, data_checkin, data_checkout'
        });
      }

      const calculo = await this._calcularValorTotal(
        alojamento_id,
        data_checkin,
        data_checkout,
        Number(numero_hospedes) || 1
      );

      res.json({
        success: true,
        ...calculo
      });
    } catch (error) {
      console.error('Erro ao calcular valor:', error);
      res.status(500).json({ success: false, error: 'Erro ao calcular valor' });
    }
  }

  async _calcularValorTotal(alojamentoId, dataCheckin, dataCheckout, numeroHospedes) {
    const query = `
      SELECT preco_noite, taxa_hospede_extra, taxa_limpeza, taxa_servico,
             desconto_semana, desconto_mes, capacidade_maxima
      FROM alojamentos
      WHERE id = $1
    `;

    const result = await pool.query(query, [alojamentoId]);

    if (result.rows.length === 0) {
      throw new Error('Alojamento não encontrado');
    }

    const alojamento = result.rows[0];

    // Calcular número de noites
    const checkin = new Date(dataCheckin);
    const checkout = new Date(dataCheckout);
    const diferencaDias = Math.ceil((checkout - checkin) / (1000 * 60 * 60 * 24));

    if (diferencaDias < 1) {
      throw new Error('Data de checkout deve ser posterior à data de check-in');
    }

    // Valor das diárias
    const valorDiarias = Number(alojamento.preco_noite) * diferencaDias;

    // Taxa de hóspede extra
    const hospedesExtras = Math.max(0, numeroHospedes - alojamento.capacidade_maxima);
    const valorHospedesExtras = hospedesExtras * (Number(alojamento.taxa_hospede_extra) || 0) * diferencaDias;

    // Taxas fixas
    const taxaLimpeza = Number(alojamento.taxa_limpeza) || 0;
    const taxaServico = Number(alojamento.taxa_servico) || 0;

    // Descontos
    let desconto = 0;
    if (diferencaDias >= 28 && alojamento.desconto_mes) {
      desconto = valorDiarias * (Number(alojamento.desconto_mes) / 100);
    } else if (diferencaDias >= 7 && alojamento.desconto_semana) {
      desconto = valorDiarias * (Number(alojamento.desconto_semana) / 100);
    }

    const valorTotal = valorDiarias + valorHospedesExtras + taxaLimpeza + taxaServico - desconto;

    return {
      numero_noites: diferencaDias,
      valor_diarias: valorDiarias,
      valor_hospedes_extras: valorHospedesExtras,
      taxa_limpeza: taxaLimpeza,
      taxa_servico: taxaServico,
      desconto: desconto,
      valor_total: Math.max(0, valorTotal)
    };
  }

  async fazerReserva(req, res) {
    const { alojamento_id, data_checkin, data_checkout, numero_hospedes, observacoes } = req.body;

    if (!alojamento_id || !data_checkin || !data_checkout || !numero_hospedes) {
      return res.status(400).json({
        success: false,
        error: 'Dados obrigatórios: alojamento_id, data_checkin, data_checkout, numero_hospedes'
      });
    }

    const client = await pool.connect();

    try {
      await client.query('BEGIN');

      // Verificar alojamento existe e está ativo
      const alojamentoQuery = 'SELECT * FROM alojamentos WHERE id = $1 AND ativo = true FOR UPDATE';
      const alojamentoResult = await client.query(alojamentoQuery, [alojamento_id]);

      if (alojamentoResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ success: false, error: 'Alojamento não encontrado' });
      }

      // Verificar disponibilidade com lock
      const disponivel = await this._verificarDisponibilidade(alojamento_id, data_checkin, data_checkout);

      if (!disponivel.disponivel) {
        await client.query('ROLLBACK');
        return res.status(400).json({
          success: false,
          error: 'Alojamento não disponível nas datas selecionadas',
          datas_ocupadas: disponivel.datas_ocupadas,
          datas_alternativas: disponivel.datas_alternativas
        });
      }

      // Calcular valor total
      const calculo = await this._calcularValorTotal(alojamento_id, data_checkin, data_checkout, numero_hospedes);

      // Gerar código de reserva único
      const codigoReserva = this.gerarCodigoReserva();

      // Criar reserva
      const reservaQuery = `
        INSERT INTO reservas_alojamentos 
        (usuario_id, alojamento_id, data_checkin, data_checkout, numero_hospedes, 
         valor_total, valor_diarias, valor_taxas, valor_desconto, 
         status, observacoes, codigo_reserva)
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, 'confirmada', $10, $11)
        RETURNING *
      `;

      const valorTaxas = calculo.taxa_limpeza + calculo.taxa_servico + calculo.valor_hospedes_extras;

      const reservaResult = await client.query(reservaQuery, [
        req.user.id,
        alojamento_id,
        data_checkin,
        data_checkout,
        numero_hospedes,
        calculo.valor_total,
        calculo.valor_diarias,
        valorTaxas,
        calculo.desconto,
        observacoes,
        codigoReserva
      ]);

      // Registrar transação
      await client.query(
        `
        INSERT INTO transacoes 
        (usuario_id, tipo_transacao, modulo, item_id, valor, metodo_pagamento, status_transacao)
        VALUES ($1, 'pagamento', 'alojamentos', $2, $3, 'pendente', 'pendente')
      `,
        [req.user.id, alojamento_id, calculo.valor_total]
      );

      await client.query('COMMIT');

      const reserva = reservaResult.rows[0];

      // Enviar confirmação (assíncrono)
      this.enviarConfirmacao(req.user.id, reserva, alojamentoResult.rows[0]).catch((err) =>
        console.error('Erro ao enviar confirmação:', err)
      );

      res.json({
        success: true,
        message: 'Reserva criada com sucesso',
        reserva: {
          ...reserva,
          detalhes_calculo: calculo
        },
        codigo_reserva: codigoReserva
      });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error('Erro ao fazer reserva:', error);
      res.status(500).json({ success: false, error: 'Erro ao processar reserva' });
    } finally {
      client.release();
    }
  }

  async minhasReservas(req, res) {
    try {
      const query = `
        SELECT ra.*, a.titulo AS alojamento_titulo, a.localizacao AS alojamento_localizacao,
               a.endereco_completo AS alojamento_endereco, a.fotos[1] AS alojamento_foto,
               u.nome AS anfitriao_nome, u.telefone AS anfitriao_telefone
        FROM reservas_alojamentos ra
        JOIN alojamentos a ON ra.alojamento_id = a.id
        JOIN usuarios u ON a.anfitriao_id = u.id
        WHERE ra.usuario_id = $1
        ORDER BY ra.created_at DESC
      `;

      const result = await pool.query(query, [req.user.id]);

      res.json({
        success: true,
        reservas: result.rows,
        total: result.rows.length
      });
    } catch (error) {
      console.error('Erro ao buscar minhas reservas:', error);
      res.status(500).json({ success: false, error: 'Erro ao buscar reservas' });
    }
  }

  async cancelarReserva(req, res) {
    try {
      const { reserva_id } = req.params;
      const { motivo_cancelamento } = req.body;

      const query = `
        UPDATE reservas_alojamentos
        SET status = 'cancelada', 
            motivo_cancelamento = $1,
            data_cancelamento = CURRENT_TIMESTAMP
        WHERE id = $2 AND usuario_id = $3 AND status = 'confirmada'
        RETURNING *
      `;

      const result = await pool.query(query, [motivo_cancelamento, reserva_id, req.user.id]);

      if (result.rows.length === 0) {
        return res.status(404).json({
          success: false,
          error: 'Reserva não encontrada ou já cancelada'
        });
      }

      res.json({
        success: true,
        message: 'Reserva cancelada com sucesso',
        reserva: result.rows[0]
      });
    } catch (error) {
      console.error('Erro ao cancelar reserva:', error);
      res.status(500).json({ success: false, error: 'Erro ao cancelar reserva' });
    }
  }

  gerarCodigoReserva() {
    const timestamp = Date.now().toString(36).toUpperCase();
    const random = crypto.randomBytes(3).toString('hex').toUpperCase();
    return `ALJ-${timestamp}-${random}`;
  }

  async enviarConfirmacao(usuarioId, reserva, alojamento) {
    // TODO: Implementar envio via email/WhatsApp
    console.log(`Enviando confirmação de reserva ${reserva.codigo_reserva} para usuário ${usuarioId}`);
    console.log(`Alojamento: ${alojamento?.titulo || 'N/A'}`);
  }
}

module.exports = new AlojamentosController();
