const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const { pool } = require('../../config/database');
const { redis } = require('../../config/database');
const { validateRegister, validateLogin } = require('../../validations/auth.validation');

class AuthController {
  async register(req, res) {
    try {
      // Validação
      const { error } = validateRegister(req.body);
      if (error) return res.status(400).json({ success: false, error: error.details[0].message });

      const { nome, email, telefone, senha, tipo_usuario } = req.body;

      // Verificar se usuário existe
      const userExists = await pool.query(
        'SELECT id FROM usuarios WHERE email = $1 OR telefone = $2',
        [email, telefone]
      );

      if (userExists.rows.length > 0) {
        return res.status(400).json({ 
          success: false, 
          error: 'Email ou telefone já cadastrado' 
        });
      }

      // Hash senha
      const hashedPassword = await bcrypt.hash(senha, 12);

      // Criar usuário
      const query = `
        INSERT INTO usuarios (nome, email, telefone, senha, tipo_usuario) 
        VALUES ($1, $2, $3, $4, $5) 
        RETURNING id, nome, email, telefone, tipo_usuario, created_at
      `;

      const result = await pool.query(query, [nome, email, telefone, hashedPassword, tipo_usuario]);

      // Gerar token
      const token = jwt.sign(
        { userId: result.rows[0].id, email: result.rows[0].email },
        process.env.JWT_SECRET,
        { expiresIn: process.env.JWT_EXPIRE }
      );

      // Enviar SMS de boas-vindas
      await this.enviarSMSBoasVindas(telefone, nome);

      res.status(201).json({
        success: true,
        message: 'Usuário criado com sucesso',
        user: result.rows[0],
        token
      });

    } catch (error) {
      console.error('Erro no registro:', error);
      res.status(500).json({ success: false, error: 'Erro ao criar usuário' });
    }
  }

  async login(req, res) {
    try {
      const { error } = validateLogin(req.body);
      if (error) return res.status(400).json({ success: false, error: error.details[0].message });

      const { email, senha } = req.body;

      // Buscar usuário
      const query = 'SELECT * FROM usuarios WHERE email = $1 AND status = $2';
      const result = await pool.query(query, [email, 'ativo']);

      if (result.rows.length === 0) {
        return res.status(401).json({ success: false, error: 'Credenciais inválidas' });
      }

      const user = result.rows[0];

      // Verificar senha
      const isValidPassword = await bcrypt.compare(senha, user.senha);
      if (!isValidPassword) {
        return res.status(401).json({ success: false, error: 'Credenciais inválidas' });
      }

      // Atualizar último acesso
      await pool.query(
        'UPDATE usuarios SET ultimo_acesso = CURRENT_TIMESTAMP WHERE id = $1',
        [user.id]
      );

      // Gerar token
      const token = jwt.sign(
        { userId: user.id, email: user.email, tipo: user.tipo_usuario },
        process.env.JWT_SECRET,
        { expiresIn: process.env.JWT_EXPIRE }
      );

      res.json({
        success: true,
        user: {
          id: user.id,
          nome: user.nome,
          email: user.email,
          telefone: user.telefone,
          tipo_usuario: user.tipo_usuario,
          foto_url: user.foto_url,
          email_verificado: user.email_verificado,
          telefone_verificado: user.telefone_verificado
        },
        token
      });

    } catch (error) {
      console.error('Erro no login:', error);
      res.status(500).json({ success: false, error: 'Erro ao fazer login' });
    }
  }

  async verificarTelefone(req, res) {
    try {
      const { telefone, codigo } = req.body;
      
      // Verificar código (implementar lógica com Redis)
      const codigoArmazenado = await redis.get(`verificacao:${telefone}`);
      
      if (codigoArmazenado !== codigo) {
        return res.status(400).json({ success: false, error: 'Código inválido' });
      }

      // Atualizar usuário
      await pool.query(
        'UPDATE usuarios SET telefone_verificado = true WHERE telefone = $1',
        [telefone]
      );

      // Remover código do Redis
      await redis.del(`verificacao:${telefone}`);

      res.json({ success: true, message: 'Telefone verificado com sucesso' });

    } catch (error) {
      res.status(500).json({ success: false, error: 'Erro na verificação' });
    }
  }

  async enviarSMSBoasVindas(telefone, nome) {
    // Implementar envio SMS via Africa's Talking ou outro serviço
    console.log(`SMS enviado para ${telefone}: Bem-vindo ao TudoAqui, ${nome}!`);
  }
}

module.exports = new AuthController();
