const express = require('express');
const router = express.Router();
const authController = require('./auth.controller');
const { validateVerificarTelefone, validateUpdateProfile } = require('../../validations/auth.validation');
const { autenticar } = require('../../middleware/auth.middleware');

/**
 * @route POST /api/v1/auth/register
 * @desc Registrar novo usuário
 * @body {nome, email, telefone, senha, tipo_usuario}
 * @returns {token, user}
 */
router.post('/register', authController.register.bind(authController));

/**
 * @route POST /api/v1/auth/login
 * @desc Fazer login
 * @body {email, senha}
 * @returns {token, user}
 */
router.post('/login', authController.login.bind(authController));

/**
 * @route POST /api/v1/auth/verificar-telefone
 * @desc Verificar código SMS de 6 dígitos
 * @body {telefone, codigo}
 * @returns {success, message}
 */
router.post('/verificar-telefone', authController.verificarTelefone.bind(authController));

/**
 * @route PUT /api/v1/auth/perfil
 * @desc Atualizar perfil do usuário (requer autenticação)
 * @auth Bearer token
 * @body {nome, foto_url, bio}
 * @returns {user}
 */
router.put('/perfil', autenticar, async (req, res) => {
  try {
    const { error } = validateUpdateProfile(req.body);
    if (error) return res.status(400).json({ success: false, error: error.details[0].message });

    const { nome, foto_url, bio } = req.body;
    const userId = req.user.userId;

    const query = `
      UPDATE usuarios 
      SET nome = COALESCE($1, nome),
          foto_url = COALESCE($2, foto_url),
          bio = COALESCE($3, bio),
          updated_at = CURRENT_TIMESTAMP
      WHERE id = $4
      RETURNING id, nome, email, telefone, tipo_usuario, foto_url, bio, updated_at
    `;

    const result = await require('../../config/database').pool.query(query, [nome, foto_url, bio, userId]);

    if (result.rows.length === 0) {
      return res.status(404).json({ success: false, error: 'Usuário não encontrado' });
    }

    res.json({
      success: true,
      message: 'Perfil atualizado com sucesso',
      user: result.rows[0]
    });

  } catch (error) {
    console.error('Erro ao atualizar perfil:', error);
    res.status(500).json({ success: false, error: 'Erro ao atualizar perfil' });
  }
});

/**
 * @route GET /api/v1/auth/me
 * @desc Obter dados do usuário autenticado
 * @auth Bearer token
 * @returns {user}
 */
router.get('/me', autenticar, async (req, res) => {
  try {
    const userId = req.user.userId;
    const query = `
      SELECT id, nome, email, telefone, tipo_usuario, foto_url, bio, 
             email_verificado, telefone_verificado, kyc_status, selo_verificado,
             provincia, municipio, created_at
      FROM usuarios 
      WHERE id = $1 AND status = 'ativo'
    `;

    const result = await require('../../config/database').pool.query(query, [userId]);

    if (result.rows.length === 0) {
      return res.status(404).json({ success: false, error: 'Usuário não encontrado' });
    }

    res.json({
      success: true,
      user: result.rows[0]
    });

  } catch (error) {
    console.error('Erro ao obter perfil:', error);
    res.status(500).json({ success: false, error: 'Erro ao obter perfil' });
  }
});

/**
 * @route POST /api/v1/auth/logout
 * @desc Logout do usuário (invalidar token no Redis)
 * @auth Bearer token
 * @returns {success, message}
 */
router.post('/logout', autenticar, async (req, res) => {
  try {
    const token = req.headers.authorization?.split(' ')[1];
    
    if (token) {
      // Adicionar token à lista de negra no Redis (TTL = tempo de expiração do JWT)
      const { redis } = require('../../config/database');
      await redis.setex(`blacklist:${token}`, 7 * 24 * 60 * 60, 'true');
    }

    res.json({
      success: true,
      message: 'Logout realizado com sucesso'
    });

  } catch (error) {
    console.error('Erro no logout:', error);
    res.status(500).json({ success: false, error: 'Erro ao fazer logout' });
  }
});

module.exports = router;
