const { pool } = require('../../config/database');
const { validateImovel } = require('../../validations/imovel.validation');

class ImoveisController {
  async listarImoveis(req, res) {
    try {
      const {
        tipo,
        finalidade,
        localizacao,
        preco_min,
        preco_max,
        quartos,
        banheiros,
        area_min,
        area_max,
        pagina = '1',
        limite = '20',
        ordenar = 'created_at',
        ordem = 'DESC'
      } = req.query;

      const page = Math.max(parseInt(pagina, 10) || 1, 1);
      const perPage = Math.min(Math.max(parseInt(limite, 10) || 20, 1), 100);
      const offset = (page - 1) * perPage;

      const where = ['i.status = $1'];
      const params = ['ativo'];
      const addFilter = (condition, value) => {
        params.push(value);
        where.push(condition.replace(/\?/g, params.length));
      };

      if (tipo) addFilter('i.tipo = $?', tipo);
      if (finalidade) addFilter('i.finalidade = $?', finalidade);
      if (localizacao) addFilter("(i.localizacao ILIKE $? OR i.endereco_completo->>'cidade' ILIKE $?)", `%${localizacao}%`);
      if (preco_min !== undefined) addFilter('i.preco >= $?', Number(preco_min));
      if (preco_max !== undefined) addFilter('i.preco <= $?', Number(preco_max));
      if (quartos !== undefined) addFilter('i.quartos >= $?', Number(quartos));
      if (banheiros !== undefined) addFilter('i.banheiros >= $?', Number(banheiros));
      if (area_min !== undefined) addFilter('i.area_total >= $?', Number(area_min));
      if (area_max !== undefined) addFilter('i.area_total <= $?', Number(area_max));

      const orderMap = {
        preco: 'i.preco',
        area_total: 'i.area_total',
        quartos: 'i.quartos',
        created_at: 'i.created_at',
        avaliacao_media: 'avaliacao_media'
      };
      const orderField = orderMap[ordenar] || 'i.created_at';
      const orderDirection = ordem && ordem.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';

      const baseParamsLength = params.length;

      const query = `
        SELECT 
          i.*, 
          u.nome AS proprietario_nome, 
          u.telefone AS proprietario_telefone,
          COALESCE(array_remove(array_agg(DISTINCT foto.url), NULL), '{}') AS fotos,
          COALESCE(AVG(avaliacao.nota)::numeric, 0) AS avaliacao_media,
          COUNT(avaliacao.id) AS total_avaliacoes
        FROM imoveis i
        JOIN usuarios u ON i.proprietario_id = u.id
        LEFT JOIN fotos_imoveis foto ON i.id = foto.imovel_id
        LEFT JOIN avaliacoes avaliacao ON i.id = avaliacao.item_id AND avaliacao.tipo_avaliacao = 'imovel'
        WHERE ${where.join(' AND ')}
        GROUP BY i.id, u.nome, u.telefone
        ORDER BY ${orderField} ${orderDirection}
        LIMIT $${baseParamsLength + 1} OFFSET $${baseParamsLength + 2}
      `;

      const queryParams = [...params, perPage, offset];
      const result = await pool.query(query, queryParams);

      const countQuery = `
        SELECT COUNT(*) AS total
        FROM imoveis i
        WHERE ${where.join(' AND ')}
      `;
      const countResult = await pool.query(countQuery, params);
      const total = Number(countResult.rows[0].total) || 0;

      res.json({
        success: true,
        imoveis: result.rows,
        paginacao: {
          pagina: page,
          limite: perPage,
          total,
          total_paginas: Math.ceil(total / perPage)
        }
      });
    } catch (error) {
      console.error('Erro ao listar imóveis:', error);
      res.status(500).json({ success: false, error: 'Erro ao listar imóveis' });
    }
  }

  async criarImovel(req, res) {
    try {
      const { error } = validateImovel(req.body);
      if (error) {
        return res.status(400).json({ success: false, error: error.details[0].message });
      }

      const {
        titulo,
        descricao,
        tipo,
        finalidade,
        preco,
        localizacao,
        quartos,
        banheiros,
        suites,
        area_total,
        vagas_garagem,
        latitude,
        longitude,
        endereco_completo,
        comodidades = [],
        fotos = []
      } = req.body;

      const query = `
        INSERT INTO imoveis (
          proprietario_id, titulo, descricao, tipo, finalidade, preco,
          localizacao, latitude, longitude, endereco_completo,
          quartos, banheiros, suites, area_total, vagas_garagem,
          comodidades, fotos
        ) VALUES (
          $1, $2, $3, $4, $5, $6,
          $7, $8, $9, $10,
          $11, $12, $13, $14, $15,
          $16, $17
        )
        RETURNING *
      `;

      const result = await pool.query(query, [
        req.user.id,
        titulo,
        descricao,
        tipo,
        finalidade,
        preco,
        localizacao,
        latitude,
        longitude,
        JSON.stringify(endereco_completo || {}),
        quartos,
        banheiros,
        suites,
        area_total,
        vagas_garagem,
        comodidades,
        fotos
      ]);

      await this.indexarImovel(result.rows[0]);

      res.status(201).json({
        success: true,
        message: 'Imóvel criado com sucesso',
        imovel: result.rows[0]
      });
    } catch (error) {
      console.error('Erro ao criar imóvel:', error);
      res.status(500).json({ success: false, error: 'Erro ao criar imóvel' });
    }
  }

  async buscarImovelPorId(req, res) {
    try {
      const { id } = req.params;

      const query = `
        SELECT 
          i.*, 
          u.nome AS proprietario_nome, 
          u.telefone AS proprietario_telefone, 
          u.email AS proprietario_email,
          COALESCE(array_remove(array_agg(DISTINCT foto.url), NULL), '{}') AS fotos_detalhadas,
          COALESCE(AVG(avaliacao.nota)::numeric, 0) AS avaliacao_media,
          COUNT(avaliacao.id) AS total_avaliacoes
        FROM imoveis i
        JOIN usuarios u ON i.proprietario_id = u.id
        LEFT JOIN fotos_imoveis foto ON i.id = foto.imovel_id
        LEFT JOIN avaliacoes avaliacao ON i.id = avaliacao.item_id AND avaliacao.tipo_avaliacao = 'imovel'
        WHERE i.id = $1 AND i.status = $2
        GROUP BY i.id, u.nome, u.telefone, u.email
      `;

      const result = await pool.query(query, [id, 'ativo']);

      if (result.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Imóvel não encontrado' });
      }

      await pool.query('UPDATE imoveis SET visualizacoes = visualizacoes + 1 WHERE id = $1', [id]);

      const imovel = result.rows[0];
      const similares = await this.buscarImoveisSimilares(imovel.tipo, imovel.preco, id);

      res.json({
        success: true,
        imovel: {
          ...imovel,
          fotos_detalhadas: imovel.fotos_detalhadas || [],
          similares
        }
      });
    } catch (error) {
      console.error('Erro ao buscar imóvel:', error);
      res.status(500).json({ success: false, error: 'Erro ao buscar imóvel' });
    }
  }

  async meusImoveis(req, res) {
    try {
      const query = `
        SELECT i.*, COALESCE(array_remove(array_agg(DISTINCT foto.url), NULL), '{}') AS fotos
        FROM imoveis i
        LEFT JOIN fotos_imoveis foto ON i.id = foto.imovel_id
        WHERE i.proprietario_id = $1
        GROUP BY i.id
        ORDER BY i.created_at DESC
      `;

      const result = await pool.query(query, [req.user.id]);

      res.json({
        success: true,
        imoveis: result.rows,
        total: result.rows.length
      });
    } catch (error) {
      console.error('Erro ao buscar meus imóveis:', error);
      res.status(500).json({ success: false, error: 'Erro ao buscar seus imóveis' });
    }
  }

  async indexarImovel(imovel) {
    console.log(`Imóvel ${imovel.id} indexado para busca`);
  }

  async buscarImoveisSimilares(tipo, preco, excludeId) {
    try {
      const query = `
        SELECT i.id, i.titulo, i.preco, i.fotos[1] AS foto_principal
        FROM imoveis i
        WHERE i.tipo = $1 AND i.status = $2 AND i.id != $3
        AND i.preco BETWEEN $4 * 0.8 AND $4 * 1.2
        ORDER BY i.created_at DESC
        LIMIT 4
      `;

      const result = await pool.query(query, [tipo, 'ativo', excludeId, preco]);
      return result.rows;
    } catch (error) {
      console.error('Erro ao buscar imóveis similares:', error);
      return [];
    }
  }
}

module.exports = new ImoveisController();
