const crypto = require('crypto');
const QRCode = require('qrcode');
const { pool } = require('../../config/database');

class IngressosController {
  async listarEventos(req, res) {
    try {
      const { categoria, data_inicio, data_fim, localizacao, pagina = '1', limite = '20' } = req.query;

      const page = Math.max(parseInt(pagina, 10) || 1, 1);
      const perPage = Math.min(Math.max(parseInt(limite, 10) || 20, 1), 100);
      const offset = (page - 1) * perPage;

      const where = ['e.status = $1'];
      const params = ['ativo'];

      if (categoria) {
        params.push(categoria);
        where.push(`e.categoria = $${params.length}`);
      }

      if (data_inicio) {
        params.push(data_inicio);
        where.push(`e.data_hora >= $${params.length}::timestamp`);
      }

      if (data_fim) {
        params.push(data_fim);
        where.push(`e.data_hora <= $${params.length}::timestamp`);
      }

      if (localizacao) {
        params.push(`%${localizacao}%`);
        where.push(`(e.localizacao ILIKE $${params.length} OR e.endereco_completo->>'cidade' ILIKE $${params.length})`);
      }

      const query = `
        SELECT e.*, u.nome AS organizador_nome,
               COALESCE(COUNT(iv.id), 0) AS ingressos_vendidos,
               (e.capacidade_total - COALESCE(COUNT(iv.id), 0)) AS ingressos_disponiveis
        FROM eventos_ingressos e
        JOIN usuarios u ON e.organizador_id = u.id
        LEFT JOIN ingressos_vendidos iv ON e.id = iv.evento_id AND iv.status = 'ativo'
        WHERE ${where.join(' AND ')}
        GROUP BY e.id, u.nome
        ORDER BY e.data_hora ASC
        LIMIT $${params.length + 1} OFFSET $${params.length + 2}
      `;

      const result = await pool.query(query, [...params, perPage, offset]);

      const countQuery = `
        SELECT COUNT(*) AS total
        FROM eventos_ingressos e
        WHERE ${where.join(' AND ')}
      `;
      const countResult = await pool.query(countQuery, params);
      const total = Number(countResult.rows[0].total) || 0;

      res.json({
        success: true,
        eventos: result.rows,
        paginacao: {
          pagina: page,
          limite: perPage,
          total,
          total_paginas: Math.ceil(total / perPage)
        }
      });
    } catch (error) {
      console.error('Erro ao listar eventos:', error);
      res.status(500).json({ success: false, error: 'Erro ao listar eventos' });
    }
  }

  async buscarEventoPorId(req, res) {
    try {
      const { id } = req.params;

      const query = `
        SELECT e.*, u.nome AS organizador_nome, u.email AS organizador_email,
               COALESCE(COUNT(iv.id), 0) AS ingressos_vendidos,
               (e.capacidade_total - COALESCE(COUNT(iv.id), 0)) AS ingressos_disponiveis
        FROM eventos_ingressos e
        JOIN usuarios u ON e.organizador_id = u.id
        LEFT JOIN ingressos_vendidos iv ON e.id = iv.evento_id AND iv.status = 'ativo'
        WHERE e.id = $1 AND e.status = $2
        GROUP BY e.id, u.nome, u.email
      `;

      const result = await pool.query(query, [id, 'ativo']);

      if (result.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Evento não encontrado' });
      }

      res.json({
        success: true,
        evento: result.rows[0]
      });
    } catch (error) {
      console.error('Erro ao buscar evento:', error);
      res.status(500).json({ success: false, error: 'Erro ao buscar evento' });
    }
  }

  async comprarIngresso(req, res) {
    const { evento_id, setor, quantidade, tipo = 'normal' } = req.body;

    if (!evento_id || !quantidade || quantidade < 1) {
      return res.status(400).json({
        success: false,
        error: 'Dados inválidos: evento_id e quantidade são obrigatórios'
      });
    }

    const client = await pool.connect();

    try {
      await client.query('BEGIN');

      // Verificar evento e disponibilidade com lock
      const eventoQuery = `
        SELECT e.*, 
               COALESCE(COUNT(iv.id), 0) AS vendidos,
               (e.capacidade_total - COALESCE(COUNT(iv.id), 0)) AS disponiveis
        FROM eventos_ingressos e
        LEFT JOIN ingressos_vendidos iv ON e.id = iv.evento_id AND iv.status = 'ativo'
        WHERE e.id = $1 AND e.status = 'ativo'
        GROUP BY e.id
        FOR UPDATE
      `;

      const eventoResult = await client.query(eventoQuery, [evento_id]);

      if (eventoResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ success: false, error: 'Evento não encontrado' });
      }

      const evento = eventoResult.rows[0];

      if (evento.disponiveis < quantidade) {
        await client.query('ROLLBACK');
        return res.status(400).json({
          success: false,
          error: `Apenas ${evento.disponiveis} ingressos disponíveis`
        });
      }

      // Calcular preço
      const preco_unitario = Number(evento.preco_ingresso) || 0;
      const taxa_servico = Number(evento.taxa_servico) || 0;
      const preco_total_unitario = preco_unitario + taxa_servico;
      const valor_total = preco_total_unitario * quantidade;

      // Criar ingressos com QR code único
      const ingressos = [];
      for (let i = 0; i < quantidade; i++) {
        const codigoQR = crypto.randomBytes(16).toString('hex');
        const codigoBarras = this.gerarCodigoBarras(evento_id, codigoQR);

        const ingresso = await client.query(
          `
          INSERT INTO ingressos_vendidos 
          (evento_id, comprador_id, codigo_qr, codigo_barras, setor, tipo, status, preco, data_compra)
          VALUES ($1, $2, $3, $4, $5, $6, 'ativo', $7, CURRENT_TIMESTAMP)
          RETURNING *
        `,
          [evento_id, req.user.id, codigoQR, codigoBarras, setor, tipo, preco_total_unitario]
        );

        // Gerar QR Code em base64
        const qrCodeData = await QRCode.toDataURL(codigoQR, { errorCorrectionLevel: 'H' });

        ingressos.push({
          ...ingresso.rows[0],
          qr_code_image: qrCodeData
        });
      }

      // Registrar transação
      await client.query(
        `
        INSERT INTO transacoes 
        (usuario_id, tipo_transacao, modulo, item_id, valor, metodo_pagamento, status_transacao)
        VALUES ($1, 'pagamento', 'ingressos', $2, $3, 'pendente', 'pendente')
      `,
        [req.user.id, evento_id, valor_total]
      );

      await client.query('COMMIT');

      // Enviar ingressos por email/WhatsApp (assíncrono)
      this.enviarIngressos(req.user.id, ingressos, evento).catch((err) =>
        console.error('Erro ao enviar ingressos:', err)
      );

      res.json({
        success: true,
        message: 'Ingressos comprados com sucesso',
        ingressos,
        valor_total
      });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error('Erro ao comprar ingresso:', error);
      res.status(500).json({ success: false, error: 'Erro ao processar compra' });
    } finally {
      client.release();
    }
  }

  async meusIngressos(req, res) {
    try {
      const query = `
        SELECT iv.*, e.titulo AS evento_titulo, e.data_hora AS evento_data, 
               e.localizacao AS evento_localizacao, e.endereco_completo AS evento_endereco
        FROM ingressos_vendidos iv
        JOIN eventos_ingressos e ON iv.evento_id = e.id
        WHERE iv.comprador_id = $1
        ORDER BY iv.data_compra DESC
      `;

      const result = await pool.query(query, [req.user.id]);

      // Gerar QR codes para ingressos ativos
      const ingressosComQR = await Promise.all(
        result.rows.map(async (ingresso) => {
          if (ingresso.status === 'ativo') {
            const qrCode = await QRCode.toDataURL(ingresso.codigo_qr, { errorCorrectionLevel: 'H' });
            return { ...ingresso, qr_code_image: qrCode };
          }
          return ingresso;
        })
      );

      res.json({
        success: true,
        ingressos: ingressosComQR,
        total: ingressosComQR.length
      });
    } catch (error) {
      console.error('Erro ao buscar meus ingressos:', error);
      res.status(500).json({ success: false, error: 'Erro ao buscar ingressos' });
    }
  }

  async validarIngresso(req, res) {
    try {
      const { codigo_qr } = req.body;

      if (!codigo_qr) {
        return res.status(400).json({ success: false, error: 'Código QR não fornecido' });
      }

      const query = `
        SELECT iv.*, e.titulo AS evento_titulo, e.data_hora AS evento_data,
               u.nome AS comprador_nome
        FROM ingressos_vendidos iv
        JOIN eventos_ingressos e ON iv.evento_id = e.id
        JOIN usuarios u ON iv.comprador_id = u.id
        WHERE iv.codigo_qr = $1
      `;

      const result = await pool.query(query, [codigo_qr]);

      if (result.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Ingresso não encontrado' });
      }

      const ingresso = result.rows[0];

      if (ingresso.status === 'usado') {
        return res.status(400).json({
          success: false,
          error: 'Ingresso já foi utilizado',
          data_uso: ingresso.data_uso
        });
      }

      if (ingresso.status === 'cancelado') {
        return res.status(400).json({ success: false, error: 'Ingresso cancelado' });
      }

      // Marcar como usado
      await pool.query(
        'UPDATE ingressos_vendidos SET status = $1, data_uso = CURRENT_TIMESTAMP WHERE id = $2',
        ['usado', ingresso.id]
      );

      res.json({
        success: true,
        message: 'Ingresso válido e marcado como usado',
        ingresso: {
          evento: ingresso.evento_titulo,
          comprador: ingresso.comprador_nome,
          setor: ingresso.setor,
          tipo: ingresso.tipo
        }
      });
    } catch (error) {
      console.error('Erro ao validar ingresso:', error);
      res.status(500).json({ success: false, error: 'Erro ao validar ingresso' });
    }
  }

  gerarCodigoBarras(eventoId, codigoQR) {
    // Gerar código de barras baseado no evento e QR code
    const hash = crypto.createHash('sha256').update(`${eventoId}-${codigoQR}`).digest('hex');
    return hash.substring(0, 13).toUpperCase();
  }

  async enviarIngressos(usuarioId, ingressos, evento) {
    // TODO: Implementar envio via email (SendGrid) e WhatsApp (Twilio/Africa's Talking)
    console.log(`Enviando ${ingressos.length} ingressos para usuário ${usuarioId}`);
    console.log(`Evento: ${evento?.titulo || 'N/A'}`);
  }
}

module.exports = new IngressosController();
