const { pool } = require('../../config/database');
const { sendSMS } = require('../../services/sms.service');
const { sendEmail } = require('../../services/email.service');
const { sendPushNotification } = require('../../services/push.service');

class NotificacoesController {
  async enviarNotificacao(tipo, dados) {
    try {
      const { usuario_id, titulo, mensagem, dados_adicionais } = dados;

      const query = `
        INSERT INTO notificacoes 
        (usuario_id, tipo_notificacao, titulo, mensagem, dados_adicionais)
        VALUES ($1, $2, $3, $4, $5)
        RETURNING *
      `;

      const result = await pool.query(query, [
        usuario_id,
        tipo,
        titulo,
        mensagem,
        JSON.stringify(dados_adicionais || {})
      ]);

      const notificacao = result.rows[0];

      switch (tipo) {
        case 'sms':
          await sendSMS(usuario_id, mensagem);
          break;
        case 'email':
          await sendEmail(usuario_id, titulo, mensagem, dados_adicionais);
          break;
        case 'push':
          await sendPushNotification(usuario_id, titulo, mensagem, dados_adicionais);
          break;
        case 'todos':
          await Promise.all([
            sendPushNotification(usuario_id, titulo, mensagem, dados_adicionais),
            sendEmail(usuario_id, titulo, mensagem, dados_adicionais)
          ]);
          break;
      }

      return notificacao;
    } catch (error) {
      console.error('Erro ao enviar notificação:', error);
      throw error;
    }
  }

  async listarNotificacoes(req, res) {
    try {
      const { pagina = '1', limite = '20', apenas_nao_lidas = 'false' } = req.query;

      const page = Math.max(parseInt(pagina, 10) || 1, 1);
      const perPage = Math.min(Math.max(parseInt(limite, 10) || 20, 1), 100);
      const offset = (page - 1) * perPage;

      const where = ['usuario_id = $1'];
      const params = [req.user.id];

      if (apenas_nao_lidas === 'true') {
        where.push('lida = false');
      }

      const query = `
        SELECT * FROM notificacoes 
        WHERE ${where.join(' AND ')}
        ORDER BY created_at DESC
        LIMIT $2 OFFSET $3
      `;

      const result = await pool.query(query, [req.user.id, perPage, offset]);

      const countQuery = `
        SELECT COUNT(*) AS total, COUNT(CASE WHEN lida = false THEN 1 END) AS nao_lidas
        FROM notificacoes
        WHERE usuario_id = $1
      `;
      const countResult = await pool.query(countQuery, [req.user.id]);

      res.json({
        success: true,
        notificacoes: result.rows,
        total: Number(countResult.rows[0].total) || 0,
        total_nao_lidas: Number(countResult.rows[0].nao_lidas) || 0,
        paginacao: {
          pagina: page,
          limite: perPage,
          total_paginas: Math.ceil((countResult.rows[0].total || 0) / perPage)
        }
      });
    } catch (error) {
      console.error('Erro ao listar notificações:', error);
      res.status(500).json({ success: false, error: 'Erro ao listar notificações' });
    }
  }

  async marcarComoLida(req, res) {
    try {
      const { notificacao_id } = req.params;

      const query = `
        UPDATE notificacoes 
        SET lida = true, data_leitura = CURRENT_TIMESTAMP 
        WHERE id = $1 AND usuario_id = $2
        RETURNING *
      `;

      const result = await pool.query(query, [notificacao_id, req.user.id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Notificação não encontrada' });
      }

      res.json({
        success: true,
        message: 'Notificação marcada como lida',
        notificacao: result.rows[0]
      });
    } catch (error) {
      console.error('Erro ao marcar notificação:', error);
      res.status(500).json({ success: false, error: 'Erro ao marcar notificação' });
    }
  }

  async marcarTodasComoLidas(req, res) {
    try {
      const query = `
        UPDATE notificacoes 
        SET lida = true, data_leitura = CURRENT_TIMESTAMP 
        WHERE usuario_id = $1 AND lida = false
        RETURNING id
      `;

      const result = await pool.query(query, [req.user.id]);

      res.json({
        success: true,
        message: `${result.rows.length} notificações marcadas como lidas`,
        total_atualizadas: result.rows.length
      });
    } catch (error) {
      console.error('Erro ao marcar todas como lidas:', error);
      res.status(500).json({ success: false, error: 'Erro ao marcar notificações' });
    }
  }

  async deletarNotificacao(req, res) {
    try {
      const { notificacao_id } = req.params;

      const query = 'DELETE FROM notificacoes WHERE id = $1 AND usuario_id = $2 RETURNING id';
      const result = await pool.query(query, [notificacao_id, req.user.id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Notificação não encontrada' });
      }

      res.json({
        success: true,
        message: 'Notificação deletada com sucesso'
      });
    } catch (error) {
      console.error('Erro ao deletar notificação:', error);
      res.status(500).json({ success: false, error: 'Erro ao deletar notificação' });
    }
  }

  async obterContadorNaoLidas(req, res) {
    try {
      const query = 'SELECT COUNT(*) AS total FROM notificacoes WHERE usuario_id = $1 AND lida = false';
      const result = await pool.query(query, [req.user.id]);

      res.json({
        success: true,
        total_nao_lidas: Number(result.rows[0].total) || 0
      });
    } catch (error) {
      console.error('Erro ao obter contador:', error);
      res.status(500).json({ success: false, error: 'Erro ao obter contador' });
    }
  }
}

module.exports = new NotificacoesController();
