const { pool } = require('../../config/database');
const axios = require('axios');
const { enviarNotificacao } = require('../notificacoes/notificacoes.controller');

/**
 * Controller de Pagamentos - TudoAqui Super App
 * 
 * Suporta múltiplos métodos de pagamento:
 * - Multicaixa Express (Angola): Pagamento via ATM/Mobile
 * - Cartão de Crédito: Via Stripe (internacional)
 * - Transferência Bancária: Gera dados para transferência manual
 * - Dinheiro: Pagamento na entrega/prestação do serviço
 * 
 * Integra com módulos: imóveis, transporte, turismo, eventos, ingressos, restaurantes, alojamentos
 */
class PagamentosController {
  /**
   * Processa pagamento para qualquer módulo do marketplace
   * POST /api/v1/pagamentos/processar
   */
  async processarPagamento(req, res) {
    try {
      const { item_id, modulo, valor, metodo_pagamento, dados_pagamento } = req.body;

      // Validar dados obrigatórios
      if (!item_id || !modulo || !valor || !metodo_pagamento) {
        return res.status(400).json({
          success: false,
          error: 'Dados de pagamento incompletos. Forneça: item_id, modulo, valor, metodo_pagamento'
        });
      }

      // Validar valor mínimo (100 AOA)
      if (valor < 100) {
        return res.status(400).json({
          success: false,
          error: 'Valor mínimo de pagamento é 100 AOA'
        });
      }

      // Validar método de pagamento
      const metodosValidos = ['multicaixa', 'cartao_credito', 'transferencia', 'dinheiro'];
      if (!metodosValidos.includes(metodo_pagamento)) {
        return res.status(400).json({
          success: false,
          error: `Método de pagamento inválido. Use: ${metodosValidos.join(', ')}`
        });
      }

      // Validar módulo
      const modulosValidos = ['imoveis', 'transporte', 'turismo', 'eventos', 'ingressos', 'restaurantes', 'alojamentos'];
      if (!modulosValidos.includes(modulo)) {
        return res.status(400).json({
          success: false,
          error: `Módulo inválido. Use: ${modulosValidos.join(', ')}`
        });
      }

      // Criar transação no banco
      const transacaoId = await this.criarTransacao({
        usuario_id: req.user.id,
        tipo_transacao: 'pagamento',
        modulo,
        item_id,
        valor,
        metodo_pagamento,
        dados_pagamento: dados_pagamento || {}
      });

      let resultadoPagamento;

      // Processar baseado no método de pagamento
      switch (metodo_pagamento) {
        case 'multicaixa':
          resultadoPagamento = await this.processarMulticaixa(transacaoId, valor, dados_pagamento);
          break;
        case 'cartao_credito':
          resultadoPagamento = await this.processarCartaoCredito(transacaoId, valor, dados_pagamento);
          break;
        case 'transferencia':
          resultadoPagamento = await this.processarTransferencia(transacaoId, valor, dados_pagamento);
          break;
        case 'dinheiro':
          resultadoPagamento = await this.processarDinheiro(transacaoId, valor);
          break;
        default:
          return res.status(400).json({ success: false, error: 'Método de pagamento não implementado' });
      }

      // Atualizar status da transação
      await this.atualizarStatusTransacao(transacaoId, resultadoPagamento.status);

      // Notificar usuário via push
      await enviarNotificacao('push', {
        usuario_id: req.user.id,
        titulo: 'Pagamento Iniciado',
        mensagem: resultadoPagamento.mensagem,
        dados_adicionais: {
          transacao_id: transacaoId,
          tipo: 'pagamento',
          ...resultadoPagamento.dados_adicionais
        }
      });

      res.json({
        success: true,
        transacao_id: transacaoId,
        status: resultadoPagamento.status,
        mensagem: resultadoPagamento.mensagem,
        dados_adicionais: resultadoPagamento.dados_adicionais
      });

    } catch (error) {
      console.error('Erro ao processar pagamento:', error);
      res.status(500).json({ success: false, error: 'Erro ao processar pagamento. Tente novamente.' });
    }
  }

  /**
   * Integração com Multicaixa Express (Angola)
   * Endpoint oficial: https://api.multicaixa.ao
   */
  async processarMulticaixa(transacaoId, valor, dados) {
    try {
      // Validar telefone angolano (9 dígitos)
      if (!dados || !dados.telefone) {
        throw new Error('Número de telefone é obrigatório para Multicaixa');
      }

      const telefone = dados.telefone.replace(/\D/g, '');
      if (telefone.length < 9) {
        throw new Error('Número de telefone inválido (mínimo 9 dígitos)');
      }

      // Integração com API Multicaixa Express
      const response = await axios.post('https://api.multicaixa.ao/v1/payment', {
        transaction_id: transacaoId,
        amount: valor,
        currency: 'AOA',
        phone_number: `+244${telefone}`,
        reference: `TUDOAQUI-${transacaoId}`,
        description: 'Pagamento TudoAqui Marketplace'
      }, {
        headers: {
          'Authorization': `Bearer ${process.env.MULTICAIXA_API_KEY}`,
          'Content-Type': 'application/json'
        },
        timeout: 10000 // 10 segundos timeout
      });

      if (response.data.status === 'success') {
        return {
          status: 'aguardando_confirmacao',
          mensagem: 'Pagamento via Multicaixa iniciado. Confirme no seu telemóvel.',
          dados_adicionais: {
            transaction_id: response.data.transaction_id,
            confirmation_code: response.data.confirmation_code,
            expires_at: response.data.expires_at
          }
        };
      } else {
        throw new Error(response.data.message || 'Erro desconhecido da Multicaixa');
      }

    } catch (error) {
      console.error('Erro Multicaixa:', error.message);
      return {
        status: 'erro',
        mensagem: `Erro ao processar Multicaixa: ${error.message}`,
        dados_adicionais: {}
      };
    }
  }

  /**
   * Integração com Stripe para pagamentos internacionais
   * Requer Stripe Secret Key no .env
   */
  async processarCartaoCredito(transacaoId, valor, dados) {
    try {
      // Verificar se Stripe está configurado
      if (!process.env.STRIPE_SECRET_KEY) {
        throw new Error('Stripe não configurado. Adicione STRIPE_SECRET_KEY ao .env');
      }

      const stripe = require('stripe')(process.env.STRIPE_SECRET_KEY);

      // Criar Payment Intent
      const paymentIntent = await stripe.paymentIntents.create({
        amount: Math.round(valor * 100), // Converter AOA para centavos
        currency: 'aoa',
        payment_method_types: ['card'],
        metadata: {
          transaction_id: transacaoId,
          platform: 'TudoAqui'
        },
        description: 'Pagamento TudoAqui Marketplace'
      });

      return {
        status: 'aguardando_confirmacao',
        mensagem: 'Pagamento com cartão criado. Complete a transação no frontend.',
        dados_adicionais: {
          client_secret: paymentIntent.client_secret,
          payment_intent_id: paymentIntent.id,
          amount: valor
        }
      };

    } catch (error) {
      console.error('Erro Stripe:', error.message);
      return {
        status: 'erro',
        mensagem: `Erro ao processar cartão: ${error.message}`,
        dados_adicionais: {}
      };
    }
  }

  /**
   * Gera dados para transferência bancária manual
   * Usuário deve transferir para conta do TudoAqui e enviar comprovativo
   */
  async processarTransferencia(transacaoId, valor, dados) {
    // Dados bancários do TudoAqui (exemplo - substituir por dados reais)
    return {
      status: 'aguardando_confirmacao',
      mensagem: 'Dados para transferência bancária. Envie o comprovativo após transferir.',
      dados_adicionais: {
        banco: 'Banco Angolano de Investimentos (BAI)',
        titular: 'TudoAqui Marketplace Lda',
        conta: '1234567890',
        nib: '0006.0000.12345678901.23',
        iban: 'AO06000012345678901234567',
        valor: valor,
        referencia: transacaoId,
        instrucoes: [
          '1. Faça a transferência bancária para a conta acima',
          '2. Use o código de referência no campo de descrição',
          '3. Envie o comprovativo via endpoint /api/v1/pagamentos/confirmar'
        ]
      }
    };
  }

  /**
   * Pagamento em dinheiro - na entrega ou no local
   * Aguarda confirmação do vendedor/prestador
   */
  async processarDinheiro(transacaoId, valor) {
    return {
      status: 'aguardando_confirmacao',
      mensagem: 'Pagamento em dinheiro registrado. Pague no ato da entrega/prestação do serviço.',
      dados_adicionais: {
        valor: valor,
        referencia: transacaoId,
        instrucoes: 'O pagamento deve ser realizado diretamente ao vendedor/prestador. Guarde este código de referência.'
      }
    };
  }

  /**
   * Cria registro da transação no banco
   */
  async criarTransacao(dados) {
    const query = `
      INSERT INTO transacoes 
      (usuario_id, tipo_transacao, modulo, item_id, valor, metodo_pagamento, dados_pagamento, status_transacao)
      VALUES ($1, $2, $3, $4, $5, $6, $7, 'pendente')
      RETURNING id
    `;

    const result = await pool.query(query, [
      dados.usuario_id,
      dados.tipo_transacao,
      dados.modulo,
      dados.item_id,
      dados.valor,
      dados.metodo_pagamento,
      JSON.stringify(dados.dados_pagamento)
    ]);

    return result.rows[0].id;
  }

  /**
   * Atualiza status da transação
   */
  async atualizarStatusTransacao(transacaoId, status) {
    const query = `
      UPDATE transacoes 
      SET status_transacao = $1, 
          data_confirmacao = CASE WHEN $1 = 'confirmada' THEN CURRENT_TIMESTAMP ELSE data_confirmacao END,
          updated_at = CURRENT_TIMESTAMP
      WHERE id = $2
      RETURNING *
    `;

    const result = await pool.query(query, [status, transacaoId]);
    return result.rows[0];
  }

  /**
   * Confirma pagamento com comprovativo (transferência/dinheiro)
   * POST /api/v1/pagamentos/confirmar
   */
  async confirmarPagamento(req, res) {
    try {
      const { transacao_id, comprovativo } = req.body;

      if (!transacao_id) {
        return res.status(400).json({ success: false, error: 'ID da transação é obrigatório' });
      }

      // Buscar transação e validar propriedade
      const transacao = await pool.query(`
        SELECT * FROM transacoes 
        WHERE id = $1 AND usuario_id = $2
      `, [transacao_id, req.user.id]);

      if (transacao.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Transação não encontrada ou não pertence a você' });
      }

      const trans = transacao.rows[0];

      // Validar se transação ainda está aguardando
      if (trans.status_transacao === 'confirmada') {
        return res.status(400).json({ success: false, error: 'Transação já confirmada' });
      }

      if (trans.status_transacao === 'cancelada') {
        return res.status(400).json({ success: false, error: 'Transação já cancelada' });
      }

      // Validar comprovativo (para transferência bancária)
      if (trans.metodo_pagamento === 'transferencia' && !comprovativo) {
        return res.status(400).json({ success: false, error: 'Comprovativo é obrigatório para transferência bancária' });
      }

      // Atualizar transação com comprovativo
      if (comprovativo) {
        await pool.query(`
          UPDATE transacoes 
          SET dados_pagamento = dados_pagamento || $1,
              updated_at = CURRENT_TIMESTAMP
          WHERE id = $2
        `, [JSON.stringify({ comprovativo, data_envio_comprovativo: new Date().toISOString() }), transacao_id]);
      }

      // Atualizar status da transação
      await this.atualizarStatusTransacao(transacao_id, 'confirmada');

      // Atualizar status do item relacionado (reserva, pedido, etc.)
      await this.atualizarStatusItem(trans.modulo, trans.item_id, 'confirmada');

      // Notificar usuário
      await enviarNotificacao('todos', {
        usuario_id: req.user.id,
        titulo: 'Pagamento Confirmado',
        mensagem: `Seu pagamento de ${trans.valor} AOA foi confirmado com sucesso!`,
        dados_adicionais: {
          transacao_id,
          tipo: 'confirmacao_pagamento',
          modulo: trans.modulo
        }
      });

      res.json({
        success: true,
        message: 'Pagamento confirmado com sucesso',
        transacao_id,
        status: 'confirmada'
      });

    } catch (error) {
      console.error('Erro ao confirmar pagamento:', error);
      res.status(500).json({ success: false, error: 'Erro ao confirmar pagamento' });
    }
  }

  /**
   * Atualiza status do item após confirmação de pagamento
   * Mapeia módulo para tabela correta
   */
  async atualizarStatusItem(modulo, itemId, status) {
    try {
      // Mapeamento seguro módulo → tabela
      const tabelasPermitidas = {
        'imoveis': 'imoveis',
        'transporte': 'viagens',
        'turismo': 'reservas_turisticas',
        'eventos': 'reservas_salas',
        'ingressos': 'ingressos_vendidos',
        'restaurantes': 'pedidos_restaurantes',
        'alojamentos': 'reservas_alojamentos'
      };

      const tabela = tabelasPermitidas[modulo];
      if (!tabela) {
        console.warn(`Módulo ${modulo} não possui mapeamento para atualização de status`);
        return;
      }

      // Usar query parametrizada (não é possível com nome de tabela, mas tabela vem de whitelist)
      await pool.query(`
        UPDATE ${tabela} 
        SET status = $1, updated_at = CURRENT_TIMESTAMP
        WHERE id = $2
      `, [status, itemId]);

      console.log(`Status do item ${itemId} (${modulo}) atualizado para ${status}`);

    } catch (error) {
      console.error(`Erro ao atualizar status do item ${itemId} em ${modulo}:`, error);
      // Não falhar toda a operação se apenas esta atualização falhar
    }
  }

  /**
   * Lista transações do usuário
   * GET /api/v1/pagamentos/historico
   */
  async listarTransacoes(req, res) {
    try {
      const { status, modulo, limite = 20, offset = 0 } = req.query;

      let query = `
        SELECT 
          t.id,
          t.tipo_transacao,
          t.modulo,
          t.item_id,
          t.valor,
          t.metodo_pagamento,
          t.status_transacao,
          t.created_at,
          t.data_confirmacao
        FROM transacoes t
        WHERE t.usuario_id = $1
      `;
      const params = [req.user.id];
      let paramCount = 1;

      // Filtro por status
      if (status) {
        paramCount++;
        query += ` AND t.status_transacao = $${paramCount}`;
        params.push(status);
      }

      // Filtro por módulo
      if (modulo) {
        paramCount++;
        query += ` AND t.modulo = $${paramCount}`;
        params.push(modulo);
      }

      query += ` ORDER BY t.created_at DESC LIMIT $${paramCount + 1} OFFSET $${paramCount + 2}`;
      params.push(parseInt(limite), parseInt(offset));

      const result = await pool.query(query, params);

      // Contar total
      const countQuery = `
        SELECT COUNT(*) as total 
        FROM transacoes 
        WHERE usuario_id = $1
        ${status ? `AND status_transacao = $2` : ''}
        ${modulo ? `AND modulo = $${status ? 3 : 2}` : ''}
      `;
      const countParams = [req.user.id];
      if (status) countParams.push(status);
      if (modulo) countParams.push(modulo);

      const countResult = await pool.query(countQuery, countParams);

      res.json({
        success: true,
        transacoes: result.rows,
        total: parseInt(countResult.rows[0].total),
        limite: parseInt(limite),
        offset: parseInt(offset)
      });

    } catch (error) {
      console.error('Erro ao listar transações:', error);
      res.status(500).json({ success: false, error: 'Erro ao buscar histórico de transações' });
    }
  }

  /**
   * Busca transação por ID
   * GET /api/v1/pagamentos/:id
   */
  async buscarTransacaoPorId(req, res) {
    try {
      const { id } = req.params;

      const query = `
        SELECT 
          t.*,
          u.nome as nome_usuario,
          u.email as email_usuario
        FROM transacoes t
        LEFT JOIN usuarios u ON t.usuario_id = u.id
        WHERE t.id = $1 AND t.usuario_id = $2
      `;

      const result = await pool.query(query, [id, req.user.id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Transação não encontrada' });
      }

      res.json({
        success: true,
        transacao: result.rows[0]
      });

    } catch (error) {
      console.error('Erro ao buscar transação:', error);
      res.status(500).json({ success: false, error: 'Erro ao buscar transação' });
    }
  }

  /**
   * Cancela transação (apenas se ainda não confirmada)
   * DELETE /api/v1/pagamentos/:id
   */
  async cancelarTransacao(req, res) {
    try {
      const { id } = req.params;

      // Buscar transação
      const transacao = await pool.query(`
        SELECT * FROM transacoes 
        WHERE id = $1 AND usuario_id = $2
      `, [id, req.user.id]);

      if (transacao.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Transação não encontrada' });
      }

      const trans = transacao.rows[0];

      // Validar se pode cancelar
      if (trans.status_transacao === 'confirmada') {
        return res.status(400).json({ success: false, error: 'Não é possível cancelar transação já confirmada' });
      }

      // Atualizar status
      await pool.query(`
        UPDATE transacoes 
        SET status_transacao = 'cancelada', updated_at = CURRENT_TIMESTAMP
        WHERE id = $1
      `, [id]);

      res.json({
        success: true,
        message: 'Transação cancelada com sucesso'
      });

    } catch (error) {
      console.error('Erro ao cancelar transação:', error);
      res.status(500).json({ success: false, error: 'Erro ao cancelar transação' });
    }
  }
}

module.exports = new PagamentosController();
