const { pool } = require('../../config/database');

class SalasController {
  async listarSalas(req, res) {
    try {
      const {
        localizacao,
        capacidade,
        preco_max,
        tipo_evento,
        data,
        pagina = '1',
        limite = '20',
        ordenar = 'created_at',
        ordem = 'DESC'
      } = req.query;

      const page = Math.max(parseInt(pagina, 10) || 1, 1);
      const perPage = Math.min(Math.max(parseInt(limite, 10) || 20, 1), 100);
      const offset = (page - 1) * perPage;

      // Parâmetro 1 reservado para verificar disponibilidade por data
      const selectParams = [data || null];
      const countParams = [];
      const whereSelect = ['s.ativa = true'];
      const whereCount = ['s.ativa = true'];

      const addFilter = (value, clauseBuilder) => {
        selectParams.push(value);
        countParams.push(value);
        const idxSelect = selectParams.length;
        const idxCount = countParams.length;
        whereSelect.push(clauseBuilder(idxSelect));
        whereCount.push(clauseBuilder(idxCount));
      };

      if (localizacao) {
        addFilter(`%${localizacao}%`, (idx) => `(s.localizacao ILIKE $${idx} OR s.endereco_completo->>'cidade' ILIKE $${idx})`);
      }

      if (capacidade !== undefined) {
        addFilter(Number(capacidade), (idx) => `s.capacidade >= $${idx}`);
      }

      if (preco_max !== undefined) {
        addFilter(Number(preco_max), (idx) => `COALESCE(s.preco_diaria, s.preco_hora) <= $${idx}`);
      }

      if (tipo_evento) {
        addFilter(tipo_evento, (idx) => `$${idx} = ANY(s.tipo_eventos_permitidos)`);
      }

      // Campo de disponibilidade baseado no parâmetro data (usando $1)
      const disponibilidadeExpr = `CASE WHEN $1::date IS NULL THEN true ELSE NOT EXISTS (SELECT 1 FROM reservas_salas rs WHERE rs.sala_id = s.id AND rs.data_evento = $1::date AND rs.status = 'confirmada') END AS disponivel`;

      const orderMap = {
        preco: 'COALESCE(s.preco_diaria, s.preco_hora)',
        capacidade: 's.capacidade',
        created_at: 's.created_at'
      };
      const orderField = orderMap[ordenar] || 's.created_at';
      const orderDirection = ordem && ordem.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';

      const selectQuery = `
        SELECT s.*, u.nome AS proprietario_nome, ${disponibilidadeExpr}
        FROM salas_eventos s
        JOIN usuarios u ON s.proprietario_id = u.id
        WHERE ${whereSelect.join(' AND ')}
        ORDER BY ${orderField} ${orderDirection}
        LIMIT $${selectParams.length + 1} OFFSET $${selectParams.length + 2}
      `;

      const selectQueryParams = [...selectParams, perPage, offset];
      const result = await pool.query(selectQuery, selectQueryParams);

      const countQuery = `
        SELECT COUNT(*) AS total
        FROM salas_eventos s
        WHERE ${whereCount.join(' AND ')}
      `;
      const countResult = await pool.query(countQuery, countParams);
      const total = Number(countResult.rows[0].total) || 0;

      res.json({
        success: true,
        salas: result.rows,
        paginacao: {
          pagina: page,
          limite: perPage,
          total,
          total_paginas: Math.ceil(total / perPage)
        }
      });
    } catch (error) {
      console.error('Erro ao listar salas:', error);
      res.status(500).json({ success: false, error: 'Erro ao listar salas' });
    }
  }
}

module.exports = new SalasController();
