const axios = require('axios');
const { pool } = require('../../config/database');

class TransporteController {
  constructor() {
    this.servicosTaxi = {
      tupuca: {
        nome: 'Tupuca',
        apiUrl: process.env.TUPUCA_API_URL,
        apiKey: process.env.TUPUCA_API_KEY,
        ativo: true
      },
      yango: {
        nome: 'Yango',
        apiUrl: process.env.YANGO_API_URL,
        apiKey: process.env.YANGO_API_KEY,
        ativo: true
      },
      bolt: {
        nome: 'Bolt',
        apiUrl: process.env.BOLT_API_URL,
        apiKey: process.env.BOLT_API_KEY,
        ativo: true
      },
      heetch: {
        nome: 'Heetch',
        apiUrl: process.env.HEETCH_API_URL,
        apiKey: process.env.HEETCH_API_KEY,
        ativo: true
      }
    };
  }

  async solicitarViagem(req, res) {
    try {
      const { origem, destino, tipo_servico, servico_preferido, forma_pagamento } = req.body;

      if (!origem?.latitude || !origem?.longitude || !destino?.latitude || !destino?.longitude) {
        return res.status(400).json({
          success: false,
          error: 'Coordenadas de origem e destino são obrigatórias'
        });
      }

      const { distancia, duracao } = await this.calcularRota(origem, destino);

      const opcoesServicos = await this.buscarOpcoesTodosServicos(
        origem,
        destino,
        tipo_servico,
        distancia,
        duracao
      );

      const opcoesFiltradas = servico_preferido
        ? opcoesServicos.filter((opcao) => opcao.servico === servico_preferido)
        : opcoesServicos;

      const viagemId = await this.criarViagemBanco({
        usuario_id: req.user.id,
        origem,
        destino,
        distancia,
        duracao,
        tipo_servico,
        servico_preferido,
        forma_pagamento,
        opcoes: opcoesFiltradas
      });

      res.json({
        success: true,
        viagem_id: viagemId,
        origem,
        destino,
        distancia_km: distancia,
        duracao_minutos: duracao,
        opcoes_servicos: opcoesFiltradas,
        melhor_opcao: this.selecionarMelhorOpcao(opcoesFiltradas)
      });
    } catch (error) {
      console.error('Erro ao solicitar viagem:', error);
      res.status(500).json({ success: false, error: 'Erro ao solicitar viagem' });
    }
  }

  async buscarOpcoesTodosServicos(origem, destino, tipoServico, distancia, duracao) {
    const opcoes = [];

    const promessas = Object.entries(this.servicosTaxi).map(async ([key, servico]) => {
      if (!servico.ativo || !servico.apiUrl || !servico.apiKey) return null;

      try {
        const response = await axios.post(
          `${servico.apiUrl}/estimate`,
          {
            origin: origem,
            destination: destino,
            service_type: tipoServico,
            distance_km: distancia,
            duration_minutes: duracao
          },
          {
            headers: {
              Authorization: `Bearer ${servico.apiKey}`,
              'Content-Type': 'application/json'
            },
            timeout: 5000
          }
        );

        return {
          servico: key,
          nome_servico: servico.nome,
          preco_estimado: response.data.price,
          tempo_estimado: response.data.duration,
          motoristas_proximos: response.data.drivers_nearby,
          tipo_veiculo: response.data.vehicle_type,
          aceita_cartao: response.data.accepts_card,
          aceita_dinheiro: response.data.accepts_cash,
          aceita_multicaixa: response.data.accepts_multicaixa
        };
      } catch (error) {
        console.error(`Erro ao buscar ${servico.nome}:`, error.message);
        return null;
      }
    });

    const resultados = await Promise.all(promessas);

    resultados.forEach((resultado) => {
      if (resultado && resultado.preco_estimado) {
        opcoes.push(resultado);
      }
    });

    return opcoes.sort((a, b) => a.preco_estimado - b.preco_estimado);
  }

  async confirmarViagem(req, res) {
    try {
      const { viagem_id, servico_escolhido } = req.body;

      const viagemQuery = 'SELECT * FROM viagens WHERE id = $1 AND usuario_id = $2';
      const viagemResult = await pool.query(viagemQuery, [viagem_id, req.user.id]);

      if (viagemResult.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Viagem não encontrada' });
      }

      const viagem = viagemResult.rows[0];

      const servico = this.servicosTaxi[servico_escolhido];
      if (!servico || !servico.ativo) {
        return res.status(400).json({ success: false, error: 'Serviço indisponível' });
      }

      const confirmacao = await this.confirmarComServico(servico, viagem);

      await pool.query(
        `
        UPDATE viagens 
        SET status = $1, 
            servico_empresa = $2,
            servico_id_externo = $3,
            preco_final = $4,
            data_confirmacao = CURRENT_TIMESTAMP
        WHERE id = $5
      `,
        [
          'confirmada',
          servico_escolhido,
          confirmacao.ride_id,
          confirmacao.preco_final,
          viagem_id
        ]
      );

      await this.notificarMotorista(confirmacao.motorista_id, viagem_id);

      res.json({
        success: true,
        message: 'Viagem confirmada',
        motorista: {
          nome: confirmacao.motorista_nome,
          telefone: confirmacao.motorista_telefone,
          foto: confirmacao.motorista_foto,
          veiculo: confirmacao.veiculo,
          placa: confirmacao.placa,
          tempo_chegada: confirmacao.tempo_chegada
        },
        preco_final: confirmacao.preco_final
      });
    } catch (error) {
      console.error('Erro ao confirmar viagem:', error);
      res.status(500).json({ success: false, error: 'Erro ao confirmar viagem' });
    }
  }

  async calcularRota(origem, destino) {
    try {
      const response = await axios.get('https://maps.googleapis.com/maps/api/directions/json', {
        params: {
          origin: `${origem.latitude},${origem.longitude}`,
          destination: `${destino.latitude},${destino.longitude}`,
          key: process.env.GOOGLE_MAPS_API_KEY,
          mode: 'driving',
          language: 'pt-BR'
        }
      });

      if (!response.data.routes?.length) {
        throw new Error('Rota não encontrada');
      }

      const rota = response.data.routes[0];
      const distancia = rota.legs[0].distance.value / 1000;
      const duracao = Math.ceil(rota.legs[0].duration.value / 60);

      return { distancia, duracao };
    } catch (error) {
      console.error('Erro ao calcular rota:', error.message);
      return { distancia: 5, duracao: 15 };
    }
  }

  async criarViagemBanco(dados) {
    const query = `
      INSERT INTO viagens (
        usuario_id, origem, destino, distancia_km, duracao_minutos,
        tipo_servico, servico_empresa, forma_pagamento, preco_estimado, status
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
      RETURNING id
    `;

    const result = await pool.query(query, [
      dados.usuario_id,
      JSON.stringify(dados.origem),
      JSON.stringify(dados.destino),
      dados.distancia,
      dados.duracao,
      dados.tipo_servico,
      dados.servico_preferido,
      dados.forma_pagamento,
      dados.opcoes.length > 0 ? dados.opcoes[0].preco_estimado : 0,
      'solicitada'
    ]);

    return result.rows[0].id;
  }

  selecionarMelhorOpcao(opcoes) {
    if (!opcoes || opcoes.length === 0) return null;

    return opcoes.reduce((melhor, atual) => {
      const pontuacaoAtual = this.calcularPontuacaoOpcao(atual);
      const pontuacaoMelhor = this.calcularPontuacaoOpcao(melhor);
      return pontuacaoAtual > pontuacaoMelhor ? atual : melhor;
    });
  }

  calcularPontuacaoOpcao(opcao) {
    const preco = Number(opcao.preco_estimado) || 0;
    const tempo = Number(opcao.tempo_estimado) || 0;
    const motoristas = Number(opcao.motoristas_proximos) || 0;

    const pontuacaoPreco = Math.max(0, 100 - preco / 10);
    const pontuacaoTempo = Math.max(0, 100 - tempo / 2);
    const pontuacaoMotoristas = Math.min(100, motoristas * 10);

    return pontuacaoPreco * 0.4 + pontuacaoTempo * 0.3 + pontuacaoMotoristas * 0.3;
  }

  async confirmarComServico(servico, viagem) {
    try {
      const response = await axios.post(
        `${servico.apiUrl}/request`,
        {
          origin: viagem.origem,
          destination: viagem.destino,
          service_type: viagem.tipo_servico,
          payment_method: viagem.forma_pagamento,
          customer_id: viagem.usuario_id
        },
        {
          headers: {
            Authorization: `Bearer ${servico.apiKey}`,
            'Content-Type': 'application/json'
          }
        }
      );

      return response.data;
    } catch (error) {
      throw new Error(`Erro ao confirmar com ${servico.nome}: ${error.message}`);
    }
  }

  async notificarMotorista(motoristaId, viagemId) {
    console.log(`Notificando motorista ${motoristaId} sobre viagem ${viagemId}`);
  }

  async acompanharViagem(req, res) {
    try {
      const { viagem_id } = req.params;

      const query = `
        SELECT v.*, u.nome AS usuario_nome
        FROM viagens v
        JOIN usuarios u ON v.usuario_id = u.id
        WHERE v.id = $1 AND (v.usuario_id = $2 OR v.motorista_id = $2)
      `;

      const result = await pool.query(query, [viagem_id, req.user.id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Viagem não encontrada' });
      }

      const viagem = result.rows[0];

      if (viagem.servico_empresa && viagem.status === 'em_curso') {
        const atualizacao = await this.buscarAtualizacaoServico(viagem);
        viagem.localizacao_motorista = atualizacao.localizacao_motorista;
        viagem.tempo_chegada = atualizacao.tempo_chegada;
      }

      res.json({
        success: true,
        viagem
      });
    } catch (error) {
      console.error('Erro ao acompanhar viagem:', error);
      res.status(500).json({ success: false, error: 'Erro ao acompanhar viagem' });
    }
  }

  async buscarAtualizacaoServico(viagem) {
    const servico = this.servicosTaxi[viagem.servico_empresa];
    if (!servico || !servico.apiUrl || !servico.apiKey) return {};

    try {
      const response = await axios.get(`${servico.apiUrl}/track/${viagem.servico_id_externo}`, {
        headers: {
          Authorization: `Bearer ${servico.apiKey}`
        }
      });

      return response.data;
    } catch (error) {
      console.error('Erro ao buscar atualização:', error.message);
      return {};
    }
  }
}

module.exports = new TransporteController();
