const AWS = require('aws-sdk');
const multer = require('multer');
const multerS3 = require('multer-s3');
const path = require('path');
const { v4: uuidv4 } = require('uuid');
const { pool } = require('../../config/database');

// Configuração S3 (usa credenciais e região do .env)
const s3 = new AWS.S3({
  accessKeyId: process.env.AWS_ACCESS_KEY_ID,
  secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
  region: process.env.AWS_REGION
});

const allowedTypes = /jpeg|jpg|png|gif|pdf|mp4|mov/;

const storage = multerS3({
  s3,
  bucket: process.env.AWS_BUCKET_NAME,
  acl: 'public-read',
  contentType: multerS3.AUTO_CONTENT_TYPE,
  key: (req, file, cb) => {
    const folder = (req.body.tipo || 'geral').toLowerCase();
    const filename = `${folder}/${uuidv4()}${path.extname(file.originalname)}`;
    cb(null, filename);
  }
});

const upload = multer({
  storage,
  limits: { fileSize: 10 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype.toLowerCase());
    if (mimetype && extname) return cb(null, true);
    return cb(new Error('Tipo de arquivo não permitido'));
  }
});

class UploadController {
  uploadFoto(req, res) {
    upload.single('foto')(req, res, async (error) => {
      if (error) {
        return res.status(400).json({ success: false, error: error.message });
      }

      if (!req.file) {
        return res.status(400).json({ success: false, error: 'Nenhuma foto enviada' });
      }

      try {
        const query = `
          INSERT INTO arquivos_upload 
          (id, usuario_id, tipo, nome_original, nome_arquivo, url, tamanho, mime_type)
          VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
          RETURNING *
        `;

        const result = await pool.query(query, [
          uuidv4(),
          req.user.id,
          req.body.tipo,
          req.file.originalname,
          req.file.key,
          req.file.location,
          req.file.size,
          req.file.mimetype
        ]);

        res.json({ success: true, arquivo: result.rows[0] });
      } catch (err) {
        console.error('Erro ao salvar upload:', err);
        res.status(500).json({ success: false, error: 'Erro ao salvar arquivo' });
      }
    });
  }

  uploadMultiplasFotos(req, res) {
    upload.array('fotos', 10)(req, res, async (error) => {
      if (error) {
        return res.status(400).json({ success: false, error: error.message });
      }

      if (!req.files || req.files.length === 0) {
        return res.status(400).json({ success: false, error: 'Nenhuma foto enviada' });
      }

      try {
        const arquivos = [];

        for (const file of req.files) {
          const query = `
            INSERT INTO arquivos_upload 
            (id, usuario_id, tipo, nome_original, nome_arquivo, url, tamanho, mime_type)
            VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
            RETURNING *
          `;

          const result = await pool.query(query, [
            uuidv4(),
            req.user.id,
            req.body.tipo,
            file.originalname,
            file.key,
            file.location,
            file.size,
            file.mimetype
          ]);

          arquivos.push(result.rows[0]);
        }

        res.json({ success: true, arquivos, total: arquivos.length });
      } catch (err) {
        console.error('Erro ao salvar uploads:', err);
        res.status(500).json({ success: false, error: 'Erro ao salvar arquivos' });
      }
    });
  }

  async deletarArquivo(req, res) {
    try {
      const { arquivo_id } = req.params;

      const query = 'SELECT * FROM arquivos_upload WHERE id = $1 AND usuario_id = $2';
      const result = await pool.query(query, [arquivo_id, req.user.id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ success: false, error: 'Arquivo não encontrado' });
      }

      const arquivo = result.rows[0];

      await s3
        .deleteObject({ Bucket: process.env.AWS_BUCKET_NAME, Key: arquivo.nome_arquivo })
        .promise();

      await pool.query('DELETE FROM arquivos_upload WHERE id = $1', [arquivo_id]);

      res.json({ success: true, message: 'Arquivo deletado com sucesso' });
    } catch (error) {
      console.error('Erro ao deletar arquivo:', error);
      res.status(500).json({ success: false, error: 'Erro ao deletar arquivo' });
    }
  }
}

module.exports = new UploadController();
