const nodemailer = require('nodemailer');
const { pool } = require('../config/database');

/**
 * Serviço de envio de emails via SendGrid/SMTP
 * Configurado para notificações do TudoAqui
 */
class EmailService {
  constructor() {
    this.transporter = nodemailer.createTransport({
      host: process.env.SMTP_HOST || 'smtp.sendgrid.net',
      port: process.env.SMTP_PORT || 587,
      secure: false,
      auth: {
        user: process.env.SMTP_USER || 'apikey',
        pass: process.env.SMTP_PASSWORD || process.env.SENDGRID_API_KEY
      }
    });

    this.fromEmail = process.env.EMAIL_FROM || 'noreply@tudoaqui.ao';
    this.fromName = process.env.EMAIL_FROM_NAME || 'TudoAqui';
  }

  async sendEmail(usuarioId, titulo, mensagem, dadosAdicionais = {}) {
    try {
      // Buscar email do usuário
      const query = 'SELECT email, nome FROM usuarios WHERE id = $1';
      const result = await pool.query(query, [usuarioId]);

      if (result.rows.length === 0) {
        throw new Error('Usuário não encontrado');
      }

      const { email, nome } = result.rows[0];

      if (!email) {
        throw new Error('Email não cadastrado');
      }

      // Montar HTML do email
      const htmlContent = this.gerarTemplateEmail(nome, titulo, mensagem, dadosAdicionais);

      // Enviar email
      const info = await this.transporter.sendMail({
        from: `"${this.fromName}" <${this.fromEmail}>`,
        to: email,
        subject: titulo,
        text: mensagem,
        html: htmlContent
      });

      console.log(`Email enviado para ${nome} (${email}):`, info.messageId);

      return {
        success: true,
        messageId: info.messageId
      };
    } catch (error) {
      console.error('Erro ao enviar email:', error.message);
      throw new Error(`Falha ao enviar email: ${error.message}`);
    }
  }

  gerarTemplateEmail(nome, titulo, mensagem, dadosAdicionais) {
    return `
      <!DOCTYPE html>
      <html lang="pt">
      <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>${titulo}</title>
        <style>
          body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; background-color: #f4f4f4; padding: 20px; }
          .container { max-width: 600px; margin: 0 auto; background: #fff; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
          .header { text-align: center; padding-bottom: 20px; border-bottom: 2px solid #0052CC; }
          .header h1 { color: #0052CC; margin: 0; }
          .content { padding: 20px 0; }
          .footer { text-align: center; padding-top: 20px; border-top: 1px solid #ddd; color: #777; font-size: 12px; }
          .button { display: inline-block; padding: 12px 24px; background-color: #0052CC; color: #fff; text-decoration: none; border-radius: 5px; margin-top: 20px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <h1>TudoAqui</h1>
          </div>
          <div class="content">
            <p>Olá, <strong>${nome}</strong>!</p>
            <h2>${titulo}</h2>
            <p>${mensagem}</p>
            ${dadosAdicionais.link ? `<a href="${dadosAdicionais.link}" class="button">Ver Detalhes</a>` : ''}
          </div>
          <div class="footer">
            <p>&copy; ${new Date().getFullYear()} TudoAqui - Encontre tudo num só lugar</p>
            <p>Luanda, Angola</p>
          </div>
        </div>
      </body>
      </html>
    `;
  }

  async enviarEmailConfirmacao(usuarioId, codigoVerificacao) {
    const titulo = 'Confirme seu email - TudoAqui';
    const mensagem = `Use o código abaixo para confirmar seu email:\n\n${codigoVerificacao}\n\nO código é válido por 24 horas.`;
    const dadosAdicionais = {
      link: `${process.env.APP_URL}/confirmar-email?codigo=${codigoVerificacao}`
    };

    return this.sendEmail(usuarioId, titulo, mensagem, dadosAdicionais);
  }

  async enviarEmailReserva(usuarioId, tipoReserva, detalhes) {
    const titulo = `Confirmação de ${tipoReserva} - TudoAqui`;
    const mensagem = `Sua reserva foi confirmada com sucesso!\n\nCódigo: ${detalhes.codigo}\nData: ${detalhes.data}\n\nObrigado por usar TudoAqui!`;
    const dadosAdicionais = {
      link: `${process.env.APP_URL}/minhas-reservas/${detalhes.id}`
    };

    return this.sendEmail(usuarioId, titulo, mensagem, dadosAdicionais);
  }
}

module.exports = new EmailService();
