const admin = require('firebase-admin');
const { pool } = require('../config/database');

/**
 * Serviço de notificações push via Firebase Cloud Messaging (FCM)
 * Requer Firebase Admin SDK configurado
 */
class PushNotificationService {
  constructor() {
    // Inicializar Firebase Admin (se ainda não inicializado)
    if (!admin.apps.length) {
      try {
        admin.initializeApp({
          credential: admin.credential.cert({
            projectId: process.env.FIREBASE_PROJECT_ID,
            clientEmail: process.env.FIREBASE_CLIENT_EMAIL,
            privateKey: process.env.FIREBASE_PRIVATE_KEY?.replace(/\\n/g, '\n')
          })
        });
        console.log('Firebase Admin inicializado com sucesso');
      } catch (error) {
        console.error('Erro ao inicializar Firebase Admin:', error.message);
      }
    }

    this.messaging = admin.messaging();
  }

  async sendPushNotification(usuarioId, titulo, mensagem, dadosAdicionais = {}) {
    try {
      // Buscar token FCM do usuário
      const query = 'SELECT fcm_token, nome FROM usuarios WHERE id = $1';
      const result = await pool.query(query, [usuarioId]);

      if (result.rows.length === 0) {
        throw new Error('Usuário não encontrado');
      }

      const { fcm_token, nome } = result.rows[0];

      if (!fcm_token) {
        console.warn(`Usuário ${nome} não possui token FCM registrado`);
        return { success: false, reason: 'Token não registrado' };
      }

      // Preparar payload da notificação
      const payload = {
        notification: {
          title: titulo,
          body: mensagem,
          icon: dadosAdicionais.icon || '/logo.png',
          badge: dadosAdicionais.badge || '/badge.png',
          sound: 'default'
        },
        data: {
          tipo: dadosAdicionais.tipo || 'geral',
          item_id: dadosAdicionais.item_id || '',
          link: dadosAdicionais.link || '',
          timestamp: new Date().toISOString()
        },
        android: {
          priority: 'high',
          notification: {
            channelId: 'default',
            color: '#0052CC'
          }
        },
        apns: {
          payload: {
            aps: {
              sound: 'default',
              badge: dadosAdicionais.badge_count || 1
            }
          }
        },
        token: fcm_token
      };

      // Enviar notificação
      const response = await this.messaging.send(payload);

      console.log(`Push notification enviado para ${nome}:`, response);

      return {
        success: true,
        messageId: response
      };
    } catch (error) {
      console.error('Erro ao enviar push notification:', error.message);

      // Se o token estiver inválido, remover do banco
      if (error.code === 'messaging/invalid-registration-token' || error.code === 'messaging/registration-token-not-registered') {
        await pool.query('UPDATE usuarios SET fcm_token = NULL WHERE id = $1', [usuarioId]);
        console.log(`Token FCM inválido removido para usuário ${usuarioId}`);
      }

      throw new Error(`Falha ao enviar push notification: ${error.message}`);
    }
  }

  async sendMulticastNotification(usuarioIds, titulo, mensagem, dadosAdicionais = {}) {
    try {
      // Buscar tokens FCM de múltiplos usuários
      const query = 'SELECT fcm_token FROM usuarios WHERE id = ANY($1) AND fcm_token IS NOT NULL';
      const result = await pool.query(query, [usuarioIds]);

      if (result.rows.length === 0) {
        console.warn('Nenhum usuário com token FCM encontrado');
        return { success: false, reason: 'Nenhum token disponível' };
      }

      const tokens = result.rows.map((row) => row.fcm_token);

      // Preparar payload multicast
      const message = {
        notification: {
          title: titulo,
          body: mensagem
        },
        data: {
          tipo: dadosAdicionais.tipo || 'geral',
          item_id: dadosAdicionais.item_id || '',
          timestamp: new Date().toISOString()
        },
        tokens
      };

      // Enviar para múltiplos dispositivos
      const response = await this.messaging.sendMulticast(message);

      console.log(`${response.successCount} notificações enviadas com sucesso`);
      console.log(`${response.failureCount} notificações falharam`);

      return {
        success: true,
        successCount: response.successCount,
        failureCount: response.failureCount
      };
    } catch (error) {
      console.error('Erro ao enviar multicast notification:', error.message);
      throw new Error(`Falha ao enviar multicast notification: ${error.message}`);
    }
  }

  async registrarTokenFCM(usuarioId, fcmToken) {
    try {
      await pool.query('UPDATE usuarios SET fcm_token = $1 WHERE id = $2', [fcmToken, usuarioId]);
      console.log(`Token FCM registrado para usuário ${usuarioId}`);
      return { success: true };
    } catch (error) {
      console.error('Erro ao registrar token FCM:', error.message);
      throw error;
    }
  }

  async removerTokenFCM(usuarioId) {
    try {
      await pool.query('UPDATE usuarios SET fcm_token = NULL WHERE id = $1', [usuarioId]);
      console.log(`Token FCM removido para usuário ${usuarioId}`);
      return { success: true };
    } catch (error) {
      console.error('Erro ao remover token FCM:', error.message);
      throw error;
    }
  }
}

module.exports = new PushNotificationService();
