const axios = require('axios');
const { pool } = require('../config/database');

/**
 * Serviço de envio de SMS via Africa's Talking
 * Configurado para o mercado angolano
 */
class SMSService {
  constructor() {
    this.apiKey = process.env.AFRICAS_TALKING_API_KEY;
    this.username = process.env.AFRICAS_TALKING_USERNAME;
    this.apiUrl = 'https://api.africastalking.com/version1/messaging';
  }

  async sendSMS(usuarioId, mensagem) {
    try {
      // Buscar telefone do usuário
      const query = 'SELECT telefone, nome FROM usuarios WHERE id = $1';
      const result = await pool.query(query, [usuarioId]);

      if (result.rows.length === 0) {
        throw new Error('Usuário não encontrado');
      }

      const { telefone, nome } = result.rows[0];

      if (!telefone) {
        throw new Error('Telefone não cadastrado');
      }

      // Formatar telefone para formato internacional (+244...)
      const telefoneFormatado = this.formatarTelefone(telefone);

      // Enviar SMS via Africa's Talking
      const response = await axios.post(
        this.apiUrl,
        new URLSearchParams({
          username: this.username,
          to: telefoneFormatado,
          message: mensagem,
          from: 'TudoAqui'
        }),
        {
          headers: {
            apiKey: this.apiKey,
            'Content-Type': 'application/x-www-form-urlencoded',
            Accept: 'application/json'
          }
        }
      );

      console.log(`SMS enviado para ${nome} (${telefoneFormatado}):`, response.data);

      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      console.error('Erro ao enviar SMS:', error.message);
      // Não falhar silenciosamente - lançar erro para retry
      throw new Error(`Falha ao enviar SMS: ${error.message}`);
    }
  }

  formatarTelefone(telefone) {
    // Remove caracteres não numéricos
    let numeroLimpo = telefone.replace(/\D/g, '');

    // Se já tem código do país (+244), retorna
    if (numeroLimpo.startsWith('244')) {
      return `+${numeroLimpo}`;
    }

    // Adiciona código de Angola (+244)
    return `+244${numeroLimpo}`;
  }

  async enviarCodigoVerificacao(telefone, codigo) {
    const mensagem = `Seu código de verificação TudoAqui é: ${codigo}. Válido por 10 minutos.`;

    try {
      const telefoneFormatado = this.formatarTelefone(telefone);

      const response = await axios.post(
        this.apiUrl,
        new URLSearchParams({
          username: this.username,
          to: telefoneFormatado,
          message: mensagem,
          from: 'TudoAqui'
        }),
        {
          headers: {
            apiKey: this.apiKey,
            'Content-Type': 'application/x-www-form-urlencoded',
            Accept: 'application/json'
          }
        }
      );

      console.log(`Código de verificação enviado para ${telefoneFormatado}`);

      return { success: true, data: response.data };
    } catch (error) {
      console.error('Erro ao enviar código de verificação:', error.message);
      throw error;
    }
  }
}

module.exports = new SMSService();
