const Joi = require('joi');

// Validação de Registro - email, telefone, senha forte, nome
const registerSchema = Joi.object({
  nome: Joi.string()
    .min(3)
    .max(100)
    .required()
    .messages({
      'string.min': 'Nome deve ter pelo menos 3 caracteres',
      'string.max': 'Nome não pode exceder 100 caracteres',
      'any.required': 'Nome é obrigatório'
    }),
  email: Joi.string()
    .email()
    .required()
    .messages({
      'string.email': 'Email inválido',
      'any.required': 'Email é obrigatório'
    }),
  telefone: Joi.string()
    .pattern(/^[0-9]{9,15}$/)
    .required()
    .messages({
      'string.pattern.base': 'Telefone deve ter 9-15 dígitos',
      'any.required': 'Telefone é obrigatório'
    }),
  senha: Joi.string()
    .min(8)
    .pattern(/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/)
    .required()
    .messages({
      'string.min': 'Senha deve ter pelo menos 8 caracteres',
      'string.pattern.base': 'Senha deve conter maiúsculas, minúsculas e números',
      'any.required': 'Senha é obrigatória'
    }),
  tipo_usuario: Joi.string()
    .valid('comprador', 'vendedor', 'prestador_servico')
    .default('comprador')
    .messages({
      'any.only': 'Tipo de usuário inválido'
    })
});

// Validação de Login - email e senha
const loginSchema = Joi.object({
  email: Joi.string()
    .email()
    .required()
    .messages({
      'string.email': 'Email inválido',
      'any.required': 'Email é obrigatório'
    }),
  senha: Joi.string()
    .required()
    .messages({
      'any.required': 'Senha é obrigatória'
    })
});

// Validação de Verificação de Telefone
const verificarTelefoneSchema = Joi.object({
  telefone: Joi.string()
    .pattern(/^[0-9]{9,15}$/)
    .required()
    .messages({
      'string.pattern.base': 'Telefone inválido',
      'any.required': 'Telefone é obrigatório'
    }),
  codigo: Joi.string()
    .length(6)
    .pattern(/^[0-9]{6}$/)
    .required()
    .messages({
      'string.length': 'Código deve ter 6 dígitos',
      'string.pattern.base': 'Código deve conter apenas números',
      'any.required': 'Código é obrigatório'
    })
});

// Validação de Atualização de Perfil
const updateProfileSchema = Joi.object({
  nome: Joi.string()
    .min(3)
    .max(100)
    .optional()
    .messages({
      'string.min': 'Nome deve ter pelo menos 3 caracteres',
      'string.max': 'Nome não pode exceder 100 caracteres'
    }),
  foto_url: Joi.string()
    .uri()
    .optional()
    .messages({
      'string.uri': 'URL da foto inválida'
    }),
  bio: Joi.string()
    .max(500)
    .optional()
    .messages({
      'string.max': 'Bio não pode exceder 500 caracteres'
    })
});

// Funções de validação
const validateRegister = (data) => registerSchema.validate(data, { abortEarly: false });
const validateLogin = (data) => loginSchema.validate(data, { abortEarly: false });
const validateVerificarTelefone = (data) => verificarTelefoneSchema.validate(data, { abortEarly: false });
const validateUpdateProfile = (data) => updateProfileSchema.validate(data, { abortEarly: false });

module.exports = {
  validateRegister,
  validateLogin,
  validateVerificarTelefone,
  validateUpdateProfile
};
