# Testes TudoAqui Backend

Suite de testes automatizados para o backend do marketplace TudoAqui.

## Estrutura

```
tests/
├── setup.js              # Configuração global de testes
├── auth.test.js          # Testes de autenticação
├── pagamentos.test.js    # Testes de pagamentos
├── imoveis.test.js       # Testes de imóveis (TODO)
├── transporte.test.js    # Testes de transporte (TODO)
└── README.md             # Este arquivo
```

## Configuração

### 1. Criar Database de Teste

```bash
# Conectar ao PostgreSQL como superuser
psql -U postgres

# Criar database de teste
CREATE DATABASE tudo_aqui_test;

# Conceder permissões
GRANT ALL PRIVILEGES ON DATABASE tudo_aqui_test TO tudo_aqui_admin;

# Sair
\q
```

### 2. Executar Migrations no Database de Teste

```bash
# Windows
cd scripts
$env:DATABASE_NAME="tudo_aqui_test"; .\run-migrations.bat

# Linux/Mac
export DATABASE_NAME=tudo_aqui_test && ./run-migrations.sh
```

### 3. Configurar Variáveis de Ambiente

Copiar `.env.test` para `.env` ou garantir que `.env.test` está carregado:

```bash
cp .env.test .env
```

### 4. Instalar Dependências de Teste

```bash
npm install --save-dev jest supertest
```

## Executar Testes

```bash
# Todos os testes
npm test

# Testes específicos
npm test auth.test.js
npm test pagamentos.test.js

# Com coverage
npm run test:coverage

# Watch mode (re-executar ao salvar)
npm run test:watch

# Verbose (mais detalhes)
npm test -- --verbose
```

## Comandos package.json

Adicionar ao `package.json`:

```json
{
  "scripts": {
    "test": "NODE_ENV=test jest --forceExit --detectOpenHandles",
    "test:watch": "NODE_ENV=test jest --watch",
    "test:coverage": "NODE_ENV=test jest --coverage --forceExit"
  }
}
```

## Estrutura de um Teste

```javascript
const request = require('supertest');
const app = require('../src/app');
const { testHelpers } = require('./setup');

describe('Módulo de Teste', () => {
  let usuario, token;

  beforeEach(async () => {
    // Setup antes de cada teste
    usuario = await testHelpers.criarUsuarioTeste();
    token = testHelpers.gerarTokenTeste(usuario);
  });

  test('Deve fazer algo esperado', async () => {
    const response = await request(app)
      .post('/api/v1/endpoint')
      .set('Authorization', `Bearer ${token}`)
      .send({ dados: 'teste' })
      .expect(200);

    expect(response.body.success).toBe(true);
  });
});
```

## Helpers Disponíveis

```javascript
const { testHelpers } = require('./setup');

// Criar usuário de teste
const usuario = await testHelpers.criarUsuarioTeste({
  nome: 'João Silva',
  tipo_usuario: 'vendedor',
  selo_verificado: true
});

// Gerar token JWT
const token = testHelpers.gerarTokenTeste(usuario);

// Criar imóvel de teste
const imovel = await testHelpers.criarImovelTeste(vendedorId, {
  preco: 50000000,
  quartos: 3
});

// Criar transação de teste
const transacao = await testHelpers.criarTransacaoTeste(usuarioId, {
  valor: 100000,
  metodo_pagamento: 'multicaixa'
});
```

## Coverage

Medir cobertura de código:

```bash
npm run test:coverage
```

Relatório gerado em `coverage/`:
- `coverage/index.html` - Relatório visual
- `coverage/lcov.info` - Para ferramentas de CI/CD

## Boas Práticas

1. **Isolamento**: Cada teste deve ser independente
2. **Limpeza**: `beforeEach` limpa todas as tabelas automaticamente
3. **Nomes descritivos**: Use `test('Deve fazer X quando Y')`
4. **Assertions claras**: Sempre verifique `success`, `error`, e dados esperados
5. **Status HTTP**: Use `.expect(200)` para validar códigos
6. **Async/Await**: Sempre use `async/await` para operações de banco

## CI/CD

Para rodar em pipelines (GitHub Actions, GitLab CI):

```yaml
# .github/workflows/tests.yml
name: Tests
on: [push, pull_request]
jobs:
  test:
    runs-on: ubuntu-latest
    services:
      postgres:
        image: postgres:13
        env:
          POSTGRES_PASSWORD: senha_segura
          POSTGRES_DB: tudo_aqui_test
        ports:
          - 5432:5432
      redis:
        image: redis:6
        ports:
          - 6379:6379
    steps:
      - uses: actions/checkout@v2
      - uses: actions/setup-node@v2
        with:
          node-version: '18'
      - run: npm install
      - run: npm run test:coverage
      - uses: codecov/codecov-action@v2
```

## Troubleshooting

### Erro: "Connection refused"
- Verificar se PostgreSQL e Redis estão rodando
- Confirmar portas corretas (5432, 6379)

### Erro: "Database does not exist"
- Criar database: `createdb tudo_aqui_test`
- Executar migrations no database de teste

### Timeout
- Aumentar timeout no `jest.config.js`: `testTimeout: 20000`

### Testes falhando por ordem de execução
- Garantir que `beforeEach` está limpando dados
- Usar `--runInBand` para executar sequencialmente

## Próximos Passos

- [ ] Criar testes para módulo de imóveis
- [ ] Criar testes para módulo de transporte
- [ ] Criar testes para módulo de alojamentos
- [ ] Criar testes para módulo de ingressos
- [ ] Adicionar testes de integração end-to-end
- [ ] Implementar testes de performance/load
- [ ] Configurar CI/CD com GitHub Actions
