const request = require('supertest');
const app = require('../src/app');
const { pool } = require('../src/config/database');
const { testHelpers } = require('./setup');

describe('Auth Module - Autenticação e Registro', () => {
  
  describe('POST /api/v1/auth/register - Registro de Usuário', () => {
    test('Deve registrar novo usuário com dados válidos', async () => {
      const dadosRegistro = {
        nome: 'João Silva',
        email: 'joao.silva@example.com',
        telefone: '923456789',
        senha: 'SenhaForte123',
        tipo_usuario: 'comprador'
      };

      const response = await request(app)
        .post('/api/v1/auth/register')
        .send(dadosRegistro)
        .expect(201);

      expect(response.body.success).toBe(true);
      expect(response.body.user).toHaveProperty('id');
      expect(response.body.user.email).toBe(dadosRegistro.email);
      expect(response.body).toHaveProperty('token');

      // Verificar se usuário foi criado no banco
      const result = await pool.query('SELECT * FROM usuarios WHERE email = $1', [dadosRegistro.email]);
      expect(result.rows.length).toBe(1);
    });

    test('Deve rejeitar registro com email duplicado', async () => {
      const email = 'duplicado@example.com';
      
      // Criar primeiro usuário
      await testHelpers.criarUsuarioTeste({ email });

      // Tentar criar segundo usuário com mesmo email
      const response = await request(app)
        .post('/api/v1/auth/register')
        .send({
          nome: 'Maria Santos',
          email: email,
          telefone: '923456780',
          senha: 'SenhaForte123',
          tipo_usuario: 'comprador'
        })
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('email');
    });

    test('Deve rejeitar senha fraca', async () => {
      const response = await request(app)
        .post('/api/v1/auth/register')
        .send({
          nome: 'Pedro Costa',
          email: 'pedro@example.com',
          telefone: '923456781',
          senha: '123', // Senha fraca
          tipo_usuario: 'comprador'
        })
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('senha');
    });
  });

  describe('POST /api/v1/auth/login - Login de Usuário', () => {
    let usuario;
    const senhaOriginal = 'SenhaForte123';

    beforeEach(async () => {
      usuario = await testHelpers.criarUsuarioTeste({
        email: 'login@example.com',
        senha: senhaOriginal
      });
    });

    test('Deve fazer login com credenciais válidas', async () => {
      const response = await request(app)
        .post('/api/v1/auth/login')
        .send({
          email: usuario.email,
          senha: senhaOriginal
        })
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body).toHaveProperty('token');
      expect(response.body.user.id).toBe(usuario.id);
    });

    test('Deve rejeitar login com senha incorreta', async () => {
      const response = await request(app)
        .post('/api/v1/auth/login')
        .send({
          email: usuario.email,
          senha: 'SenhaErrada123'
        })
        .expect(401);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('inválidas');
    });

    test('Deve rejeitar login de usuário inativo', async () => {
      // Desativar usuário
      await pool.query('UPDATE usuarios SET status = $1 WHERE id = $2', ['inativo', usuario.id]);

      const response = await request(app)
        .post('/api/v1/auth/login')
        .send({
          email: usuario.email,
          senha: senhaOriginal
        })
        .expect(403);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('inativo');
    });
  });

  describe('GET /api/v1/auth/me - Perfil do Usuário', () => {
    let usuario, token;

    beforeEach(async () => {
      usuario = await testHelpers.criarUsuarioTeste();
      token = testHelpers.gerarTokenTeste(usuario);
    });

    test('Deve retornar dados do usuário autenticado', async () => {
      const response = await request(app)
        .get('/api/v1/auth/me')
        .set('Authorization', `Bearer ${token}`)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.user.id).toBe(usuario.id);
      expect(response.body.user.email).toBe(usuario.email);
    });

    test('Deve rejeitar requisição sem token', async () => {
      const response = await request(app)
        .get('/api/v1/auth/me')
        .expect(401);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('Token');
    });

    test('Deve rejeitar token inválido', async () => {
      const response = await request(app)
        .get('/api/v1/auth/me')
        .set('Authorization', 'Bearer token_invalido')
        .expect(401);

      expect(response.body.success).toBe(false);
    });
  });

  describe('POST /api/v1/auth/logout - Logout', () => {
    let usuario, token;

    beforeEach(async () => {
      usuario = await testHelpers.criarUsuarioTeste();
      token = testHelpers.gerarTokenTeste(usuario);
    });

    test('Deve fazer logout e invalidar token', async () => {
      // Logout
      const response = await request(app)
        .post('/api/v1/auth/logout')
        .set('Authorization', `Bearer ${token}`)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.message).toContain('sucesso');

      // Tentar usar token após logout (deve falhar)
      const responseAposLogout = await request(app)
        .get('/api/v1/auth/me')
        .set('Authorization', `Bearer ${token}`)
        .expect(401);

      expect(responseAposLogout.body.error).toContain('inválido');
    });
  });

  describe('PUT /api/v1/auth/perfil - Atualizar Perfil', () => {
    let usuario, token;

    beforeEach(async () => {
      usuario = await testHelpers.criarUsuarioTeste();
      token = testHelpers.gerarTokenTeste(usuario);
    });

    test('Deve atualizar nome do usuário', async () => {
      const novoNome = 'Nome Atualizado';
      
      const response = await request(app)
        .put('/api/v1/auth/perfil')
        .set('Authorization', `Bearer ${token}`)
        .send({ nome: novoNome })
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.user.nome).toBe(novoNome);

      // Verificar no banco
      const result = await pool.query('SELECT nome FROM usuarios WHERE id = $1', [usuario.id]);
      expect(result.rows[0].nome).toBe(novoNome);
    });

    test('Deve rejeitar atualização com email já existente', async () => {
      // Criar outro usuário
      const outroUsuario = await testHelpers.criarUsuarioTeste({ email: 'existente@example.com' });

      // Tentar atualizar para email existente
      const response = await request(app)
        .put('/api/v1/auth/perfil')
        .set('Authorization', `Bearer ${token}`)
        .send({ email: outroUsuario.email })
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('email');
    });
  });
});
