const request = require('supertest');
const app = require('../src/app');
const { pool } = require('../src/config/database');
const { testHelpers } = require('./setup');

describe('Imóveis Module - Sistema de Imobiliário', () => {
  
  let comprador, vendedor, tokenComprador, tokenVendedor;

  beforeEach(async () => {
    // Criar usuários de teste
    comprador = await testHelpers.criarUsuarioTeste({
      nome: 'Comprador Imóveis',
      email: 'comprador.imoveis@example.com',
      tipo_usuario: 'comprador'
    });
    tokenComprador = testHelpers.gerarTokenTeste(comprador);

    vendedor = await testHelpers.criarUsuarioTeste({
      nome: 'Vendedor Imóveis',
      email: 'vendedor.imoveis@example.com',
      tipo_usuario: 'vendedor',
      selo_verificado: true
    });
    tokenVendedor = testHelpers.gerarTokenTeste(vendedor);
  });

  describe('POST /api/v1/imoveis - Criar Imóvel', () => {
    
    test('Deve criar imóvel com dados válidos', async () => {
      const dadosImovel = {
        titulo: 'Casa 3 quartos em Luanda',
        descricao: 'Casa com excelente localização no Talatona',
        tipo: 'casa',
        finalidade: 'venda',
        preco: 15000000,
        localizacao: 'Luanda, Talatona',
        provincia: 'Luanda',
        municipio: 'Talatona',
        latitude: -8.8383,
        longitude: 13.2344,
        quartos: 3,
        banheiros: 2,
        suites: 1,
        vagas_garagem: 2,
        area_total: 250,
        area_construida: 180,
        comodidades: ['piscina', 'jardim', 'churrasqueira'],
        fotos: ['foto1.jpg', 'foto2.jpg', 'foto3.jpg']
      };

      const response = await request(app)
        .post('/api/v1/imoveis')
        .set('Authorization', `Bearer ${tokenVendedor}`)
        .send(dadosImovel)
        .expect(201);

      expect(response.body.success).toBe(true);
      expect(response.body.imovel).toHaveProperty('id');
      expect(response.body.imovel.titulo).toBe(dadosImovel.titulo);
      expect(response.body.imovel.preco).toBe('15000000.00');
      expect(response.body.imovel.vendedor_id).toBe(vendedor.id);

      // Verificar no banco
      const result = await pool.query('SELECT * FROM imoveis WHERE id = $1', [response.body.imovel.id]);
      expect(result.rows.length).toBe(1);
      expect(result.rows[0].tipo).toBe('casa');
    });

    test('Deve criar apartamento para aluguel', async () => {
      const dadosApartamento = {
        titulo: 'Apartamento T2 em Miramar',
        descricao: 'Apartamento moderno com vista para o mar',
        tipo: 'apartamento',
        finalidade: 'aluguel',
        preco: 150000, // Aluguel mensal
        localizacao: 'Luanda, Miramar',
        provincia: 'Luanda',
        municipio: 'Miramar',
        latitude: -8.8168,
        longitude: 13.2332,
        quartos: 2,
        banheiros: 2,
        area_total: 90,
        andar: 5,
        comodidades: ['elevador', 'varanda', 'condominio_fechado'],
        fotos: ['apt1.jpg']
      };

      const response = await request(app)
        .post('/api/v1/imoveis')
        .set('Authorization', `Bearer ${tokenVendedor}`)
        .send(dadosApartamento)
        .expect(201);

      expect(response.body.success).toBe(true);
      expect(response.body.imovel.tipo).toBe('apartamento');
      expect(response.body.imovel.finalidade).toBe('aluguel');
    });

    test('Deve criar terreno para venda', async () => {
      const dadosTerreno = {
        titulo: 'Terreno 1000m² em Viana',
        descricao: 'Terreno plano, murado',
        tipo: 'terreno',
        finalidade: 'venda',
        preco: 8000000,
        localizacao: 'Luanda, Viana',
        provincia: 'Luanda',
        municipio: 'Viana',
        area_total: 1000,
        fotos: ['terreno1.jpg']
      };

      const response = await request(app)
        .post('/api/v1/imoveis')
        .set('Authorization', `Bearer ${tokenVendedor}`)
        .send(dadosTerreno)
        .expect(201);

      expect(response.body.success).toBe(true);
      expect(response.body.imovel.tipo).toBe('terreno');
    });

    test('Deve rejeitar criação sem autenticação', async () => {
      const response = await request(app)
        .post('/api/v1/imoveis')
        .send({
          titulo: 'Casa teste',
          tipo: 'casa',
          finalidade: 'venda',
          preco: 10000000
        })
        .expect(401);

      expect(response.body.success).toBe(false);
    });

    test('Deve rejeitar criação por usuário não vendedor', async () => {
      const response = await request(app)
        .post('/api/v1/imoveis')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .send({
          titulo: 'Casa teste',
          tipo: 'casa',
          finalidade: 'venda',
          preco: 10000000,
          localizacao: 'Luanda'
        })
        .expect(403);

      expect(response.body.success).toBe(false);
    });

    test('Deve rejeitar criação com dados inválidos', async () => {
      const response = await request(app)
        .post('/api/v1/imoveis')
        .set('Authorization', `Bearer ${tokenVendedor}`)
        .send({
          titulo: 'AB', // Título muito curto
          tipo: 'invalido',
          preco: -1000 // Preço negativo
        })
        .expect(400);

      expect(response.body.success).toBe(false);
    });
  });

  describe('GET /api/v1/imoveis - Listar Imóveis', () => {
    
    beforeEach(async () => {
      // Criar 10 imóveis variados para testes
      const imoveis = [
        { titulo: 'Casa Talatona 1', tipo: 'casa', finalidade: 'venda', preco: 10000000, quartos: 2, localizacao: 'Luanda, Talatona' },
        { titulo: 'Casa Talatona 2', tipo: 'casa', finalidade: 'venda', preco: 12000000, quartos: 3, localizacao: 'Luanda, Talatona' },
        { titulo: 'Apartamento Miramar', tipo: 'apartamento', finalidade: 'aluguel', preco: 100000, quartos: 2, localizacao: 'Luanda, Miramar' },
        { titulo: 'Casa Viana', tipo: 'casa', finalidade: 'venda', preco: 8000000, quartos: 3, localizacao: 'Luanda, Viana' },
        { titulo: 'Terreno Kilamba', tipo: 'terreno', finalidade: 'venda', preco: 5000000, area_total: 500, localizacao: 'Luanda, Kilamba' },
        { titulo: 'Apartamento Maianga', tipo: 'apartamento', finalidade: 'aluguel', preco: 150000, quartos: 3, localizacao: 'Luanda, Maianga' },
        { titulo: 'Casa Benfica', tipo: 'casa', finalidade: 'temporada', preco: 50000, quartos: 4, localizacao: 'Luanda, Benfica' },
        { titulo: 'Escritório Centro', tipo: 'escritorio', finalidade: 'aluguel', preco: 200000, area_total: 120, localizacao: 'Luanda, Centro' },
        { titulo: 'Casa Luxo Chicala', tipo: 'casa', finalidade: 'venda', preco: 50000000, quartos: 5, localizacao: 'Luanda, Chicala' },
        { titulo: 'Apartamento Ingombota', tipo: 'apartamento', finalidade: 'venda', preco: 15000000, quartos: 2, localizacao: 'Luanda, Ingombota' }
      ];

      for (const dados of imoveis) {
        await testHelpers.criarImovelTeste(vendedor.id, dados);
      }
    });

    test('Deve listar todos os imóveis com paginação', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis?limite=5&offset=0')
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.imoveis.length).toBe(5);
      expect(response.body.total).toBe(10);
    });

    test('Deve filtrar por tipo de imóvel', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis?tipo=apartamento')
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.imoveis.every(i => i.tipo === 'apartamento')).toBe(true);
      expect(response.body.imoveis.length).toBe(3);
    });

    test('Deve filtrar por finalidade', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis?finalidade=venda')
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.imoveis.every(i => i.finalidade === 'venda')).toBe(true);
    });

    test('Deve filtrar por faixa de preço', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis?preco_min=10000000&preco_max=20000000')
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.imoveis.every(i => 
        parseFloat(i.preco) >= 10000000 && parseFloat(i.preco) <= 20000000
      )).toBe(true);
    });

    test('Deve filtrar por número mínimo de quartos', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis?quartos_min=3')
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.imoveis.every(i => i.quartos >= 3 || i.quartos === null)).toBe(true);
    });

    test('Deve buscar por localização', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis?localizacao=Talatona')
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.imoveis.every(i => 
        i.localizacao.toLowerCase().includes('talatona')
      )).toBe(true);
    });

    test('Deve ordenar por preço crescente', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis?ordenar=preco')
        .expect(200);

      expect(response.body.success).toBe(true);
      const precos = response.body.imoveis.map(i => parseFloat(i.preco));
      const precosOrdenados = [...precos].sort((a, b) => a - b);
      expect(precos).toEqual(precosOrdenados);
    });

    test('Deve ordenar por quartos decrescente', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis?ordenar=quartos_desc')
        .expect(200);

      expect(response.body.success).toBe(true);
      const quartos = response.body.imoveis.map(i => i.quartos || 0);
      const quartosOrdenados = [...quartos].sort((a, b) => b - a);
      expect(quartos).toEqual(quartosOrdenados);
    });

    test('Deve combinar múltiplos filtros', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis?tipo=casa&finalidade=venda&preco_min=8000000&quartos_min=3')
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.imoveis.every(i => 
        i.tipo === 'casa' && 
        i.finalidade === 'venda' && 
        parseFloat(i.preco) >= 8000000 &&
        i.quartos >= 3
      )).toBe(true);
    });

    test('Deve respeitar limite máximo de 100 registros', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis?limite=200')
        .expect(200);

      expect(response.body.imoveis.length).toBeLessThanOrEqual(100);
    });
  });

  describe('GET /api/v1/imoveis/:id - Buscar Imóvel por ID', () => {
    
    let imovel;

    beforeEach(async () => {
      imovel = await testHelpers.criarImovelTeste(vendedor.id, {
        titulo: 'Casa Detalhes',
        preco: 20000000,
        quartos: 4
      });
    });

    test('Deve retornar detalhes do imóvel', async () => {
      const response = await request(app)
        .get(`/api/v1/imoveis/${imovel.id}`)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.imovel.id).toBe(imovel.id);
      expect(response.body.imovel.titulo).toBe('Casa Detalhes');
      expect(response.body.imovel).toHaveProperty('vendedor');
    });

    test('Deve incrementar visualizações', async () => {
      // Primeira visualização
      await request(app).get(`/api/v1/imoveis/${imovel.id}`).expect(200);

      // Segunda visualização
      const response = await request(app).get(`/api/v1/imoveis/${imovel.id}`).expect(200);

      // Verificar contador no banco
      const result = await pool.query('SELECT visualizacoes FROM imoveis WHERE id = $1', [imovel.id]);
      expect(result.rows[0].visualizacoes).toBeGreaterThan(0);
    });

    test('Deve retornar imóveis similares', async () => {
      // Criar imóveis similares
      await testHelpers.criarImovelTeste(vendedor.id, {
        tipo: 'casa',
        preco: 19000000, // ±20% do preço original
        quartos: 4
      });

      const response = await request(app)
        .get(`/api/v1/imoveis/${imovel.id}`)
        .expect(200);

      expect(response.body.success).toBe(true);
      if (response.body.similares) {
        expect(Array.isArray(response.body.similares)).toBe(true);
      }
    });

    test('Deve retornar 404 para imóvel inexistente', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis/00000000-0000-0000-0000-000000000000')
        .expect(404);

      expect(response.body.success).toBe(false);
    });
  });

  describe('GET /api/v1/imoveis/meus - Meus Imóveis', () => {
    
    beforeEach(async () => {
      // Criar 3 imóveis do vendedor
      for (let i = 1; i <= 3; i++) {
        await testHelpers.criarImovelTeste(vendedor.id, {
          titulo: `Meu Imóvel ${i}`,
          preco: 10000000 * i
        });
      }

      // Criar 1 imóvel de outro vendedor
      const outroVendedor = await testHelpers.criarUsuarioTeste({
        tipo_usuario: 'vendedor',
        email: 'outro@example.com'
      });
      await testHelpers.criarImovelTeste(outroVendedor.id, {
        titulo: 'Imóvel de Outro'
      });
    });

    test('Deve listar apenas imóveis do vendedor autenticado', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis/meus')
        .set('Authorization', `Bearer ${tokenVendedor}`)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.imoveis.length).toBe(3);
      expect(response.body.imoveis.every(i => i.vendedor_id === vendedor.id)).toBe(true);
      expect(response.body.imoveis.every(i => i.titulo.startsWith('Meu Imóvel'))).toBe(true);
    });

    test('Deve exigir autenticação', async () => {
      const response = await request(app)
        .get('/api/v1/imoveis/meus')
        .expect(401);

      expect(response.body.success).toBe(false);
    });
  });

  describe('PUT /api/v1/imoveis/:id - Atualizar Imóvel', () => {
    
    let imovel;

    beforeEach(async () => {
      imovel = await testHelpers.criarImovelTeste(vendedor.id, {
        titulo: 'Casa Original',
        preco: 10000000,
        quartos: 3
      });
    });

    test('Deve atualizar imóvel com sucesso', async () => {
      const dadosAtualizacao = {
        titulo: 'Casa Atualizada',
        preco: 12000000,
        quartos: 4
      };

      const response = await request(app)
        .put(`/api/v1/imoveis/${imovel.id}`)
        .set('Authorization', `Bearer ${tokenVendedor}`)
        .send(dadosAtualizacao)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.imovel.titulo).toBe('Casa Atualizada');
      expect(parseFloat(response.body.imovel.preco)).toBe(12000000);
      expect(response.body.imovel.quartos).toBe(4);

      // Verificar no banco
      const result = await pool.query('SELECT * FROM imoveis WHERE id = $1', [imovel.id]);
      expect(result.rows[0].titulo).toBe('Casa Atualizada');
    });

    test('Deve rejeitar atualização de imóvel de outro vendedor', async () => {
      const outroVendedor = await testHelpers.criarUsuarioTeste({
        tipo_usuario: 'vendedor',
        email: 'outro2@example.com'
      });
      const tokenOutro = testHelpers.gerarTokenTeste(outroVendedor);

      const response = await request(app)
        .put(`/api/v1/imoveis/${imovel.id}`)
        .set('Authorization', `Bearer ${tokenOutro}`)
        .send({ titulo: 'Tentativa de Hack' })
        .expect(403);

      expect(response.body.success).toBe(false);
    });
  });

  describe('DELETE /api/v1/imoveis/:id - Deletar Imóvel', () => {
    
    let imovel;

    beforeEach(async () => {
      imovel = await testHelpers.criarImovelTeste(vendedor.id, {
        titulo: 'Casa para Deletar',
        preco: 10000000
      });
    });

    test('Deve deletar imóvel com sucesso', async () => {
      const response = await request(app)
        .delete(`/api/v1/imoveis/${imovel.id}`)
        .set('Authorization', `Bearer ${tokenVendedor}`)
        .expect(200);

      expect(response.body.success).toBe(true);

      // Verificar que foi deletado
      const result = await pool.query('SELECT * FROM imoveis WHERE id = $1', [imovel.id]);
      expect(result.rows.length).toBe(0);
    });

    test('Deve rejeitar deleção de imóvel de outro vendedor', async () => {
      const outroVendedor = await testHelpers.criarUsuarioTeste({
        tipo_usuario: 'vendedor',
        email: 'outro3@example.com'
      });
      const tokenOutro = testHelpers.gerarTokenTeste(outroVendedor);

      const response = await request(app)
        .delete(`/api/v1/imoveis/${imovel.id}`)
        .set('Authorization', `Bearer ${tokenOutro}`)
        .expect(403);

      expect(response.body.success).toBe(false);

      // Verificar que NÃO foi deletado
      const result = await pool.query('SELECT * FROM imoveis WHERE id = $1', [imovel.id]);
      expect(result.rows.length).toBe(1);
    });
  });
});
