const request = require('supertest');
const app = require('../src/app');
const { pool } = require('../src/config/database');
const { testHelpers } = require('./setup');

describe('Pagamentos Module - Sistema de Pagamentos', () => {
  
  let comprador, vendedor, tokenComprador, tokenVendedor, imovel;

  beforeEach(async () => {
    // Criar usuários de teste
    comprador = await testHelpers.criarUsuarioTeste({
      nome: 'Comprador Teste',
      tipo_usuario: 'comprador'
    });
    tokenComprador = testHelpers.gerarTokenTeste(comprador);

    vendedor = await testHelpers.criarUsuarioTeste({
      nome: 'Vendedor Teste',
      tipo_usuario: 'vendedor',
      email: 'vendedor@example.com'
    });
    tokenVendedor = testHelpers.gerarTokenTeste(vendedor);

    // Criar imóvel de teste
    imovel = await testHelpers.criarImovelTeste(vendedor.id, {
      preco: 50000000 // 50 milhões AOA
    });
  });

  describe('POST /api/v1/pagamentos/processar - Processar Pagamento', () => {
    
    test('Deve processar pagamento via Multicaixa com sucesso', async () => {
      const dadosPagamento = {
        item_id: imovel.id,
        modulo: 'imoveis',
        valor: imovel.preco,
        metodo_pagamento: 'multicaixa',
        dados_pagamento: {
          telefone: '923456789'
        }
      };

      const response = await request(app)
        .post('/api/v1/pagamentos/processar')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .send(dadosPagamento)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body).toHaveProperty('transacao_id');
      expect(response.body.status).toBeTruthy();
      expect(response.body.mensagem).toContain('Multicaixa');

      // Verificar se transação foi criada no banco
      const transacao = await pool.query(
        'SELECT * FROM transacoes WHERE id = $1',
        [response.body.transacao_id]
      );
      expect(transacao.rows.length).toBe(1);
      expect(transacao.rows[0].metodo_pagamento).toBe('multicaixa');
    });

    test('Deve processar pagamento via transferência bancária', async () => {
      const dadosPagamento = {
        item_id: imovel.id,
        modulo: 'imoveis',
        valor: 1000000, // 1 milhão AOA
        metodo_pagamento: 'transferencia'
      };

      const response = await request(app)
        .post('/api/v1/pagamentos/processar')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .send(dadosPagamento)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.dados_adicionais).toHaveProperty('banco');
      expect(response.body.dados_adicionais).toHaveProperty('nib');
      expect(response.body.dados_adicionais).toHaveProperty('iban');
      expect(response.body.dados_adicionais.referencia).toBe(response.body.transacao_id);
    });

    test('Deve processar pagamento em dinheiro', async () => {
      const dadosPagamento = {
        item_id: imovel.id,
        modulo: 'imoveis',
        valor: 500000,
        metodo_pagamento: 'dinheiro'
      };

      const response = await request(app)
        .post('/api/v1/pagamentos/processar')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .send(dadosPagamento)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.status).toBe('aguardando_confirmacao');
      expect(response.body.dados_adicionais.instrucoes).toContain('diretamente');
    });

    test('Deve rejeitar pagamento sem dados obrigatórios', async () => {
      const response = await request(app)
        .post('/api/v1/pagamentos/processar')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .send({
          // Faltam item_id, modulo, valor
          metodo_pagamento: 'multicaixa'
        })
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('incompletos');
    });

    test('Deve rejeitar pagamento com valor abaixo do mínimo', async () => {
      const response = await request(app)
        .post('/api/v1/pagamentos/processar')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .send({
          item_id: imovel.id,
          modulo: 'imoveis',
          valor: 50, // Abaixo de 100 AOA
          metodo_pagamento: 'dinheiro'
        })
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('mínimo');
    });

    test('Deve rejeitar método de pagamento inválido', async () => {
      const response = await request(app)
        .post('/api/v1/pagamentos/processar')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .send({
          item_id: imovel.id,
          modulo: 'imoveis',
          valor: 100000,
          metodo_pagamento: 'bitcoin' // Não suportado
        })
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('inválido');
    });

    test('Deve rejeitar módulo inválido', async () => {
      const response = await request(app)
        .post('/api/v1/pagamentos/processar')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .send({
          item_id: imovel.id,
          modulo: 'modulo_inexistente',
          valor: 100000,
          metodo_pagamento: 'dinheiro'
        })
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('Módulo inválido');
    });
  });

  describe('POST /api/v1/pagamentos/confirmar - Confirmar Pagamento', () => {
    
    let transacao;

    beforeEach(async () => {
      // Criar transação de teste
      transacao = await testHelpers.criarTransacaoTeste(comprador.id, {
        modulo: 'imoveis',
        item_id: imovel.id,
        valor: 1000000,
        metodo_pagamento: 'transferencia',
        status_transacao: 'aguardando_confirmacao'
      });
    });

    test('Deve confirmar pagamento com comprovativo', async () => {
      const response = await request(app)
        .post('/api/v1/pagamentos/confirmar')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .send({
          transacao_id: transacao.id,
          comprovativo: 'https://s3.amazonaws.com/comprovativo123.jpg'
        })
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.status).toBe('confirmada');

      // Verificar status no banco
      const result = await pool.query(
        'SELECT status_transacao, data_confirmacao FROM transacoes WHERE id = $1',
        [transacao.id]
      );
      expect(result.rows[0].status_transacao).toBe('confirmada');
      expect(result.rows[0].data_confirmacao).toBeTruthy();
    });

    test('Deve rejeitar confirmação de transação já confirmada', async () => {
      // Confirmar primeira vez
      await pool.query(
        'UPDATE transacoes SET status_transacao = $1 WHERE id = $2',
        ['confirmada', transacao.id]
      );

      // Tentar confirmar novamente
      const response = await request(app)
        .post('/api/v1/pagamentos/confirmar')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .send({
          transacao_id: transacao.id,
          comprovativo: 'https://s3.amazonaws.com/comprovativo123.jpg'
        })
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('já confirmada');
    });

    test('Deve rejeitar confirmação de transação de outro usuário', async () => {
      const response = await request(app)
        .post('/api/v1/pagamentos/confirmar')
        .set('Authorization', `Bearer ${tokenVendedor}`) // Usando token do vendedor
        .send({
          transacao_id: transacao.id,
          comprovativo: 'https://s3.amazonaws.com/comprovativo123.jpg'
        })
        .expect(404);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('não encontrada');
    });
  });

  describe('GET /api/v1/pagamentos/historico - Histórico de Transações', () => {
    
    beforeEach(async () => {
      // Criar várias transações de teste
      await testHelpers.criarTransacaoTeste(comprador.id, {
        valor: 100000,
        status_transacao: 'confirmada'
      });
      await testHelpers.criarTransacaoTeste(comprador.id, {
        valor: 200000,
        status_transacao: 'pendente'
      });
      await testHelpers.criarTransacaoTeste(comprador.id, {
        valor: 300000,
        status_transacao: 'cancelada',
        modulo: 'transporte'
      });
    });

    test('Deve listar todas as transações do usuário', async () => {
      const response = await request(app)
        .get('/api/v1/pagamentos/historico')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.transacoes.length).toBe(3);
      expect(response.body.total).toBe(3);
    });

    test('Deve filtrar transações por status', async () => {
      const response = await request(app)
        .get('/api/v1/pagamentos/historico?status=confirmada')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.transacoes.length).toBe(1);
      expect(response.body.transacoes[0].status_transacao).toBe('confirmada');
    });

    test('Deve filtrar transações por módulo', async () => {
      const response = await request(app)
        .get('/api/v1/pagamentos/historico?modulo=transporte')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.transacoes.length).toBe(1);
      expect(response.body.transacoes[0].modulo).toBe('transporte');
    });

    test('Deve respeitar paginação', async () => {
      const response = await request(app)
        .get('/api/v1/pagamentos/historico?limite=2&offset=0')
        .set('Authorization', `Bearer ${tokenComprador}`)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.transacoes.length).toBe(2);
      expect(response.body.total).toBe(3);
      expect(response.body.limite).toBe(2);
    });
  });

  describe('GET /api/v1/pagamentos/:id - Buscar Transação', () => {
    
    let transacao;

    beforeEach(async () => {
      transacao = await testHelpers.criarTransacaoTeste(comprador.id, {
        valor: 500000
      });
    });

    test('Deve retornar detalhes da transação', async () => {
      const response = await request(app)
        .get(`/api/v1/pagamentos/${transacao.id}`)
        .set('Authorization', `Bearer ${tokenComprador}`)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.transacao.id).toBe(transacao.id);
      expect(response.body.transacao.valor).toBe('500000.00');
    });

    test('Deve rejeitar acesso a transação de outro usuário', async () => {
      const response = await request(app)
        .get(`/api/v1/pagamentos/${transacao.id}`)
        .set('Authorization', `Bearer ${tokenVendedor}`)
        .expect(404);

      expect(response.body.success).toBe(false);
    });
  });

  describe('DELETE /api/v1/pagamentos/:id - Cancelar Transação', () => {
    
    let transacao;

    beforeEach(async () => {
      transacao = await testHelpers.criarTransacaoTeste(comprador.id, {
        status_transacao: 'pendente'
      });
    });

    test('Deve cancelar transação pendente', async () => {
      const response = await request(app)
        .delete(`/api/v1/pagamentos/${transacao.id}`)
        .set('Authorization', `Bearer ${tokenComprador}`)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.message).toContain('cancelada');

      // Verificar no banco
      const result = await pool.query(
        'SELECT status_transacao FROM transacoes WHERE id = $1',
        [transacao.id]
      );
      expect(result.rows[0].status_transacao).toBe('cancelada');
    });

    test('Deve rejeitar cancelamento de transação confirmada', async () => {
      // Confirmar transação primeiro
      await pool.query(
        'UPDATE transacoes SET status_transacao = $1 WHERE id = $2',
        ['confirmada', transacao.id]
      );

      const response = await request(app)
        .delete(`/api/v1/pagamentos/${transacao.id}`)
        .set('Authorization', `Bearer ${tokenComprador}`)
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toContain('já confirmada');
    });
  });
});
