const { pool } = require('../src/config/database');
const redis = require('../src/config/redis');

/**
 * Setup de testes para TudoAqui Backend
 * 
 * Configura ambiente de teste isolado:
 * - Database PostgreSQL de teste (tudo_aqui_test)
 * - Redis cache/sessions
 * - Limpeza automática de dados entre testes
 */

// Configurar variáveis de ambiente para teste
process.env.NODE_ENV = 'test';
process.env.JWT_SECRET = 'test_jwt_secret_key_super_secure';
process.env.DATABASE_URL = 'postgresql://tudo_aqui_admin:senha_segura@localhost:5432/tudo_aqui_test';
process.env.REDIS_URL = 'redis://localhost:6379/1'; // DB 1 para testes

// Aumentar timeout para operações de banco
jest.setTimeout(15000);

// Hook executado ANTES de todos os testes
beforeAll(async () => {
  console.log('\n🚀 Iniciando setup de testes...\n');

  try {
    // Verificar conexão com PostgreSQL
    await pool.query('SELECT NOW()');
    console.log('✅ Conectado ao PostgreSQL');

    // Verificar conexão com Redis
    await redis.ping();
    console.log('✅ Conectado ao Redis');

    // Criar database de teste se não existir (PostgreSQL não suporta IF NOT EXISTS no CREATE DATABASE)
    // Nota: Este código assume que o database já foi criado manualmente
    // Para criar: psql -U postgres -c "CREATE DATABASE tudo_aqui_test;"

    console.log('\n✅ Setup de testes concluído\n');
  } catch (error) {
    console.error('❌ Erro no setup de testes:', error.message);
    throw error;
  }
});

// Hook executado ANTES de cada teste
beforeEach(async () => {
  try {
    // Limpar todas as tabelas (em ordem para respeitar foreign keys)
    await pool.query('BEGIN');

    // Desabilitar triggers temporariamente para limpeza rápida
    await pool.query('SET session_replication_role = replica');

    // Limpar tabelas de dados (em ordem de dependências)
    await pool.query('TRUNCATE TABLE ingressos_vendidos CASCADE');
    await pool.query('TRUNCATE TABLE eventos_ingressos CASCADE');
    await pool.query('TRUNCATE TABLE reservas_salas CASCADE');
    await pool.query('TRUNCATE TABLE salas_eventos CASCADE');
    await pool.query('TRUNCATE TABLE pedidos_restaurantes CASCADE');
    await pool.query('TRUNCATE TABLE itens_cardapio CASCADE');
    await pool.query('TRUNCATE TABLE reservas_restaurantes CASCADE');
    await pool.query('TRUNCATE TABLE restaurantes CASCADE');
    await pool.query('TRUNCATE TABLE reservas_alojamentos CASCADE');
    await pool.query('TRUNCATE TABLE alojamentos CASCADE');
    await pool.query('TRUNCATE TABLE reservas_turisticas CASCADE');
    await pool.query('TRUNCATE TABLE pacotes_turisticos CASCADE');
    await pool.query('TRUNCATE TABLE viagens CASCADE');
    await pool.query('TRUNCATE TABLE imoveis CASCADE');
    await pool.query('TRUNCATE TABLE transacoes CASCADE');
    await pool.query('TRUNCATE TABLE notificacoes CASCADE');
    await pool.query('TRUNCATE TABLE avaliacoes CASCADE');
    await pool.query('TRUNCATE TABLE auditoria_usuarios CASCADE');
    await pool.query('TRUNCATE TABLE verificacao_telefone CASCADE');
    await pool.query('TRUNCATE TABLE usuarios CASCADE');

    // Reabilitar triggers
    await pool.query('SET session_replication_role = DEFAULT');

    await pool.query('COMMIT');

    // Limpar Redis
    await redis.flushdb();

  } catch (error) {
    await pool.query('ROLLBACK');
    console.error('❌ Erro ao limpar dados de teste:', error.message);
    throw error;
  }
});

// Hook executado APÓS todos os testes
afterAll(async () => {
  console.log('\n🧹 Limpando recursos de teste...\n');

  try {
    // Fechar conexões
    await pool.end();
    console.log('✅ Pool PostgreSQL fechado');

    await redis.quit();
    console.log('✅ Conexão Redis fechada');

    console.log('\n✅ Limpeza concluída\n');
  } catch (error) {
    console.error('❌ Erro ao limpar recursos:', error.message);
  }
});

// Helpers globais para testes
global.testHelpers = {
  /**
   * Cria usuário de teste
   */
  async criarUsuarioTeste(dados = {}) {
    const bcrypt = require('bcryptjs');
    const senha_hash = await bcrypt.hash(dados.senha || 'Teste123456', 12);

    const query = `
      INSERT INTO usuarios 
      (nome, email, telefone, senha_hash, tipo_usuario, kyc_status, selo_verificado, status)
      VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
      RETURNING id, nome, email, tipo_usuario
    `;

    const result = await pool.query(query, [
      dados.nome || 'Usuário Teste',
      dados.email || `teste${Date.now()}@example.com`,
      dados.telefone || `92${Math.floor(1000000 + Math.random() * 9000000)}`,
      senha_hash,
      dados.tipo_usuario || 'comprador',
      dados.kyc_status || 'aprovado',
      dados.selo_verificado !== undefined ? dados.selo_verificado : true,
      dados.status || 'ativo'
    ]);

    return result.rows[0];
  },

  /**
   * Gera token JWT para testes
   */
  gerarTokenTeste(usuario) {
    const jwt = require('jsonwebtoken');
    return jwt.sign(
      {
        id: usuario.id,
        email: usuario.email,
        tipo: usuario.tipo_usuario
      },
      process.env.JWT_SECRET,
      { expiresIn: '7d' }
    );
  },

  /**
   * Cria imóvel de teste
   */
  async criarImovelTeste(vendedorId, dados = {}) {
    const query = `
      INSERT INTO imoveis 
      (vendedor_id, titulo, tipo, finalidade, preco, localizacao, latitude, longitude, quartos)
      VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
      RETURNING id, titulo, preco
    `;

    const result = await pool.query(query, [
      vendedorId,
      dados.titulo || 'Casa de Teste',
      dados.tipo || 'casa',
      dados.finalidade || 'venda',
      dados.preco || 50000000,
      dados.localizacao || 'Luanda, Angola',
      dados.latitude || -8.8383,
      dados.longitude || 13.2344,
      dados.quartos || 3
    ]);

    return result.rows[0];
  },

  /**
   * Cria transação de teste
   */
  async criarTransacaoTeste(usuarioId, dados = {}) {
    const query = `
      INSERT INTO transacoes 
      (usuario_id, tipo_transacao, modulo, item_id, valor, metodo_pagamento, status_transacao)
      VALUES ($1, $2, $3, $4, $5, $6, $7)
      RETURNING id, valor, status_transacao
    `;

    const result = await pool.query(query, [
      usuarioId,
      dados.tipo_transacao || 'pagamento',
      dados.modulo || 'imoveis',
      dados.item_id || '00000000-0000-0000-0000-000000000000',
      dados.valor || 100000,
      dados.metodo_pagamento || 'multicaixa',
      dados.status_transacao || 'pendente'
    ]);

    return result.rows[0];
  }
};

module.exports = {
  pool,
  redis,
  testHelpers: global.testHelpers
};
