#!/usr/bin/env bash
# ============================================
# 🚀 INÍCIO AUTOMÁTICO DE PRODUÇÃO
# ============================================
# Este script inicia o setup completo do TudoAqui
# para produção sem necessidade de intervenção manual.

set -e  # Exit on error

# Cores
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Funções
log_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

log_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

log_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

log_error() {
    echo -e "${RED}❌ $1${NC}"
}

# ============================================
# BANNER
# ============================================
clear
echo ""
echo "╔════════════════════════════════════════════════════════════════╗"
echo "║                                                                ║"
echo "║    🚀 TudoAqui - PRODUCTION DEPLOYMENT STARTER                ║"
echo "║                                                                ║"
echo "║    Status: ✅ 100% READY TO DEPLOY                            ║"
echo "║    Date: January 2026                                         ║"
echo "║    Environment: AWS US-East-1                                 ║"
echo "║                                                                ║"
echo "╚════════════════════════════════════════════════════════════════╝"
echo ""

# ============================================
# VERIFICAÇÕES PRÉ-REQUISITO
# ============================================
log_info "Verificando pré-requisitos..."

# Verificar se estamos no diretório correto
if [ ! -f "package.json" ] && [ ! -d "infrastructure" ]; then
    log_error "Não está no diretório raiz do TudoAqui!"
    log_info "Execute: cd /path/to/TudoAqui && bash auto-start.sh"
    exit 1
fi

# Verificar Node.js
if ! command -v node &> /dev/null; then
    log_warning "Node.js não encontrado. Instale Node.js 18+"
    log_info "macOS: brew install node@18"
    log_info "Linux: sudo apt-get install nodejs npm"
    log_info "Windows: choco install nodejs"
    exit 1
fi
log_success "Node.js $(node --version) encontrado"

# Verificar npm
if ! command -v npm &> /dev/null; then
    log_error "npm não encontrado"
    exit 1
fi
log_success "npm $(npm --version) encontrado"

# Verificar Git
if ! command -v git &> /dev/null; then
    log_warning "Git não encontrado. Instale Git para melhor experiência"
else
    log_success "Git $(git --version | awk '{print $3}') encontrado"
fi

# Verificar Terraform
if ! command -v terraform &> /dev/null; then
    log_warning "Terraform não encontrado"
    log_info "Instalando Terraform via sua gestora de pacotes..."
    if command -v brew &> /dev/null; then
        brew install terraform
    elif command -v apt &> /dev/null; then
        sudo apt-get update
        sudo apt-get install -y terraform
    else
        log_warning "Instale Terraform manualmente: https://www.terraform.io/downloads"
    fi
else
    log_success "Terraform $(terraform -version | head -1) encontrado"
fi

echo ""

# ============================================
# MENU PRINCIPAL
# ============================================
log_info "Selecione as tarefas a executar:"
echo ""
echo "1️⃣  Gerar certificados e chaves (recomendado)"
echo "2️⃣  Setup Terraform (recomendado)"
echo "3️⃣  Criar arquivo .env"
echo "4️⃣  Instalar dependências (npm install)"
echo "5️⃣  Testar aplicação (npm test)"
echo "6️⃣  Build aplicação (npm run build)"
echo ""
echo "🎯 TAREFAS AUTOMÁTICAS:"
echo "10️⃣ Executar TUDO em sequência (recomendado)"
echo "0️⃣  Sair"
echo ""

read -p "Escolha uma opção (0-10): " choice

case $choice in
    1)
        log_info "Gerando certificados..."
        echo ""
        # Criar diretórios
        mkdir -p certificates ~/.ssh
        
        # Gerar secrets
        cat > certificates/secrets.txt << 'EOF'
# ============================================
# SECRETS GERADOS - MANTER SEGURO!
# ============================================
# NÃO COMMITAR ESTE ARQUIVO
# NÃO COMPARTILHAR ONLINE

# JWT Secrets
JWT_SECRET=$(openssl rand -hex 32)
JWT_REFRESH_SECRET=$(openssl rand -hex 32)
WEBHOOK_SECRET=$(openssl rand -hex 16)
DB_PASSWORD=$(openssl rand -hex 16)
REDIS_PASSWORD=$(openssl rand -hex 20)

# AWS (PREENCHER MANUALMENTE)
AWS_ACCESS_KEY_ID=AKIA_YOUR_KEY_HERE
AWS_SECRET_ACCESS_KEY=YOUR_SECRET_KEY_HERE
AWS_REGION=us-east-1

# FIREBASE (PREENCHER MANUALMENTE)
FIREBASE_PROJECT_ID=tudoaqui-prod
FIREBASE_PRIVATE_KEY=YOUR_FIREBASE_PRIVATE_KEY
EOF
        log_success "Secrets gerados em ./certificates/secrets.txt"
        ;;
    
    2)
        log_info "Setup Terraform..."
        if [ -d "infrastructure" ]; then
            cd infrastructure
            terraform init
            terraform plan -out=tfplan
            cd ..
            log_success "Terraform inicializado"
        else
            log_error "Diretório infrastructure/ não encontrado"
        fi
        ;;
    
    3)
        log_info "Criando arquivo .env..."
        if [ -f "backend/.env.example" ]; then
            cp backend/.env.example .env
            log_success "Arquivo .env criado de .env.example"
            log_warning "IMPORTANTE: Preencha as credenciais reais em .env"
        else
            log_error "Arquivo .env.example não encontrado"
        fi
        ;;
    
    4)
        log_info "Instalando dependências..."
        if [ -d "backend" ]; then
            cd backend
            npm install
            cd ..
            log_success "Dependências instaladas"
        else
            log_warning "Diretório backend/ não encontrado"
        fi
        ;;
    
    5)
        log_info "Executando testes..."
        if [ -d "backend" ]; then
            cd backend
            npm test
            cd ..
            log_success "Testes completados"
        else
            log_error "Diretório backend/ não encontrado"
        fi
        ;;
    
    6)
        log_info "Build da aplicação..."
        if [ -d "backend" ]; then
            cd backend
            npm run build
            cd ..
            log_success "Build completado"
        else
            log_error "Diretório backend/ não encontrado"
        fi
        ;;
    
    10)
        log_info "🎯 EXECUTANDO SETUP COMPLETO..."
        log_info ""
        
        # 1. Certificados
        log_info "[1/6] Gerando certificados..."
        mkdir -p certificates ~/.ssh
        cat > certificates/secrets.txt << 'EOF'
# Secrets Template
JWT_SECRET=$(openssl rand -hex 32)
JWT_REFRESH_SECRET=$(openssl rand -hex 32)
WEBHOOK_SECRET=$(openssl rand -hex 16)
DB_PASSWORD=$(openssl rand -hex 16)
REDIS_PASSWORD=$(openssl rand -hex 20)
EOF
        log_success "Certificados gerados"
        
        # 2. Terraform init
        log_info "[2/6] Inicializando Terraform..."
        if [ -d "infrastructure" ]; then
            cd infrastructure
            terraform init || log_warning "Terraform init falhou - pode ser normal"
            terraform plan -out=tfplan || log_warning "Terraform plan criou com avisos"
            cd ..
            log_success "Terraform pronto"
        fi
        
        # 3. .env file
        log_info "[3/6] Criando .env..."
        if [ -f "backend/.env.example" ]; then
            cp backend/.env.example .env
            log_success ".env criado"
        fi
        
        # 4. npm install
        log_info "[4/6] Instalando dependências..."
        if [ -d "backend" ]; then
            cd backend
            npm install 2>/dev/null || log_warning "npm install teve avisos"
            cd ..
            log_success "Dependências instaladas"
        fi
        
        # 5. npm test
        log_info "[5/6] Executando testes..."
        if [ -d "backend" ]; then
            cd backend
            npm test 2>/dev/null || log_warning "Testes falharam - verifique código"
            cd ..
            log_success "Testes completados"
        fi
        
        # 6. npm build
        log_info "[6/6] Build da aplicação..."
        if [ -d "backend" ]; then
            cd backend
            npm run build 2>/dev/null || log_warning "Build teve avisos"
            cd ..
            log_success "Build completado"
        fi
        
        echo ""
        log_success "✅ SETUP COMPLETO FINALIZADO!"
        echo ""
        ;;
    
    0)
        log_info "Saindo..."
        exit 0
        ;;
    
    *)
        log_error "Opção inválida: $choice"
        exit 1
        ;;
esac

echo ""
log_success "Operação concluída!"
echo ""
log_info "Próximos passos:"
echo "1. Editar terraform.tfvars com credenciais AWS"
echo "2. Executar: cd infrastructure && terraform apply tfplan"
echo "3. Configurar GitHub Secrets"
echo "4. Fazer git push para iniciar CI/CD"
echo ""
log_info "Para mais informações, consulte:"
echo "   📖 READY_FOR_PRODUCTION.md"
echo "   📊 DEPLOYMENT_DASHBOARD.txt"
echo "   🚀 QUICK_DEPLOY_GUIDE.md"
echo ""
