# 🚀 TudoAqui - Super App Backend API
**5 Módulos Principais Implementados**

## 📋 Endpoints Implementados

### **Módulo 1: Core de Identidade e Segurança**

#### Autenticação
```bash
# Registar utilizador
POST /api/v1/auth/register
{
  "email": "vendedor@tudoaqui.ao",
  "senha": "Senha123!",
  "nome": "João Silva",
  "telefone": "+244923456789",
  "tipo": "vendedor"
}

# Login
POST /api/v1/auth/login
{
  "email": "vendedor@tudoaqui.ao",
  "senha": "Senha123!"
}
```

#### KYC (Verificação)
```bash
# Upload de BI (com OCR automático)
POST /api/v1/kyc/upload-bi
Content-Type: multipart/form-data
Authorization: Bearer {token}
file: [imagem do BI]

# Upload de Selfie (Liveness Detection)
POST /api/v1/kyc/upload-selfie
Authorization: Bearer {token}
file: [selfie]

# Status KYC
GET /api/v1/kyc/status/:userId
Authorization: Bearer {token}

# Analisar com IA (AWS Rekognition)
POST /api/v1/kyc/analyze
Authorization: Bearer {token}
{ "userId": "uuid" }
```

---

### **Módulo 2: Marketplace de Imóveis e Veículos**

#### Criar Anúncio (com marca d'água automática)
```bash
POST /api/v1/marketplace/anuncios
Content-Type: multipart/form-data
Authorization: Bearer {token}

# Campos:
categoria: "imoveis" | "veiculos"
titulo: "Apartamento T3 Kilamba"
descricao: "Apartamento mobilado com piscina"
preco: 150000000  # Kz
provincia: "Luanda"
municipio: "Belas"
bairro: "Kilamba"
latitude: -8.9667
longitude: 13.2333

# Detalhes Imóvel (se categoria = imoveis):
tipoImovel: "apartamento"
tipoNegocio: "venda"
quartos: 3
banheiros: 2
garagem: 2
area: 120.5
mobilado: true
piscina: true

# Detalhes Veículo (se categoria = veiculos):
tipoVeiculo: "carro"
marca: "Toyota"
modelo: "Corolla"
ano: 2022
combustivel: "gasolina"
transmissao: "automatica"
quilometragem: 15000
cor: "Preto"

# Upload de até 10 fotos:
fotos: [file1.jpg, file2.jpg, ...]
```

#### Listar Anúncios com Filtros
```bash
# Busca geral
GET /api/v1/marketplace/anuncios?categoria=imoveis&provincia=Luanda

# Filtro "Perto de Mim" (PostGIS)
GET /api/v1/marketplace/anuncios?latitude=-8.8383&longitude=13.2344&raioKm=5

# Filtro por preço
GET /api/v1/marketplace/anuncios?precoMin=50000000&precoMax=200000000

# Busca por texto
GET /api/v1/marketplace/anuncios?busca=Kilamba

# Busca específica por localidade
GET /api/v1/marketplace/anuncios/localidade/Luanda/Belas
```

#### Detalhes do Anúncio
```bash
GET /api/v1/marketplace/anuncios/:id
# Retorna anúncio completo + detalhes de imóvel/veículo + vendedor
```

---

### **Módulo 3: Integração Logística (Tupuca)**

#### Calcular Estimativa de Entrega
```bash
POST /api/v1/logistica/estimativa
{
  "restauranteLatitude": -8.8383,
  "restauranteLongitude": 13.2344,
  "clienteLatitude": -8.8500,
  "clienteLongitude": 13.2500
}

# Resposta:
{
  "distanciaKm": 2.3,
  "tempoEstimadoMinutos": 21,
  "precoEntrega": 845,  # Kz
  "disponivel": true
}
```

#### Criar Pedido de Entrega
```bash
POST /api/v1/logistica/pedido
Authorization: Bearer {token}
{
  "pedidoId": "uuid-do-pedido",
  "origem": {
    "latitude": -8.8383,
    "longitude": 13.2344,
    "endereco": "Restaurante XYZ, Talatona"
  },
  "destino": {
    "latitude": -8.8500,
    "longitude": 13.2500,
    "endereco": "Casa do Cliente, Kilamba"
  }
}

# Resposta:
{
  "success": true,
  "trackingId": "TUPUCA-1769032403",
  "message": "Entregador a caminho"
}
```

#### Rastrear Entrega em Tempo Real
```bash
GET /api/v1/logistica/rastrear/:trackingId

# Resposta (atualizada via Redis):
{
  "trackingId": "TUPUCA-1769032403",
  "status": "em_rota",
  "latitude": -8.8420,
  "longitude": 13.2380,
  "tempoRestante": 12  # minutos
}
```

---

### **Módulo 4: Pagamentos (Multicaixa Express + GPO)**

#### Iniciar Pagamento
```bash
POST /api/v1/pagamentos/iniciar
Authorization: Bearer {token}
{
  "valor": 5000,  # Kz
  "telefone": "+244923456789",
  "descricao": "Destaque de anúncio - 7 dias",
  "referencia": "anuncio-uuid"
}

# Resposta:
{
  "success": true,
  "transacaoId": "MCX-1769032403",
  "mensagem": "Pagamento iniciado. Confirme no seu Multicaixa Express.",
  "status": "pendente",
  "valor": 5000
}
```

#### Verificar Status de Pagamento
```bash
GET /api/v1/pagamentos/status/:transacaoId
Authorization: Bearer {token}

# Resposta:
{
  "transacaoId": "MCX-1769032403",
  "status": "aprovado",  # pendente | aprovado | rejeitado
  "valor": 5000,
  "dataHora": "2026-01-22T10:30:00Z"
}
```

#### Webhook Multicaixa (recebe confirmação)
```bash
POST /api/v1/pagamentos/webhook
# Sem autenticação - valida assinatura internamente
{
  "transacao_id": "MCX-1769032403",
  "status": "aprovado",
  "valor": 5000,
  "referencia": "anuncio-uuid",
  "timestamp": "2026-01-22T10:30:00Z"
}
```

#### Calcular Taxa da Plataforma
```bash
GET /api/v1/pagamentos/taxa/:valor

# Exemplo: GET /api/v1/pagamentos/taxa/100000
{
  "valorOriginal": 100000,
  "taxa": 2500,  # 2.5%
  "valorComTaxa": 102500
}
```

---

### **Módulo 5: Serviços e Turismo**

#### Criar Reserva (Restaurante, Hotel, Guia)
```bash
POST /api/v1/servicos/reservas
Authorization: Bearer {token}
{
  "tipoServico": "restaurante",  # restaurante | guia_turistico | hotel
  "servicoNome": "Restaurante Tamariz",
  "dataReserva": "2026-01-25T19:00:00Z",
  "horario": "19:00",
  "numeroPessoas": 4,
  "observacoes": "Mesa perto da janela",
  "detalhesAdicionais": {
    "preferencias": "Comida vegetariana"
  }
}

# Resposta:
{
  "message": "Reserva criada com sucesso",
  "reserva": {
    "id": "uuid",
    "servicoNome": "Restaurante Tamariz",
    "dataReserva": "2026-01-25T19:00:00Z",
    "status": "pendente"
  }
}
```

#### Listar Reservas do Utilizador
```bash
GET /api/v1/servicos/reservas
Authorization: Bearer {token}

# Resposta:
{
  "total": 2,
  "reservas": [
    {
      "id": "uuid",
      "tipoServico": "restaurante",
      "servicoNome": "Restaurante Tamariz",
      "dataReserva": "2026-01-25T19:00:00Z",
      "status": "confirmada",
      "numeroPessoas": 4
    },
    ...
  ]
}
```

#### Solicitar Cotação de Seguro Automóvel
```bash
POST /api/v1/servicos/seguros/cotacao
Authorization: Bearer {token}
{
  "tipo": "automovel",  # automovel | residencial | vida
  "nomeCompleto": "João Silva",
  "telefone": "+244923456789",
  "email": "joao@email.com",
  
  # Para seguro automóvel:
  "veiculoMarca": "Toyota",
  "veiculoModelo": "Corolla",
  "veiculoAno": 2022,
  "veiculoChaveta": "LD-12-34-AB"
}

# Resposta (cotação instantânea):
{
  "message": "Cotação gerada com sucesso",
  "cotacao": {
    "id": "uuid",
    "tipo": "automovel",
    "valorMensal": 23000,  # Kz
    "valorAnual": 276000,
    "seguradoraParceira": "ENSA Seguros"
  }
}
```

#### Obter Detalhes de Cotação
```bash
GET /api/v1/servicos/seguros/cotacao/:id
Authorization: Bearer {token}

# Resposta completa da cotação
```

---

## 🔑 Autenticação

Todos os endpoints protegidos requerem token JWT no header:
```
Authorization: Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...
```

Token obtido após login ou registo em `/api/v1/auth/register` ou `/api/v1/auth/login`.

---

## ✨ Funcionalidades Especiais

### 1. **Marca d'água Automática (Módulo 2)**
Todas as fotos de anúncios recebem marca d'água "TudoAqui" no canto inferior direito (Sharp + SVG).

### 2. **Filtro "Perto de Mim" (Módulo 2)**
Usa PostGIS (geography type) para buscar anúncios num raio específico.

### 3. **Cálculo de Distância Haversine (Módulo 3)**
Calcula distância real entre restaurante e cliente para estimativa de entrega.

### 4. **Cotação Instantânea de Seguros (Módulo 5)**
Algoritmo simula cotação com base no ano do veículo (em produção, integrar API de seguradora).

### 5. **Webhooks Multicaixa (Módulo 4)**
Backend recebe notificação assíncrona quando pagamento é confirmado.

---

## 🛠️ Tecnologias Utilizadas

- **NestJS 10.3** + TypeScript
- **PostgreSQL 16** + PostGIS (geolocalização)
- **Redis 7** (tracking GPS em tempo real)
- **TypeORM** (autoLoadEntities)
- **Sharp** (processamento de imagens + marca d'água)
- **JWT** (autenticação stateless)
- **AWS Rekognition** (OCR + Facial Recognition)
- **Firebase Admin** (notificações push - TODO)

---

## 📦 Estrutura de Entidades

### User (Auth)
- id, email, senha, nome, telefone, tipo, seloVerificado, kycStatus

### Anuncio (Marketplace)
- id, vendedorId, categoria, titulo, descricao, preco, provincia, municipio, localizacao (Point), fotos, status, destaque

### DetalhesImovel
- id, anuncioId, tipo, tipoNegocio, quartos, banheiros, garagem, area, mobilado, piscina

### DetalhesVeiculo
- id, anuncioId, tipo, marca, modelo, ano, combustivel, transmissao, quilometragem

### Reserva (Serviços)
- id, clienteId, tipoServico, servicoNome, dataReserva, horario, numeroPessoas, status

### CotacaoSeguro
- id, userId, tipo, veiculoMarca, veiculoModelo, veiculoAno, valorMensal, valorAnual

---

## 🚀 Status de Implementação

✅ **Módulo 1**: Auth + KYC (100%)
✅ **Módulo 2**: Marketplace com marca d'água (100%)
✅ **Módulo 3**: Logística Tupuca com Haversine (100% mock, pronto para API real)
✅ **Módulo 4**: Pagamentos Multicaixa + webhook (100% mock)
✅ **Módulo 5**: Serviços, Turismo, Seguros (100%)

**Next Steps:**
- Integrar APIs reais (Tupuca, Multicaixa, ENSA Seguros)
- Implementar Firebase Cloud Messaging
- Criar interface Flutter Mobile
- Testes E2E com Postman

---

**Backend rodando em:** `http://localhost:3000/api/v1`
**Documentação:** Este ficheiro (API_ENDPOINTS.md)
