"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.KycService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const vision_1 = require("@google-cloud/vision");
const node_fs_1 = require("node:fs");
const path = __importStar(require("node:path"));
const sharp_1 = __importDefault(require("sharp"));
const user_entity_1 = require("../auth/entities/user.entity");
let KycService = class KycService {
    constructor(userRepository) {
        this.userRepository = userRepository;
        try {
            this.visionClient = new vision_1.ImageAnnotatorClient();
        }
        catch (err) {
            console.warn('[KYC] Google Cloud Vision não configurado - Usando modo MOCK para desenvolvimento');
            console.warn('[KYC] Para produção: Configure GOOGLE_APPLICATION_CREDENTIALS no .env');
        }
    }
    async processBI(userId, file) {
        if (!file?.buffer) {
            throw new common_1.BadRequestException('Ficheiro do BI é obrigatório');
        }
        const user = await this.userRepository.findOne({ where: { id: userId } });
        if (!user) {
            throw new common_1.BadRequestException('Utilizador não encontrado');
        }
        const outputDir = path.join(process.cwd(), 'uploads', 'bi');
        await this.ensureDir(outputDir);
        const filename = `${userId}_${Date.now()}.jpg`;
        const outputPath = path.join(outputDir, filename);
        const watermarked = await this.addWatermark(file.buffer, 'TudoAqui - KYC');
        await node_fs_1.promises.writeFile(outputPath, watermarked);
        let dadosBI = {};
        try {
            dadosBI = await this.extractBIData(watermarked);
        }
        catch (err) {
            console.error('[KYC] Erro no OCR do BI:', err.message);
        }
        user.fotoBI = outputPath;
        if (dadosBI.numeroBI) {
            user.biNumero = dadosBI.numeroBI;
        }
        await this.userRepository.save(user);
        return {
            message: 'Foto do BI carregada com sucesso',
            url: outputPath,
            dadosExtraidos: dadosBI,
        };
    }
    async processSelfie(userId, file) {
        if (!file?.buffer) {
            throw new common_1.BadRequestException('Selfie é obrigatória');
        }
        const user = await this.userRepository.findOne({ where: { id: userId } });
        if (!user) {
            throw new common_1.BadRequestException('Utilizador não encontrado');
        }
        const outputDir = path.join(process.cwd(), 'uploads', 'selfie');
        await this.ensureDir(outputDir);
        const filename = `${userId}_${Date.now()}.jpg`;
        const outputPath = path.join(outputDir, filename);
        await node_fs_1.promises.writeFile(outputPath, file.buffer);
        let livenessOk = false;
        try {
            livenessOk = await this.verifyLiveness(file.buffer);
        }
        catch (err) {
            console.error('[KYC] Erro no Liveness Detection:', err.message);
        }
        user.fotoSelfie = outputPath;
        await this.userRepository.save(user);
        return {
            message: 'Selfie carregada com sucesso',
            url: outputPath,
            livenessDetected: livenessOk,
        };
    }
    async analyzeWithAI(userId) {
        const user = await this.userRepository.findOne({ where: { id: userId } });
        if (!user) {
            throw new common_1.BadRequestException('Utilizador não encontrado');
        }
        if (!user.fotoBI || !user.fotoSelfie) {
            throw new common_1.BadRequestException('É necessário enviar BI e selfie antes de analisar');
        }
        const validacoes = {
            nomeCoincide: false,
            rostosCorrespondem: false,
            aprovado: false,
        };
        if (user.biNumero) {
            const biBuffer = await node_fs_1.promises.readFile(user.fotoBI);
            const dadosBI = await this.extractBIData(biBuffer);
            if (dadosBI.nome) {
                validacoes.nomeCoincide = this.compareNames(user.nome, dadosBI.nome);
            }
        }
        try {
            const biBuffer = await node_fs_1.promises.readFile(user.fotoBI);
            const selfieBuffer = await node_fs_1.promises.readFile(user.fotoSelfie);
            const facesMatch = await this.compareFaces(biBuffer, selfieBuffer);
            validacoes.rostosCorrespondem = facesMatch;
        }
        catch (err) {
            console.error('[KYC] Erro na comparação de rostos:', err.message);
        }
        if (validacoes.nomeCoincide && validacoes.rostosCorrespondem) {
            user.kycStatus = 'aprovado';
            user.seloVerificado = true;
            validacoes.aprovado = true;
        }
        else {
            user.kycStatus = 'rejeitado';
            validacoes.aprovado = false;
        }
        await this.userRepository.save(user);
        return {
            message: validacoes.aprovado
                ? '✅ Verificação aprovada! Selo verde atribuído.'
                : '❌ Verificação rejeitada. Dados não coincidem.',
            validacoes,
            seloVerificado: user.seloVerificado,
        };
    }
    async getKycStatus(userId) {
        const user = await this.userRepository.findOne({
            where: { id: userId },
            select: ['id', 'nome', 'email', 'kycStatus', 'seloVerificado', 'biNumero', 'fotoBI', 'fotoSelfie'],
        });
        if (!user) {
            throw new common_1.BadRequestException('Utilizador não encontrado');
        }
        return {
            kycStatus: user.kycStatus,
            seloVerificado: user.seloVerificado,
            biEnviado: !!user.fotoBI,
            selfieEnviada: !!user.fotoSelfie,
            numeroBI: user.biNumero || null,
        };
    }
    async extractBIData(imageBuffer) {
        if (!this.visionClient) {
            console.log('[KYC] Modo MOCK: Retornando dados simulados do BI');
            return {
                numeroBI: '123456789AO01',
                nome: 'João Silva Mock',
            };
        }
        try {
            const [result] = await this.visionClient.textDetection({
                image: { content: imageBuffer },
            });
            const detections = result.textAnnotations || [];
            if (detections.length === 0) {
                return {};
            }
            const textoCompleto = detections[0]?.description || '';
            const biRegex = /(\d{9}[A-Z]{2}\d{2})/;
            const matchBI = textoCompleto.match(biRegex);
            let nome;
            const linhas = textoCompleto.split('\n');
            for (let i = 0; i < linhas.length; i++) {
                if (/nome/i.test(linhas[i]) && linhas[i + 1]) {
                    nome = linhas[i + 1].trim();
                    break;
                }
            }
            return {
                numeroBI: matchBI ? matchBI[1] : undefined,
                nome: nome,
            };
        }
        catch (err) {
            console.error('[KYC] Erro no OCR:', err);
            return {};
        }
    }
    async verifyLiveness(imageBuffer) {
        if (!this.visionClient) {
            console.log('[KYC] Modo MOCK: Liveness aprovado automaticamente');
            return true;
        }
        try {
            const [result] = await this.visionClient.faceDetection({
                image: { content: imageBuffer },
            });
            const faces = result.faceAnnotations || [];
            if (faces.length !== 1) {
                return false;
            }
            const face = faces[0];
            const confidence = face.detectionConfidence || 0;
            return confidence > 0.8;
        }
        catch (err) {
            console.error('[KYC] Erro no Liveness:', err);
            return false;
        }
    }
    async compareFaces(biBuffer, selfieBuffer) {
        if (!this.visionClient) {
            console.log('[KYC] Modo MOCK: Rostos correspondem automaticamente');
            return true;
        }
        try {
            const [biResult] = await this.visionClient.faceDetection({ image: { content: biBuffer } });
            const [selfieResult] = await this.visionClient.faceDetection({ image: { content: selfieBuffer } });
            const biFaces = biResult.faceAnnotations || [];
            const selfieFaces = selfieResult.faceAnnotations || [];
            if (biFaces.length !== 1 || selfieFaces.length !== 1) {
                return false;
            }
            const biLandmarks = biFaces[0].landmarks || [];
            const selfieLandmarks = selfieFaces[0].landmarks || [];
            const biHasLandmarks = biLandmarks.length > 5;
            const selfieHasLandmarks = selfieLandmarks.length > 5;
            return biHasLandmarks && selfieHasLandmarks;
        }
        catch (err) {
            console.error('[KYC] Erro na comparação de rostos:', err);
            return false;
        }
    }
    compareNames(nomeConta, nomeBI) {
        const normalize = (str) => str
            .toLowerCase()
            .normalize('NFD')
            .replace(/[\u0300-\u036f]/g, '')
            .replace(/[^a-z\s]/g, '')
            .trim();
        const nomeContaNorm = normalize(nomeConta);
        const nomeBINorm = normalize(nomeBI);
        return (nomeContaNorm.includes(nomeBINorm) ||
            nomeBINorm.includes(nomeContaNorm) ||
            nomeContaNorm === nomeBINorm);
    }
    async addWatermark(imageBuffer, text) {
        const watermarkSvg = Buffer.from(`
      <svg width="350" height="60">
        <style>
          .texto {
            font-family: Arial, sans-serif;
            font-size: 24px;
            font-weight: bold;
            fill: rgba(255, 255, 255, 0.8);
            stroke: rgba(0, 0, 0, 0.5);
            stroke-width: 1px;
          }
        </style>
        <text x="10" y="40" class="texto">${text}</text>
      </svg>
    `);
        return (0, sharp_1.default)(imageBuffer)
            .resize(1200, 1600, { fit: 'inside' })
            .composite([
            {
                input: watermarkSvg,
                gravity: 'southeast',
            },
        ])
            .jpeg({ quality: 85 })
            .toBuffer();
    }
    async ensureDir(dirPath) {
        try {
            await node_fs_1.promises.access(dirPath);
        }
        catch {
            await node_fs_1.promises.mkdir(dirPath, { recursive: true });
        }
    }
    async processAlvara(userId, file) {
        if (!file?.buffer) {
            throw new common_1.BadRequestException('Ficheiro do Alvará é obrigatório');
        }
        const user = await this.userRepository.findOne({ where: { id: userId } });
        if (!user) {
            throw new common_1.BadRequestException('Utilizador não encontrado');
        }
        const outputDir = path.join(process.cwd(), 'uploads', 'alvara');
        await this.ensureDir(outputDir);
        const filename = `${userId}_${Date.now()}.pdf`;
        const outputPath = path.join(outputDir, filename);
        await node_fs_1.promises.writeFile(outputPath, file.buffer);
        user.alvaraUrl = outputPath;
        await this.userRepository.save(user);
        return {
            message: 'Alvará carregado com sucesso',
            url: outputPath,
        };
    }
};
exports.KycService = KycService;
exports.KycService = KycService = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(user_entity_1.User)),
    __metadata("design:paramtypes", [typeorm_2.Repository])
], KycService);
//# sourceMappingURL=kyc.service.js.map