'use strict';

var middlewareHostHeader = require('@aws-sdk/middleware-host-header');
var middlewareLogger = require('@aws-sdk/middleware-logger');
var middlewareRecursionDetection = require('@aws-sdk/middleware-recursion-detection');
var middlewareUserAgent = require('@aws-sdk/middleware-user-agent');
var configResolver = require('@smithy/config-resolver');
var core = require('@smithy/core');
var schema = require('@smithy/core/schema');
var middlewareContentLength = require('@smithy/middleware-content-length');
var middlewareEndpoint = require('@smithy/middleware-endpoint');
var middlewareRetry = require('@smithy/middleware-retry');
var smithyClient = require('@smithy/smithy-client');
var httpAuthSchemeProvider = require('./auth/httpAuthSchemeProvider');
var runtimeConfig = require('./runtimeConfig');
var regionConfigResolver = require('@aws-sdk/region-config-resolver');
var protocolHttp = require('@smithy/protocol-http');
var utilWaiter = require('@smithy/util-waiter');

const resolveClientEndpointParameters = (options) => {
    return Object.assign(options, {
        useDualstackEndpoint: options.useDualstackEndpoint ?? false,
        useFipsEndpoint: options.useFipsEndpoint ?? false,
        defaultSigningName: "rekognition",
    });
};
const commonParams = {
    UseFIPS: { type: "builtInParams", name: "useFipsEndpoint" },
    Endpoint: { type: "builtInParams", name: "endpoint" },
    Region: { type: "builtInParams", name: "region" },
    UseDualStack: { type: "builtInParams", name: "useDualstackEndpoint" },
};

const getHttpAuthExtensionConfiguration = (runtimeConfig) => {
    const _httpAuthSchemes = runtimeConfig.httpAuthSchemes;
    let _httpAuthSchemeProvider = runtimeConfig.httpAuthSchemeProvider;
    let _credentials = runtimeConfig.credentials;
    return {
        setHttpAuthScheme(httpAuthScheme) {
            const index = _httpAuthSchemes.findIndex((scheme) => scheme.schemeId === httpAuthScheme.schemeId);
            if (index === -1) {
                _httpAuthSchemes.push(httpAuthScheme);
            }
            else {
                _httpAuthSchemes.splice(index, 1, httpAuthScheme);
            }
        },
        httpAuthSchemes() {
            return _httpAuthSchemes;
        },
        setHttpAuthSchemeProvider(httpAuthSchemeProvider) {
            _httpAuthSchemeProvider = httpAuthSchemeProvider;
        },
        httpAuthSchemeProvider() {
            return _httpAuthSchemeProvider;
        },
        setCredentials(credentials) {
            _credentials = credentials;
        },
        credentials() {
            return _credentials;
        },
    };
};
const resolveHttpAuthRuntimeConfig = (config) => {
    return {
        httpAuthSchemes: config.httpAuthSchemes(),
        httpAuthSchemeProvider: config.httpAuthSchemeProvider(),
        credentials: config.credentials(),
    };
};

const resolveRuntimeExtensions = (runtimeConfig, extensions) => {
    const extensionConfiguration = Object.assign(regionConfigResolver.getAwsRegionExtensionConfiguration(runtimeConfig), smithyClient.getDefaultExtensionConfiguration(runtimeConfig), protocolHttp.getHttpHandlerExtensionConfiguration(runtimeConfig), getHttpAuthExtensionConfiguration(runtimeConfig));
    extensions.forEach((extension) => extension.configure(extensionConfiguration));
    return Object.assign(runtimeConfig, regionConfigResolver.resolveAwsRegionExtensionConfiguration(extensionConfiguration), smithyClient.resolveDefaultRuntimeConfig(extensionConfiguration), protocolHttp.resolveHttpHandlerRuntimeConfig(extensionConfiguration), resolveHttpAuthRuntimeConfig(extensionConfiguration));
};

class RekognitionClient extends smithyClient.Client {
    config;
    constructor(...[configuration]) {
        const _config_0 = runtimeConfig.getRuntimeConfig(configuration || {});
        super(_config_0);
        this.initConfig = _config_0;
        const _config_1 = resolveClientEndpointParameters(_config_0);
        const _config_2 = middlewareUserAgent.resolveUserAgentConfig(_config_1);
        const _config_3 = middlewareRetry.resolveRetryConfig(_config_2);
        const _config_4 = configResolver.resolveRegionConfig(_config_3);
        const _config_5 = middlewareHostHeader.resolveHostHeaderConfig(_config_4);
        const _config_6 = middlewareEndpoint.resolveEndpointConfig(_config_5);
        const _config_7 = httpAuthSchemeProvider.resolveHttpAuthSchemeConfig(_config_6);
        const _config_8 = resolveRuntimeExtensions(_config_7, configuration?.extensions || []);
        this.config = _config_8;
        this.middlewareStack.use(schema.getSchemaSerdePlugin(this.config));
        this.middlewareStack.use(middlewareUserAgent.getUserAgentPlugin(this.config));
        this.middlewareStack.use(middlewareRetry.getRetryPlugin(this.config));
        this.middlewareStack.use(middlewareContentLength.getContentLengthPlugin(this.config));
        this.middlewareStack.use(middlewareHostHeader.getHostHeaderPlugin(this.config));
        this.middlewareStack.use(middlewareLogger.getLoggerPlugin(this.config));
        this.middlewareStack.use(middlewareRecursionDetection.getRecursionDetectionPlugin(this.config));
        this.middlewareStack.use(core.getHttpAuthSchemeEndpointRuleSetPlugin(this.config, {
            httpAuthSchemeParametersProvider: httpAuthSchemeProvider.defaultRekognitionHttpAuthSchemeParametersProvider,
            identityProviderConfigProvider: async (config) => new core.DefaultIdentityProviderConfig({
                "aws.auth#sigv4": config.credentials,
            }),
        }));
        this.middlewareStack.use(core.getHttpSigningPlugin(this.config));
    }
    destroy() {
        super.destroy();
    }
}

class RekognitionServiceException extends smithyClient.ServiceException {
    constructor(options) {
        super(options);
        Object.setPrototypeOf(this, RekognitionServiceException.prototype);
    }
}

class AccessDeniedException extends RekognitionServiceException {
    name = "AccessDeniedException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "AccessDeniedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, AccessDeniedException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class ConflictException extends RekognitionServiceException {
    name = "ConflictException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "ConflictException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ConflictException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class IdempotentParameterMismatchException extends RekognitionServiceException {
    name = "IdempotentParameterMismatchException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "IdempotentParameterMismatchException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, IdempotentParameterMismatchException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class InternalServerError extends RekognitionServiceException {
    name = "InternalServerError";
    $fault = "server";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "InternalServerError",
            $fault: "server",
            ...opts,
        });
        Object.setPrototypeOf(this, InternalServerError.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class InvalidParameterException extends RekognitionServiceException {
    name = "InvalidParameterException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "InvalidParameterException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidParameterException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class ProvisionedThroughputExceededException extends RekognitionServiceException {
    name = "ProvisionedThroughputExceededException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "ProvisionedThroughputExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ProvisionedThroughputExceededException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class ResourceNotFoundException extends RekognitionServiceException {
    name = "ResourceNotFoundException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "ResourceNotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceNotFoundException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class ServiceQuotaExceededException extends RekognitionServiceException {
    name = "ServiceQuotaExceededException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "ServiceQuotaExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ServiceQuotaExceededException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class ThrottlingException extends RekognitionServiceException {
    name = "ThrottlingException";
    $fault = "server";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "ThrottlingException",
            $fault: "server",
            ...opts,
        });
        Object.setPrototypeOf(this, ThrottlingException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class ImageTooLargeException extends RekognitionServiceException {
    name = "ImageTooLargeException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "ImageTooLargeException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ImageTooLargeException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class InvalidImageFormatException extends RekognitionServiceException {
    name = "InvalidImageFormatException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "InvalidImageFormatException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidImageFormatException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class InvalidS3ObjectException extends RekognitionServiceException {
    name = "InvalidS3ObjectException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "InvalidS3ObjectException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidS3ObjectException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class LimitExceededException extends RekognitionServiceException {
    name = "LimitExceededException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "LimitExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, LimitExceededException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class ResourceInUseException extends RekognitionServiceException {
    name = "ResourceInUseException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "ResourceInUseException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceInUseException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class ResourceAlreadyExistsException extends RekognitionServiceException {
    name = "ResourceAlreadyExistsException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "ResourceAlreadyExistsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceAlreadyExistsException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class InvalidPolicyRevisionIdException extends RekognitionServiceException {
    name = "InvalidPolicyRevisionIdException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "InvalidPolicyRevisionIdException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidPolicyRevisionIdException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class InvalidPaginationTokenException extends RekognitionServiceException {
    name = "InvalidPaginationTokenException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "InvalidPaginationTokenException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidPaginationTokenException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class ResourceNotReadyException extends RekognitionServiceException {
    name = "ResourceNotReadyException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "ResourceNotReadyException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceNotReadyException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class HumanLoopQuotaExceededException extends RekognitionServiceException {
    name = "HumanLoopQuotaExceededException";
    $fault = "client";
    ResourceType;
    QuotaCode;
    ServiceCode;
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "HumanLoopQuotaExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, HumanLoopQuotaExceededException.prototype);
        this.ResourceType = opts.ResourceType;
        this.QuotaCode = opts.QuotaCode;
        this.ServiceCode = opts.ServiceCode;
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class SessionNotFoundException extends RekognitionServiceException {
    name = "SessionNotFoundException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "SessionNotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, SessionNotFoundException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class InvalidManifestException extends RekognitionServiceException {
    name = "InvalidManifestException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "InvalidManifestException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidManifestException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class MalformedPolicyDocumentException extends RekognitionServiceException {
    name = "MalformedPolicyDocumentException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "MalformedPolicyDocumentException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, MalformedPolicyDocumentException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}
class VideoTooLargeException extends RekognitionServiceException {
    name = "VideoTooLargeException";
    $fault = "client";
    Message;
    Code;
    Logref;
    constructor(opts) {
        super({
            name: "VideoTooLargeException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, VideoTooLargeException.prototype);
        this.Message = opts.Message;
        this.Code = opts.Code;
        this.Logref = opts.Logref;
    }
}

const _A = "Asset";
const _AB = "AggregateBy";
const _AC = "AutoCreate";
const _ADE = "AccessDeniedException";
const _AF = "AssociatedFace";
const _AFL = "AssociatedFacesList";
const _AFR = "AssociateFacesRequest";
const _AFRs = "AssociateFacesResponse";
const _AFs = "AssociatedFaces";
const _AFss = "AssociateFaces";
const _AI = "AuditImage";
const _AIL = "AuditImagesLimit";
const _AIu = "AuditImages";
const _AM = "AudioMetadata";
const _AML = "AudioMetadataList";
const _AR = "AgeRange";
const _AU = "AutoUpdate";
const _Al = "Aliases";
const _Ar = "Arn";
const _As = "Assets";
const _At = "Attributes";
const _B = "Bytes";
const _BB = "BoundingBox";
const _BBC = "BoundingBoxCount";
const _BF = "BlackFrame";
const _BMV = "BaseModelVersion";
const _BP = "BodyParts";
const _BTTIS = "BillableTrainingTimeInSeconds";
const _Ba = "Background";
const _Be = "Beard";
const _Bl = "Blue";
const _Br = "Brightness";
const _Bu = "Bucket";
const _C = "Code";
const _CA = "CollectionArn";
const _CARN = "CollectionARN";
const _CBP = "CoversBodyPart";
const _CC = "ContentClassifiers";
const _CCR = "CreateCollectionRequest";
const _CCRr = "CreateCollectionResponse";
const _CCr = "CreateCollection";
const _CD = "CelebrityDetail";
const _CDR = "CreateDatasetRequest";
const _CDRr = "CreateDatasetResponse";
const _CDr = "CreateDataset";
const _CE = "ConflictException";
const _CF = "ComparedFace";
const _CFC = "CustomizationFeatureConfig";
const _CFCMC = "CustomizationFeatureContentModerationConfig";
const _CFL = "ComparedFaceList";
const _CFLS = "CreateFaceLivenessSession";
const _CFLSR = "CreateFaceLivenessSessionRequest";
const _CFLSRS = "CreateFaceLivenessSessionRequestSettings";
const _CFLSRr = "CreateFaceLivenessSessionResponse";
const _CFM = "CompareFacesMatch";
const _CFML = "CompareFacesMatchList";
const _CFR = "CompareFacesRequest";
const _CFRo = "CompareFacesResponse";
const _CFUL = "CompareFacesUnmatchList";
const _CFe = "CelebrityFaces";
const _CFo = "CompareFaces";
const _CH = "ConnectedHome";
const _CHFU = "ConnectedHomeForUpdate";
const _CHS = "ConnectedHomeSettings";
const _CHSFU = "ConnectedHomeSettingsForUpdate";
const _CI = "CollectionId";
const _CIo = "CollectionIds";
const _CL = "CustomLabel";
const _CLe = "CelebrityList";
const _CLo = "ContainsLabels";
const _CLu = "CustomLabels";
const _CM = "ContentModeration";
const _CMD = "ContentModerationDetection";
const _CMDo = "ContentModerationDetections";
const _CP = "ChallengePreference";
const _CPR = "CreateProjectRequest";
const _CPRr = "CreateProjectResponse";
const _CPV = "CopyProjectVersion";
const _CPVR = "CopyProjectVersionRequest";
const _CPVRo = "CopyProjectVersionResponse";
const _CPVRr = "CreateProjectVersionRequest";
const _CPVRre = "CreateProjectVersionResponse";
const _CPVr = "CreateProjectVersion";
const _CPh = "ChallengePreferences";
const _CPr = "CreateProject";
const _CR = "CelebrityRecognition";
const _CRT = "ClientRequestToken";
const _CRe = "CelebrityRecognitions";
const _CRo = "ColorRange";
const _CSIF = "ComparedSourceImageFace";
const _CSP = "CreateStreamProcessor";
const _CSPR = "CreateStreamProcessorRequest";
const _CSPRr = "CreateStreamProcessorResponse";
const _CSSC = "CSSColor";
const _CT = "ContentTypes";
const _CTo = "ContentType";
const _CTom = "CompletionTimestamp";
const _CTon = "ConfidenceThreshold";
const _CTr = "CreationTimestamp";
const _CU = "CreateUser";
const _CUR = "CreateUserRequest";
const _CURr = "CreateUserResponse";
const _Ca = "Categories";
const _Ce = "Celebrity";
const _Cel = "Celebrities";
const _Ch = "Challenge";
const _Cha = "Changes";
const _Co = "Codec";
const _Con = "Confidence";
const _Cont = "Contrast";
const _D = "Datasets";
const _DA = "DatasetArn";
const _DAa = "DataAttributes";
const _DAe = "DetectionAttributes";
const _DC = "DatasetChanges";
const _DCL = "DetectCustomLabels";
const _DCLR = "DetectCustomLabelsRequest";
const _DCLRe = "DetectCustomLabelsResponse";
const _DCR = "DeleteCollectionRequest";
const _DCRe = "DeleteCollectionResponse";
const _DCRes = "DescribeCollectionRequest";
const _DCResc = "DescribeCollectionResponse";
const _DCe = "DeleteCollection";
const _DCes = "DescribeCollection";
const _DCo = "DominantColors";
const _DCom = "DominantColor";
const _DD = "DatasetDescription";
const _DDE = "DistributeDatasetEntries";
const _DDER = "DistributeDatasetEntriesRequest";
const _DDERi = "DistributeDatasetEntriesResponse";
const _DDML = "DistributeDatasetMetadataList";
const _DDR = "DeleteDatasetRequest";
const _DDRe = "DeleteDatasetResponse";
const _DDRes = "DescribeDatasetRequest";
const _DDResc = "DescribeDatasetResponse";
const _DDe = "DeleteDataset";
const _DDes = "DescribeDataset";
const _DDi = "DistributeDataset";
const _DE = "DatasetEntries";
const _DF = "DeletedFaces";
const _DFL = "DisassociatedFacesList";
const _DFR = "DeleteFacesRequest";
const _DFRe = "DeleteFacesResponse";
const _DFRet = "DetectFacesRequest";
const _DFRete = "DetectFacesResponse";
const _DFRi = "DisassociateFacesRequest";
const _DFRis = "DisassociateFacesResponse";
const _DFe = "DetectionFilter";
const _DFel = "DeleteFaces";
const _DFet = "DetectFaces";
const _DFi = "DisassociatedFace";
const _DFis = "DisassociatedFaces";
const _DFisa = "DisassociateFaces";
const _DFu = "DurationFrames";
const _DL = "DetectLabels";
const _DLD = "DatasetLabelDescription";
const _DLDa = "DatasetLabelDescriptions";
const _DLIB = "DetectLabelsImageBackground";
const _DLIF = "DetectLabelsImageForeground";
const _DLIP = "DetectLabelsImageProperties";
const _DLIPS = "DetectLabelsImagePropertiesSettings";
const _DLIQ = "DetectLabelsImageQuality";
const _DLR = "DetectLabelsRequest";
const _DLRe = "DetectLabelsResponse";
const _DLS = "DatasetLabelStats";
const _DLSe = "DetectLabelsSettings";
const _DM = "DurationMillis";
const _DML = "DetectModerationLabels";
const _DMLR = "DetectModerationLabelsRequest";
const _DMLRe = "DetectModerationLabelsResponse";
const _DMLa = "DatasetMetadataList";
const _DMa = "DatasetMetadata";
const _DP = "DeleteProject";
const _DPA = "DestinationProjectArn";
const _DPE = "DetectProtectiveEquipment";
const _DPER = "DetectProtectiveEquipmentRequest";
const _DPERe = "DetectProtectiveEquipmentResponse";
const _DPP = "DeleteProjectPolicy";
const _DPPR = "DeleteProjectPolicyRequest";
const _DPPRe = "DeleteProjectPolicyResponse";
const _DPR = "DeleteProjectRequest";
const _DPRe = "DeleteProjectResponse";
const _DPRes = "DescribeProjectsRequest";
const _DPResc = "DescribeProjectsResponse";
const _DPV = "DeleteProjectVersion";
const _DPVR = "DeleteProjectVersionRequest";
const _DPVRe = "DeleteProjectVersionResponse";
const _DPVRes = "DescribeProjectVersionsRequest";
const _DPVResc = "DescribeProjectVersionsResponse";
const _DPVe = "DescribeProjectVersions";
const _DPe = "DescribeProjects";
const _DS = "DatasetSource";
const _DSMPTE = "DurationSMPTE";
const _DSP = "DataSharingPreference";
const _DSPFU = "DataSharingPreferenceForUpdate";
const _DSPR = "DeleteStreamProcessorRequest";
const _DSPRe = "DeleteStreamProcessorResponse";
const _DSPRes = "DescribeStreamProcessorRequest";
const _DSPResc = "DescribeStreamProcessorResponse";
const _DSPe = "DeleteStreamProcessor";
const _DSPes = "DescribeStreamProcessor";
const _DSa = "DatasetStats";
const _DT = "DatasetType";
const _DTF = "DetectTextFilters";
const _DTR = "DetectTextRequest";
const _DTRe = "DetectTextResponse";
const _DTe = "DetectedText";
const _DTet = "DetectText";
const _DU = "DeleteUser";
const _DUR = "DeleteUserRequest";
const _DURe = "DeleteUserResponse";
const _E = "Emotions";
const _EC = "EntryCount";
const _ED = "EquipmentDetection";
const _EDq = "EquipmentDetections";
const _EDy = "EyeDirection";
const _EE = "ErrorEntries";
const _EFN = "EndFrameNumber";
const _EII = "ExternalImageId";
const _EO = "EyeOpen";
const _EOy = "EyesOpen";
const _ER = "EvaluationResult";
const _ETM = "EndTimestampMillis";
const _ETSMPTE = "EndTimecodeSMPTE";
const _Em = "Emotion";
const _Ey = "Eyeglasses";
const _F = "Face";
const _FA = "FaceAttributes";
const _FC = "FeatureConfig";
const _FCa = "FaceCount";
const _FD = "FaceDetails";
const _FDA = "FlowDefinitionArn";
const _FDL = "FaceDetailList";
const _FDa = "FaceDetail";
const _FDac = "FaceDetection";
const _FDace = "FaceDetections";
const _FDai = "FailureDetails";
const _FH = "FrameHeight";
const _FI = "FaceId";
const _FIa = "FaceIds";
const _FL = "FaceList";
const _FM = "FaceMatches";
const _FML = "FaceMatchList";
const _FMT = "FaceMatchThreshold";
const _FMV = "FaceModelVersion";
const _FMVa = "FaceModelVersions";
const _FMa = "FaceMatch";
const _FN = "FragmentNumber";
const _FO = "FaceOccluded";
const _FR = "FaceRecord";
const _FRL = "FaceRecordList";
const _FRa = "FaceRecords";
const _FRr = "FrameRate";
const _FS = "F1Score";
const _FSS = "FaceSearchSettings";
const _FSa = "FaceSearch";
const _FW = "FrameWidth";
const _Fa = "Faces";
const _Fe = "Feature";
const _Fea = "Features";
const _Fi = "Filters";
const _Fo = "Foreground";
const _For = "Format";
const _G = "Geometry";
const _GCI = "GetCelebrityInfo";
const _GCIR = "GetCelebrityInfoRequest";
const _GCIRe = "GetCelebrityInfoResponse";
const _GCM = "GetContentModeration";
const _GCMR = "GetContentModerationRequest";
const _GCMRM = "GetContentModerationRequestMetadata";
const _GCMRe = "GetContentModerationResponse";
const _GCR = "GetCelebrityRecognition";
const _GCRR = "GetCelebrityRecognitionRequest";
const _GCRRe = "GetCelebrityRecognitionResponse";
const _GFD = "GetFaceDetection";
const _GFDR = "GetFaceDetectionRequest";
const _GFDRe = "GetFaceDetectionResponse";
const _GFLSR = "GetFaceLivenessSessionResults";
const _GFLSRR = "GetFaceLivenessSessionResultsRequest";
const _GFLSRRe = "GetFaceLivenessSessionResultsResponse";
const _GFS = "GetFaceSearch";
const _GFSR = "GetFaceSearchRequest";
const _GFSRe = "GetFaceSearchResponse";
const _GL = "GeneralLabels";
const _GLD = "GetLabelDetection";
const _GLDR = "GetLabelDetectionRequest";
const _GLDRM = "GetLabelDetectionRequestMetadata";
const _GLDRe = "GetLabelDetectionResponse";
const _GLS = "GeneralLabelsSettings";
const _GMAJ = "GetMediaAnalysisJob";
const _GMAJR = "GetMediaAnalysisJobRequest";
const _GMAJRe = "GetMediaAnalysisJobResponse";
const _GPT = "GetPersonTracking";
const _GPTR = "GetPersonTrackingRequest";
const _GPTRe = "GetPersonTrackingResponse";
const _GRM = "GetRequestMetadata";
const _GSD = "GetSegmentDetection";
const _GSDR = "GetSegmentDetectionRequest";
const _GSDRe = "GetSegmentDetectionResponse";
const _GT = "GroundTruth";
const _GTD = "GetTextDetection";
const _GTDR = "GetTextDetectionRequest";
const _GTDRe = "GetTextDetectionResponse";
const _GTM = "GroundTruthManifest";
const _Ge = "Gender";
const _Gr = "Green";
const _H = "High";
const _HC = "HexCode";
const _HE = "HasErrors";
const _HLA = "HumanLoopArn";
const _HLACER = "HumanLoopActivationConditionsEvaluationResults";
const _HLAO = "HumanLoopActivationOutput";
const _HLAR = "HumanLoopActivationReasons";
const _HLC = "HumanLoopConfig";
const _HLDA = "HumanLoopDataAttributes";
const _HLN = "HumanLoopName";
const _HLQEE = "HumanLoopQuotaExceededException";
const _He = "Height";
const _I = "Id";
const _IF = "IndexFaces";
const _IFMV = "IndexFacesModelVersion";
const _IFR = "IndexFacesRequest";
const _IFRn = "IndexFacesResponse";
const _II = "ImageId";
const _IIFE = "InvalidImageFormatException";
const _IME = "InvalidManifestException";
const _IP = "ImageProperties";
const _IPE = "InvalidParameterException";
const _IPME = "IdempotentParameterMismatchException";
const _IPRIE = "InvalidPolicyRevisionIdException";
const _IPTE = "InvalidPaginationTokenException";
const _IQ = "ImageQuality";
const _ISE = "InternalServerError";
const _ISOE = "InvalidS3ObjectException";
const _ITLE = "ImageTooLargeException";
const _Im = "Image";
const _In = "Input";
const _Ind = "Index";
const _Ins = "Instance";
const _Inst = "Instances";
const _JI = "JobId";
const _JN = "JobName";
const _JS = "JobStatus";
const _JT = "JobTag";
const _KDS = "KinesisDataStream";
const _KG = "KnownGender";
const _KKI = "KmsKeyId";
const _KP = "KeyPrefix";
const _KVS = "KinesisVideoStream";
const _KVSSS = "KinesisVideoStreamStartSelector";
const _KVSSSS = "KVSStreamStartSelector";
const _L = "Logref";
const _LA = "LabelAlias";
const _LAa = "LabelAliases";
const _LC = "LabelCategory";
const _LCEF = "LabelCategoryExclusionFilters";
const _LCIF = "LabelCategoryInclusionFilters";
const _LCR = "ListCollectionsRequest";
const _LCRi = "ListCollectionsResponse";
const _LCa = "LabelCategories";
const _LCi = "ListCollections";
const _LD = "LabelDetection";
const _LDE = "ListDatasetEntries";
const _LDER = "ListDatasetEntriesRequest";
const _LDERi = "ListDatasetEntriesResponse";
const _LDL = "ListDatasetLabels";
const _LDLR = "ListDatasetLabelsRequest";
const _LDLRi = "ListDatasetLabelsResponse";
const _LDS = "LabelDetectionSettings";
const _LDa = "LabelDetections";
const _LE = "LabeledEntries";
const _LEE = "LimitExceededException";
const _LEF = "LabelExclusionFilters";
const _LF = "ListFaces";
const _LFR = "ListFacesRequest";
const _LFRi = "ListFacesResponse";
const _LIB = "LivenessImageBlob";
const _LIF = "LabelInclusionFilters";
const _LMAJ = "ListMediaAnalysisJobs";
const _LMAJR = "ListMediaAnalysisJobsRequest";
const _LMAJRi = "ListMediaAnalysisJobsResponse";
const _LMV = "LabelModelVersion";
const _LN = "LabelName";
const _LOC = "LivenessOutputConfig";
const _LPP = "ListProjectPolicies";
const _LPPR = "ListProjectPoliciesRequest";
const _LPPRi = "ListProjectPoliciesResponse";
const _LS = "LabelStats";
const _LSP = "ListStreamProcessors";
const _LSPR = "ListStreamProcessorsRequest";
const _LSPRi = "ListStreamProcessorsResponse";
const _LTFR = "ListTagsForResource";
const _LTFRR = "ListTagsForResourceRequest";
const _LTFRRi = "ListTagsForResourceResponse";
const _LU = "ListUsers";
const _LUR = "ListUsersRequest";
const _LURi = "ListUsersResponse";
const _LUT = "LastUpdatedTimestamp";
const _LUTa = "LastUpdateTimestamp";
const _La = "Landmarks";
const _Lab = "Labels";
const _Labe = "Label";
const _Label = "Labeled";
const _Lan = "Landmark";
const _Le = "Left";
const _Lo = "Low";
const _M = "Message";
const _MADMLC = "MediaAnalysisDetectModerationLabelsConfig";
const _MAI = "MediaAnalysisInput";
const _MAJ = "MediaAnalysisJobs";
const _MAJD = "MediaAnalysisJobDescription";
const _MAJDe = "MediaAnalysisJobDescriptions";
const _MAJFD = "MediaAnalysisJobFailureDetails";
const _MAMS = "MediaAnalysisManifestSummary";
const _MAMV = "MediaAnalysisModelVersions";
const _MAOC = "MediaAnalysisOperationsConfig";
const _MAOCe = "MediaAnalysisOutputConfig";
const _MAR = "MediaAnalysisResults";
const _MBBH = "MinBoundingBoxHeight";
const _MBBW = "MinBoundingBoxWidth";
const _MC = "MatchConfidence";
const _MCP = "MinCoveragePercentage";
const _MCi = "MinConfidence";
const _MDC = "MaxDominantColors";
const _MDIS = "MaxDurationInSeconds";
const _MF = "MaxFaces";
const _MIU = "MinInferenceUnits";
const _MIUa = "MaxInferenceUnits";
const _ML = "ModerationLabel";
const _MLa = "MaxLabels";
const _MLo = "ModerationLabels";
const _MMV = "ModerationModelVersion";
const _MO = "MouthOpen";
const _MPDE = "MalformedPolicyDocumentException";
const _MPT = "MaxPixelThreshold";
const _MR = "MaxResults";
const _MS = "ManifestSummary";
const _MSC = "MinSegmentConfidence";
const _MU = "MatchedUser";
const _MUa = "MaxUsers";
const _MV = "ModelVersions";
const _MVo = "ModelVersion";
const _Ma = "Maximum";
const _Mi = "Minimum";
const _Mo = "Moderation";
const _Mu = "Mustache";
const _N = "Name";
const _NC = "NotificationChannel";
const _NOC = "NumberOfChannels";
const _NT = "NextToken";
const _O = "Output";
const _OC = "OutputConfig";
const _OCp = "OperationsConfig";
const _OCr = "OrientationCorrection";
const _OI = "OptIn";
const _P = "Pose";
const _PA = "ProjectArn";
const _PD = "ProjectDescriptions";
const _PDe = "PersonDetail";
const _PDer = "PersonDetection";
const _PDers = "PersonDetections";
const _PDo = "PolicyDocument";
const _PDr = "ProjectDescription";
const _PEBP = "ProtectiveEquipmentBodyPart";
const _PEMV = "ProtectiveEquipmentModelVersion";
const _PEP = "ProtectiveEquipmentPerson";
const _PEPr = "ProtectiveEquipmentPersons";
const _PES = "ProtectiveEquipmentSummary";
const _PESA = "ProtectiveEquipmentSummarizationAttributes";
const _PI = "PersonsIndeterminate";
const _PIa = "ParentId";
const _PM = "PersonMatch";
const _PMe = "PersonMatches";
const _PN = "ProjectName";
const _PNa = "ParentName";
const _PNo = "PolicyName";
const _PNr = "ProjectNames";
const _PP = "PixelPercent";
const _PPP = "PutProjectPolicy";
const _PPPR = "PutProjectPolicyRequest";
const _PPPRu = "PutProjectPolicyResponse";
const _PPr = "ProjectPolicies";
const _PPro = "ProjectPolicy";
const _PRI = "PolicyRevisionId";
const _PT = "ProducerTimestamp";
const _PTD = "ParametersToDelete";
const _PTEE = "ProvisionedThroughputExceededException";
const _PV = "ProjectVersion";
const _PVA = "ProjectVersionArn";
const _PVD = "ProjectVersionDescriptions";
const _PVDr = "ProjectVersionDescription";
const _PWRE = "PersonsWithRequiredEquipment";
const _PWREe = "PersonsWithoutRequiredEquipment";
const _Pa = "Parents";
const _Par = "Parent";
const _Pe = "Persons";
const _Per = "Person";
const _Pi = "Pitch";
const _Po = "Polygon";
const _Poi = "Point";
const _Q = "Quality";
const _QC = "QuotaCode";
const _QF = "QualityFilter";
const _R = "Red";
const _RA = "RoleArn";
const _RAEE = "ResourceAlreadyExistsException";
const _RAe = "ResourceArn";
const _RC = "RecognizeCelebrities";
const _RCR = "RecognizeCelebritiesRequest";
const _RCRe = "RecognizeCelebritiesResponse";
const _RET = "RequiredEquipmentTypes";
const _RI = "ReferenceImage";
const _RIUE = "ResourceInUseException";
const _RNFE = "ResourceNotFoundException";
const _RNRE = "ResourceNotReadyException";
const _ROI = "RegionsOfInterest";
const _ROIFU = "RegionsOfInterestForUpdate";
const _ROIe = "RegionOfInterest";
const _RT = "ResourceType";
const _Re = "Results";
const _Rea = "Reasons";
const _Ro = "Roll";
const _S = "Smile";
const _SA = "SummarizationAttributes";
const _SB = "SortBy";
const _SBu = "S3Bucket";
const _SC = "StatusCode";
const _SCM = "StartContentModeration";
const _SCMR = "StartContentModerationRequest";
const _SCMRt = "StartContentModerationResponse";
const _SCR = "StartCelebrityRecognition";
const _SCRR = "StartCelebrityRecognitionRequest";
const _SCRRt = "StartCelebrityRecognitionResponse";
const _SCe = "ServiceCode";
const _SCi = "SimplifiedColor";
const _SD = "S3Destination";
const _SDe = "SegmentDetection";
const _SDeg = "SegmentDetections";
const _SF = "SearchedFace";
const _SFBB = "SearchedFaceBoundingBox";
const _SFBI = "SearchFacesByImage";
const _SFBIR = "SearchFacesByImageRequest";
const _SFBIRe = "SearchFacesByImageResponse";
const _SFC = "SearchedFaceConfidence";
const _SFD = "SearchedFaceDetails";
const _SFDR = "StartFaceDetectionRequest";
const _SFDRt = "StartFaceDetectionResponse";
const _SFDt = "StartFaceDetection";
const _SFI = "SearchedFaceId";
const _SFN = "StartFrameNumber";
const _SFR = "SearchFacesRequest";
const _SFRe = "SearchFacesResponse";
const _SFS = "StartFaceSearch";
const _SFSR = "StartFaceSearchRequest";
const _SFSRt = "StartFaceSearchResponse";
const _SFU = "SettingsForUpdate";
const _SFe = "SearchFaces";
const _SFh = "ShotFilter";
const _SI = "SourceImage";
const _SIF = "SourceImageFace";
const _SIOC = "SourceImageOrientationCorrection";
const _SIe = "SessionId";
const _SJHLACER = "SynthesizedJsonHumanLoopActivationConditionsEvaluationResults";
const _SKP = "S3KeyPrefix";
const _SLD = "StartLabelDetection";
const _SLDR = "StartLabelDetectionRequest";
const _SLDRt = "StartLabelDetectionResponse";
const _SM = "StatusMessage";
const _SMAJ = "StartMediaAnalysisJob";
const _SMAJR = "StartMediaAnalysisJobRequest";
const _SMAJRt = "StartMediaAnalysisJobResponse";
const _SMC = "StatusMessageCode";
const _SNFE = "SessionNotFoundException";
const _SNSTA = "SNSTopicArn";
const _SO = "S3Object";
const _SP = "StreamProcessors";
const _SPA = "SourceProjectArn";
const _SPAt = "StreamProcessorArn";
const _SPDSP = "StreamProcessorDataSharingPreference";
const _SPI = "StreamProcessorInput";
const _SPL = "StreamProcessorList";
const _SPNC = "StreamProcessorNotificationChannel";
const _SPO = "StreamProcessorOutput";
const _SPS = "StreamProcessorSettings";
const _SPSFU = "StreamProcessorSettingsForUpdate";
const _SPSS = "StreamProcessingStartSelector";
const _SPSSt = "StreamProcessingStopSelector";
const _SPT = "StartPersonTracking";
const _SPTR = "StartPersonTrackingRequest";
const _SPTRt = "StartPersonTrackingResponse";
const _SPV = "StartProjectVersion";
const _SPVA = "SourceProjectVersionArn";
const _SPVR = "StartProjectVersionRequest";
const _SPVRt = "StartProjectVersionResponse";
const _SPVRto = "StopProjectVersionRequest";
const _SPVRtop = "StopProjectVersionResponse";
const _SPVt = "StopProjectVersion";
const _SPt = "StreamProcessor";
const _SQEE = "ServiceQuotaExceededException";
const _SR = "SampleRate";
const _SRC = "SourceRefContains";
const _SS = "ShotSegment";
const _SSD = "StartSegmentDetection";
const _SSDF = "StartSegmentDetectionFilters";
const _SSDFt = "StartShotDetectionFilter";
const _SSDR = "StartSegmentDetectionRequest";
const _SSDRt = "StartSegmentDetectionResponse";
const _SSP = "StartStreamProcessor";
const _SSPR = "StartStreamProcessorRequest";
const _SSPRt = "StartStreamProcessorResponse";
const _SSPRto = "StopStreamProcessorRequest";
const _SSPRtop = "StopStreamProcessorResponse";
const _SSPt = "StopStreamProcessor";
const _SST = "SelectedSegmentTypes";
const _SSt = "StartSelector";
const _SSto = "StopSelector";
const _ST = "SimilarityThreshold";
const _STCDF = "StartTechnicalCueDetectionFilter";
const _STD = "StartTextDetection";
const _STDF = "StartTextDetectionFilters";
const _STDR = "StartTextDetectionRequest";
const _STDRt = "StartTextDetectionResponse";
const _STI = "SegmentTypeInfo";
const _STIe = "SegmentTypesInfo";
const _STM = "StartTimestampMillis";
const _STSMPTE = "StartTimecodeSMPTE";
const _STe = "SegmentTypes";
const _SU = "SearchedUser";
const _SUBI = "SearchUsersByImage";
const _SUBIR = "SearchUsersByImageRequest";
const _SUBIRe = "SearchUsersByImageResponse";
const _SUR = "SearchUsersRequest";
const _SURe = "SearchUsersResponse";
const _SUe = "SearchUsers";
const _Se = "Settings";
const _Seg = "Segments";
const _Sh = "Sharpness";
const _Si = "Similarity";
const _St = "Status";
const _Su = "Summary";
const _Sun = "Sunglasses";
const _T = "Top";
const _TCF = "TechnicalCueFilter";
const _TCS = "TechnicalCueSegment";
const _TD = "TrainingData";
const _TDL = "TextDetectionList";
const _TDR = "TrainingDataResult";
const _TDRe = "TestingDataResult";
const _TDRex = "TextDetectionResult";
const _TDRext = "TextDetectionResults";
const _TDe = "TestingData";
const _TDex = "TextDetections";
const _TDext = "TextDetection";
const _TE = "TotalEntries";
const _TET = "TrainingEndTimestamp";
const _TEh = "ThrottlingException";
const _TI = "TargetImage";
const _TIOC = "TargetImageOrientationCorrection";
const _TK = "TagKeys";
const _TL = "TotalLabels";
const _TLa = "TaxonomyLevel";
const _TMV = "TextModelVersion";
const _TR = "TagResource";
const _TRR = "TagResourceRequest";
const _TRRa = "TagResourceResponse";
const _Ta = "Tags";
const _Ti = "Timestamp";
const _Ty = "Type";
const _U = "Urls";
const _UC = "UserCount";
const _UDE = "UpdateDatasetEntries";
const _UDER = "UpdateDatasetEntriesRequest";
const _UDERp = "UpdateDatasetEntriesResponse";
const _UF = "UnmatchedFaces";
const _UFA = "UnsuccessfulFaceAssociations";
const _UFAL = "UnsuccessfulFaceAssociationList";
const _UFAn = "UnsuccessfulFaceAssociation";
const _UFD = "UnsuccessfulFaceDeletions";
const _UFDL = "UnsuccessfulFaceDeletionsList";
const _UFDLn = "UnsuccessfulFaceDisassociationList";
const _UFDn = "UnsuccessfulFaceDisassociations";
const _UFDns = "UnsuccessfulFaceDeletion";
const _UFDnsu = "UnsuccessfulFaceDisassociation";
const _UFL = "UnsearchedFacesList";
const _UFn = "UnindexedFaces";
const _UFni = "UnindexedFace";
const _UFnr = "UnrecognizedFaces";
const _UFns = "UnsearchedFaces";
const _UFnse = "UnsearchedFace";
const _UI = "UserId";
const _UL = "UserList";
const _UM = "UserMatches";
const _UML = "UserMatchList";
const _UMT = "UserMatchThreshold";
const _UMs = "UserMatch";
const _UR = "UntagResource";
const _URR = "UntagResourceRequest";
const _URRn = "UntagResourceResponse";
const _US = "UserStatus";
const _USP = "UpdateStreamProcessor";
const _USPR = "UpdateStreamProcessorRequest";
const _USPRp = "UpdateStreamProcessorResponse";
const _Us = "Users";
const _Use = "User";
const _V = "Value";
const _VD = "VersionDescription";
const _VDa = "ValidationData";
const _VM = "VideoMetadata";
const _VML = "VideoMetadataList";
const _VN = "VersionName";
const _VNe = "VersionNames";
const _VTLE = "VideoTooLargeException";
const _Va = "Validation";
const _Ve = "Version";
const _Ver = "Versions";
const _Vi = "Video";
const _W = "Width";
const _WF = "WordFilter";
const _X = "X";
const _Y = "Yaw";
const _Y_ = "Y";
const _a = "application/json";
const _c = "client";
const _e = "error";
const _hE = "httpError";
const _mT = "mediaType";
const _s = "server";
const _sm = "smithy.ts.sdk.synthetic.com.amazonaws.rekognition";
const n0 = "com.amazonaws.rekognition";
var LivenessImageBlob = [0, n0, _LIB, 8, 21];
var SynthesizedJsonHumanLoopActivationConditionsEvaluationResults = [0, n0, _SJHLACER, { [_mT]: _a }, 0];
var AccessDeniedException$ = [-3, n0, _ADE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(AccessDeniedException$, AccessDeniedException);
var AgeRange$ = [3, n0, _AR,
    0,
    [_Lo, _H],
    [1, 1]
];
var Asset$ = [3, n0, _A,
    0,
    [_GTM],
    [() => GroundTruthManifest$]
];
var AssociatedFace$ = [3, n0, _AF,
    0,
    [_FI],
    [0]
];
var AssociateFacesRequest$ = [3, n0, _AFR,
    0,
    [_CI, _UI, _FIa, _UMT, _CRT],
    [0, 0, 64 | 0, 1, [0, 4]], 3
];
var AssociateFacesResponse$ = [3, n0, _AFRs,
    0,
    [_AFs, _UFA, _US],
    [() => AssociatedFacesList, () => UnsuccessfulFaceAssociationList, 0]
];
var AudioMetadata$ = [3, n0, _AM,
    0,
    [_Co, _DM, _SR, _NOC],
    [0, 1, 1, 1]
];
var AuditImage$ = [3, n0, _AI,
    0,
    [_B, _SO, _BB],
    [[() => LivenessImageBlob, 0], () => S3Object$, () => BoundingBox$]
];
var Beard$ = [3, n0, _Be,
    0,
    [_V, _Con],
    [2, 1]
];
var BlackFrame$ = [3, n0, _BF,
    0,
    [_MPT, _MCP],
    [1, 1]
];
var BoundingBox$ = [3, n0, _BB,
    0,
    [_W, _He, _Le, _T],
    [1, 1, 1, 1]
];
var Celebrity$ = [3, n0, _Ce,
    0,
    [_U, _N, _I, _F, _MC, _KG],
    [64 | 0, 0, 0, () => ComparedFace$, 1, () => KnownGender$]
];
var CelebrityDetail$ = [3, n0, _CD,
    0,
    [_U, _N, _I, _Con, _BB, _F, _KG],
    [64 | 0, 0, 0, 1, () => BoundingBox$, () => FaceDetail$, () => KnownGender$]
];
var CelebrityRecognition$ = [3, n0, _CR,
    0,
    [_Ti, _Ce],
    [1, () => CelebrityDetail$]
];
var Challenge$ = [3, n0, _Ch,
    0,
    [_Ty, _Ve],
    [0, 0], 2
];
var ChallengePreference$ = [3, n0, _CP,
    0,
    [_Ty, _Ver],
    [0, () => Versions$], 1
];
var ComparedFace$ = [3, n0, _CF,
    0,
    [_BB, _Con, _La, _P, _Q, _E, _S],
    [() => BoundingBox$, 1, () => Landmarks, () => Pose$, () => ImageQuality$, () => Emotions, () => Smile$]
];
var ComparedSourceImageFace$ = [3, n0, _CSIF,
    0,
    [_BB, _Con],
    [() => BoundingBox$, 1]
];
var CompareFacesMatch$ = [3, n0, _CFM,
    0,
    [_Si, _F],
    [1, () => ComparedFace$]
];
var CompareFacesRequest$ = [3, n0, _CFR,
    0,
    [_SI, _TI, _ST, _QF],
    [() => Image$, () => Image$, 1, 0], 2
];
var CompareFacesResponse$ = [3, n0, _CFRo,
    0,
    [_SIF, _FM, _UF, _SIOC, _TIOC],
    [() => ComparedSourceImageFace$, () => CompareFacesMatchList, () => CompareFacesUnmatchList, 0, 0]
];
var ConflictException$ = [-3, n0, _CE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(ConflictException$, ConflictException);
var ConnectedHomeSettings$ = [3, n0, _CHS,
    0,
    [_Lab, _MCi],
    [64 | 0, 1], 1
];
var ConnectedHomeSettingsForUpdate$ = [3, n0, _CHSFU,
    0,
    [_Lab, _MCi],
    [64 | 0, 1]
];
var ContentModerationDetection$ = [3, n0, _CMD,
    0,
    [_Ti, _ML, _STM, _ETM, _DM, _CT],
    [1, () => ModerationLabel$, 1, 1, 1, () => ContentTypes]
];
var ContentType$ = [3, n0, _CTo,
    0,
    [_Con, _N],
    [1, 0]
];
var CopyProjectVersionRequest$ = [3, n0, _CPVR,
    0,
    [_SPA, _SPVA, _DPA, _VN, _OC, _Ta, _KKI],
    [0, 0, 0, 0, () => OutputConfig$, 128 | 0, 0], 5
];
var CopyProjectVersionResponse$ = [3, n0, _CPVRo,
    0,
    [_PVA],
    [0]
];
var CoversBodyPart$ = [3, n0, _CBP,
    0,
    [_Con, _V],
    [1, 2]
];
var CreateCollectionRequest$ = [3, n0, _CCR,
    0,
    [_CI, _Ta],
    [0, 128 | 0], 1
];
var CreateCollectionResponse$ = [3, n0, _CCRr,
    0,
    [_SC, _CA, _FMV],
    [1, 0, 0]
];
var CreateDatasetRequest$ = [3, n0, _CDR,
    0,
    [_DT, _PA, _DS, _Ta],
    [0, 0, () => DatasetSource$, 128 | 0], 2
];
var CreateDatasetResponse$ = [3, n0, _CDRr,
    0,
    [_DA],
    [0]
];
var CreateFaceLivenessSessionRequest$ = [3, n0, _CFLSR,
    0,
    [_KKI, _Se, _CRT],
    [0, () => CreateFaceLivenessSessionRequestSettings$, 0]
];
var CreateFaceLivenessSessionRequestSettings$ = [3, n0, _CFLSRS,
    0,
    [_OC, _AIL, _CPh],
    [() => LivenessOutputConfig$, 1, () => ChallengePreferences]
];
var CreateFaceLivenessSessionResponse$ = [3, n0, _CFLSRr,
    0,
    [_SIe],
    [0], 1
];
var CreateProjectRequest$ = [3, n0, _CPR,
    0,
    [_PN, _Fe, _AU, _Ta],
    [0, 0, 0, 128 | 0], 1
];
var CreateProjectResponse$ = [3, n0, _CPRr,
    0,
    [_PA],
    [0]
];
var CreateProjectVersionRequest$ = [3, n0, _CPVRr,
    0,
    [_PA, _VN, _OC, _TD, _TDe, _Ta, _KKI, _VD, _FC],
    [0, 0, () => OutputConfig$, () => TrainingData$, () => TestingData$, 128 | 0, 0, 0, () => CustomizationFeatureConfig$], 3
];
var CreateProjectVersionResponse$ = [3, n0, _CPVRre,
    0,
    [_PVA],
    [0]
];
var CreateStreamProcessorRequest$ = [3, n0, _CSPR,
    0,
    [_In, _O, _N, _Se, _RA, _Ta, _NC, _KKI, _ROI, _DSP],
    [() => StreamProcessorInput$, () => StreamProcessorOutput$, 0, () => StreamProcessorSettings$, 0, 128 | 0, () => StreamProcessorNotificationChannel$, 0, () => RegionsOfInterest, () => StreamProcessorDataSharingPreference$], 5
];
var CreateStreamProcessorResponse$ = [3, n0, _CSPRr,
    0,
    [_SPAt],
    [0]
];
var CreateUserRequest$ = [3, n0, _CUR,
    0,
    [_CI, _UI, _CRT],
    [0, 0, [0, 4]], 2
];
var CreateUserResponse$ = [3, n0, _CURr,
    0,
    [],
    []
];
var CustomizationFeatureConfig$ = [3, n0, _CFC,
    0,
    [_CM],
    [() => CustomizationFeatureContentModerationConfig$]
];
var CustomizationFeatureContentModerationConfig$ = [3, n0, _CFCMC,
    0,
    [_CTon],
    [1]
];
var CustomLabel$ = [3, n0, _CL,
    0,
    [_N, _Con, _G],
    [0, 1, () => Geometry$]
];
var DatasetChanges$ = [3, n0, _DC,
    0,
    [_GT],
    [21], 1
];
var DatasetDescription$ = [3, n0, _DD,
    0,
    [_CTr, _LUT, _St, _SM, _SMC, _DSa],
    [4, 4, 0, 0, 0, () => DatasetStats$]
];
var DatasetLabelDescription$ = [3, n0, _DLD,
    0,
    [_LN, _LS],
    [0, () => DatasetLabelStats$]
];
var DatasetLabelStats$ = [3, n0, _DLS,
    0,
    [_EC, _BBC],
    [1, 1]
];
var DatasetMetadata$ = [3, n0, _DMa,
    0,
    [_CTr, _DT, _DA, _St, _SM, _SMC],
    [4, 0, 0, 0, 0, 0]
];
var DatasetSource$ = [3, n0, _DS,
    0,
    [_GTM, _DA],
    [() => GroundTruthManifest$, 0]
];
var DatasetStats$ = [3, n0, _DSa,
    0,
    [_LE, _TE, _TL, _EE],
    [1, 1, 1, 1]
];
var DeleteCollectionRequest$ = [3, n0, _DCR,
    0,
    [_CI],
    [0], 1
];
var DeleteCollectionResponse$ = [3, n0, _DCRe,
    0,
    [_SC],
    [1]
];
var DeleteDatasetRequest$ = [3, n0, _DDR,
    0,
    [_DA],
    [0], 1
];
var DeleteDatasetResponse$ = [3, n0, _DDRe,
    0,
    [],
    []
];
var DeleteFacesRequest$ = [3, n0, _DFR,
    0,
    [_CI, _FIa],
    [0, 64 | 0], 2
];
var DeleteFacesResponse$ = [3, n0, _DFRe,
    0,
    [_DF, _UFD],
    [64 | 0, () => UnsuccessfulFaceDeletionsList]
];
var DeleteProjectPolicyRequest$ = [3, n0, _DPPR,
    0,
    [_PA, _PNo, _PRI],
    [0, 0, 0], 2
];
var DeleteProjectPolicyResponse$ = [3, n0, _DPPRe,
    0,
    [],
    []
];
var DeleteProjectRequest$ = [3, n0, _DPR,
    0,
    [_PA],
    [0], 1
];
var DeleteProjectResponse$ = [3, n0, _DPRe,
    0,
    [_St],
    [0]
];
var DeleteProjectVersionRequest$ = [3, n0, _DPVR,
    0,
    [_PVA],
    [0], 1
];
var DeleteProjectVersionResponse$ = [3, n0, _DPVRe,
    0,
    [_St],
    [0]
];
var DeleteStreamProcessorRequest$ = [3, n0, _DSPR,
    0,
    [_N],
    [0], 1
];
var DeleteStreamProcessorResponse$ = [3, n0, _DSPRe,
    0,
    [],
    []
];
var DeleteUserRequest$ = [3, n0, _DUR,
    0,
    [_CI, _UI, _CRT],
    [0, 0, [0, 4]], 2
];
var DeleteUserResponse$ = [3, n0, _DURe,
    0,
    [],
    []
];
var DescribeCollectionRequest$ = [3, n0, _DCRes,
    0,
    [_CI],
    [0], 1
];
var DescribeCollectionResponse$ = [3, n0, _DCResc,
    0,
    [_FCa, _FMV, _CARN, _CTr, _UC],
    [1, 0, 0, 4, 1]
];
var DescribeDatasetRequest$ = [3, n0, _DDRes,
    0,
    [_DA],
    [0], 1
];
var DescribeDatasetResponse$ = [3, n0, _DDResc,
    0,
    [_DD],
    [() => DatasetDescription$]
];
var DescribeProjectsRequest$ = [3, n0, _DPRes,
    0,
    [_NT, _MR, _PNr, _Fea],
    [0, 1, 64 | 0, 64 | 0]
];
var DescribeProjectsResponse$ = [3, n0, _DPResc,
    0,
    [_PD, _NT],
    [() => ProjectDescriptions, 0]
];
var DescribeProjectVersionsRequest$ = [3, n0, _DPVRes,
    0,
    [_PA, _VNe, _NT, _MR],
    [0, 64 | 0, 0, 1], 1
];
var DescribeProjectVersionsResponse$ = [3, n0, _DPVResc,
    0,
    [_PVD, _NT],
    [() => ProjectVersionDescriptions, 0]
];
var DescribeStreamProcessorRequest$ = [3, n0, _DSPRes,
    0,
    [_N],
    [0], 1
];
var DescribeStreamProcessorResponse$ = [3, n0, _DSPResc,
    0,
    [_N, _SPAt, _St, _SM, _CTr, _LUTa, _In, _O, _RA, _Se, _NC, _KKI, _ROI, _DSP],
    [0, 0, 0, 0, 4, 4, () => StreamProcessorInput$, () => StreamProcessorOutput$, 0, () => StreamProcessorSettings$, () => StreamProcessorNotificationChannel$, 0, () => RegionsOfInterest, () => StreamProcessorDataSharingPreference$]
];
var DetectCustomLabelsRequest$ = [3, n0, _DCLR,
    0,
    [_PVA, _Im, _MR, _MCi],
    [0, () => Image$, 1, 1], 2
];
var DetectCustomLabelsResponse$ = [3, n0, _DCLRe,
    0,
    [_CLu],
    [() => CustomLabels]
];
var DetectFacesRequest$ = [3, n0, _DFRet,
    0,
    [_Im, _At],
    [() => Image$, 64 | 0], 1
];
var DetectFacesResponse$ = [3, n0, _DFRete,
    0,
    [_FD, _OCr],
    [() => FaceDetailList, 0]
];
var DetectionFilter$ = [3, n0, _DFe,
    0,
    [_MCi, _MBBH, _MBBW],
    [1, 1, 1]
];
var DetectLabelsImageBackground$ = [3, n0, _DLIB,
    0,
    [_Q, _DCo],
    [() => DetectLabelsImageQuality$, () => DominantColors]
];
var DetectLabelsImageForeground$ = [3, n0, _DLIF,
    0,
    [_Q, _DCo],
    [() => DetectLabelsImageQuality$, () => DominantColors]
];
var DetectLabelsImageProperties$ = [3, n0, _DLIP,
    0,
    [_Q, _DCo, _Fo, _Ba],
    [() => DetectLabelsImageQuality$, () => DominantColors, () => DetectLabelsImageForeground$, () => DetectLabelsImageBackground$]
];
var DetectLabelsImagePropertiesSettings$ = [3, n0, _DLIPS,
    0,
    [_MDC],
    [1]
];
var DetectLabelsImageQuality$ = [3, n0, _DLIQ,
    0,
    [_Br, _Sh, _Cont],
    [1, 1, 1]
];
var DetectLabelsRequest$ = [3, n0, _DLR,
    0,
    [_Im, _MLa, _MCi, _Fea, _Se],
    [() => Image$, 1, 1, 64 | 0, () => DetectLabelsSettings$], 1
];
var DetectLabelsResponse$ = [3, n0, _DLRe,
    0,
    [_Lab, _OCr, _LMV, _IP],
    [() => Labels, 0, 0, () => DetectLabelsImageProperties$]
];
var DetectLabelsSettings$ = [3, n0, _DLSe,
    0,
    [_GL, _IP],
    [() => GeneralLabelsSettings$, () => DetectLabelsImagePropertiesSettings$]
];
var DetectModerationLabelsRequest$ = [3, n0, _DMLR,
    0,
    [_Im, _MCi, _HLC, _PV],
    [() => Image$, 1, () => HumanLoopConfig$, 0], 1
];
var DetectModerationLabelsResponse$ = [3, n0, _DMLRe,
    0,
    [_MLo, _MMV, _HLAO, _PV, _CT],
    [() => ModerationLabels, 0, [() => HumanLoopActivationOutput$, 0], 0, () => ContentTypes]
];
var DetectProtectiveEquipmentRequest$ = [3, n0, _DPER,
    0,
    [_Im, _SA],
    [() => Image$, () => ProtectiveEquipmentSummarizationAttributes$], 1
];
var DetectProtectiveEquipmentResponse$ = [3, n0, _DPERe,
    0,
    [_PEMV, _Pe, _Su],
    [0, () => ProtectiveEquipmentPersons, () => ProtectiveEquipmentSummary$]
];
var DetectTextFilters$ = [3, n0, _DTF,
    0,
    [_WF, _ROI],
    [() => DetectionFilter$, () => RegionsOfInterest]
];
var DetectTextRequest$ = [3, n0, _DTR,
    0,
    [_Im, _Fi],
    [() => Image$, () => DetectTextFilters$], 1
];
var DetectTextResponse$ = [3, n0, _DTRe,
    0,
    [_TDex, _TMV],
    [() => TextDetectionList, 0]
];
var DisassociatedFace$ = [3, n0, _DFi,
    0,
    [_FI],
    [0]
];
var DisassociateFacesRequest$ = [3, n0, _DFRi,
    0,
    [_CI, _UI, _FIa, _CRT],
    [0, 0, 64 | 0, [0, 4]], 3
];
var DisassociateFacesResponse$ = [3, n0, _DFRis,
    0,
    [_DFis, _UFDn, _US],
    [() => DisassociatedFacesList, () => UnsuccessfulFaceDisassociationList, 0]
];
var DistributeDataset$ = [3, n0, _DDi,
    0,
    [_Ar],
    [0], 1
];
var DistributeDatasetEntriesRequest$ = [3, n0, _DDER,
    0,
    [_D],
    [() => DistributeDatasetMetadataList], 1
];
var DistributeDatasetEntriesResponse$ = [3, n0, _DDERi,
    0,
    [],
    []
];
var DominantColor$ = [3, n0, _DCom,
    0,
    [_R, _Bl, _Gr, _HC, _CSSC, _SCi, _PP],
    [1, 1, 1, 0, 0, 0, 1]
];
var Emotion$ = [3, n0, _Em,
    0,
    [_Ty, _Con],
    [0, 1]
];
var EquipmentDetection$ = [3, n0, _ED,
    0,
    [_BB, _Con, _Ty, _CBP],
    [() => BoundingBox$, 1, 0, () => CoversBodyPart$]
];
var EvaluationResult$ = [3, n0, _ER,
    0,
    [_FS, _Su],
    [1, () => Summary$]
];
var EyeDirection$ = [3, n0, _EDy,
    0,
    [_Y, _Pi, _Con],
    [1, 1, 1]
];
var Eyeglasses$ = [3, n0, _Ey,
    0,
    [_V, _Con],
    [2, 1]
];
var EyeOpen$ = [3, n0, _EO,
    0,
    [_V, _Con],
    [2, 1]
];
var Face$ = [3, n0, _F,
    0,
    [_FI, _BB, _II, _EII, _Con, _IFMV, _UI],
    [0, () => BoundingBox$, 0, 0, 1, 0, 0]
];
var FaceDetail$ = [3, n0, _FDa,
    0,
    [_BB, _AR, _S, _Ey, _Sun, _Ge, _Be, _Mu, _EOy, _MO, _E, _La, _P, _Q, _Con, _FO, _EDy],
    [() => BoundingBox$, () => AgeRange$, () => Smile$, () => Eyeglasses$, () => Sunglasses$, () => Gender$, () => Beard$, () => Mustache$, () => EyeOpen$, () => MouthOpen$, () => Emotions, () => Landmarks, () => Pose$, () => ImageQuality$, 1, () => FaceOccluded$, () => EyeDirection$]
];
var FaceDetection$ = [3, n0, _FDac,
    0,
    [_Ti, _F],
    [1, () => FaceDetail$]
];
var FaceMatch$ = [3, n0, _FMa,
    0,
    [_Si, _F],
    [1, () => Face$]
];
var FaceOccluded$ = [3, n0, _FO,
    0,
    [_V, _Con],
    [2, 1]
];
var FaceRecord$ = [3, n0, _FR,
    0,
    [_F, _FDa],
    [() => Face$, () => FaceDetail$]
];
var FaceSearchSettings$ = [3, n0, _FSS,
    0,
    [_CI, _FMT],
    [0, 1]
];
var Gender$ = [3, n0, _Ge,
    0,
    [_V, _Con],
    [0, 1]
];
var GeneralLabelsSettings$ = [3, n0, _GLS,
    0,
    [_LIF, _LEF, _LCIF, _LCEF],
    [64 | 0, 64 | 0, 64 | 0, 64 | 0]
];
var Geometry$ = [3, n0, _G,
    0,
    [_BB, _Po],
    [() => BoundingBox$, () => Polygon]
];
var GetCelebrityInfoRequest$ = [3, n0, _GCIR,
    0,
    [_I],
    [0], 1
];
var GetCelebrityInfoResponse$ = [3, n0, _GCIRe,
    0,
    [_U, _N, _KG],
    [64 | 0, 0, () => KnownGender$]
];
var GetCelebrityRecognitionRequest$ = [3, n0, _GCRR,
    0,
    [_JI, _MR, _NT, _SB],
    [0, 1, 0, 0], 1
];
var GetCelebrityRecognitionResponse$ = [3, n0, _GCRRe,
    0,
    [_JS, _SM, _VM, _NT, _Cel, _JI, _Vi, _JT],
    [0, 0, () => VideoMetadata$, 0, () => CelebrityRecognitions, 0, () => Video$, 0]
];
var GetContentModerationRequest$ = [3, n0, _GCMR,
    0,
    [_JI, _MR, _NT, _SB, _AB],
    [0, 1, 0, 0, 0], 1
];
var GetContentModerationRequestMetadata$ = [3, n0, _GCMRM,
    0,
    [_SB, _AB],
    [0, 0]
];
var GetContentModerationResponse$ = [3, n0, _GCMRe,
    0,
    [_JS, _SM, _VM, _MLo, _NT, _MMV, _JI, _Vi, _JT, _GRM],
    [0, 0, () => VideoMetadata$, () => ContentModerationDetections, 0, 0, 0, () => Video$, 0, () => GetContentModerationRequestMetadata$]
];
var GetFaceDetectionRequest$ = [3, n0, _GFDR,
    0,
    [_JI, _MR, _NT],
    [0, 1, 0], 1
];
var GetFaceDetectionResponse$ = [3, n0, _GFDRe,
    0,
    [_JS, _SM, _VM, _NT, _Fa, _JI, _Vi, _JT],
    [0, 0, () => VideoMetadata$, 0, () => FaceDetections, 0, () => Video$, 0]
];
var GetFaceLivenessSessionResultsRequest$ = [3, n0, _GFLSRR,
    0,
    [_SIe],
    [0], 1
];
var GetFaceLivenessSessionResultsResponse$ = [3, n0, _GFLSRRe,
    0,
    [_SIe, _St, _Con, _RI, _AIu, _Ch],
    [0, 0, 1, [() => AuditImage$, 0], [() => AuditImages, 0], () => Challenge$], 2
];
var GetFaceSearchRequest$ = [3, n0, _GFSR,
    0,
    [_JI, _MR, _NT, _SB],
    [0, 1, 0, 0], 1
];
var GetFaceSearchResponse$ = [3, n0, _GFSRe,
    0,
    [_JS, _SM, _NT, _VM, _Pe, _JI, _Vi, _JT],
    [0, 0, 0, () => VideoMetadata$, () => PersonMatches, 0, () => Video$, 0]
];
var GetLabelDetectionRequest$ = [3, n0, _GLDR,
    0,
    [_JI, _MR, _NT, _SB, _AB],
    [0, 1, 0, 0, 0], 1
];
var GetLabelDetectionRequestMetadata$ = [3, n0, _GLDRM,
    0,
    [_SB, _AB],
    [0, 0]
];
var GetLabelDetectionResponse$ = [3, n0, _GLDRe,
    0,
    [_JS, _SM, _VM, _NT, _Lab, _LMV, _JI, _Vi, _JT, _GRM],
    [0, 0, () => VideoMetadata$, 0, () => LabelDetections, 0, 0, () => Video$, 0, () => GetLabelDetectionRequestMetadata$]
];
var GetMediaAnalysisJobRequest$ = [3, n0, _GMAJR,
    0,
    [_JI],
    [0], 1
];
var GetMediaAnalysisJobResponse$ = [3, n0, _GMAJRe,
    0,
    [_JI, _OCp, _St, _CTr, _In, _OC, _JN, _FDai, _CTom, _KKI, _Re, _MS],
    [0, () => MediaAnalysisOperationsConfig$, 0, 4, () => MediaAnalysisInput$, () => MediaAnalysisOutputConfig$, 0, () => MediaAnalysisJobFailureDetails$, 4, 0, () => MediaAnalysisResults$, () => MediaAnalysisManifestSummary$], 6
];
var GetPersonTrackingRequest$ = [3, n0, _GPTR,
    0,
    [_JI, _MR, _NT, _SB],
    [0, 1, 0, 0], 1
];
var GetPersonTrackingResponse$ = [3, n0, _GPTRe,
    0,
    [_JS, _SM, _VM, _NT, _Pe, _JI, _Vi, _JT],
    [0, 0, () => VideoMetadata$, 0, () => PersonDetections, 0, () => Video$, 0]
];
var GetSegmentDetectionRequest$ = [3, n0, _GSDR,
    0,
    [_JI, _MR, _NT],
    [0, 1, 0], 1
];
var GetSegmentDetectionResponse$ = [3, n0, _GSDRe,
    0,
    [_JS, _SM, _VM, _AM, _NT, _Seg, _SST, _JI, _Vi, _JT],
    [0, 0, () => VideoMetadataList, () => AudioMetadataList, 0, () => SegmentDetections, () => SegmentTypesInfo, 0, () => Video$, 0]
];
var GetTextDetectionRequest$ = [3, n0, _GTDR,
    0,
    [_JI, _MR, _NT],
    [0, 1, 0], 1
];
var GetTextDetectionResponse$ = [3, n0, _GTDRe,
    0,
    [_JS, _SM, _VM, _TDex, _NT, _TMV, _JI, _Vi, _JT],
    [0, 0, () => VideoMetadata$, () => TextDetectionResults, 0, 0, 0, () => Video$, 0]
];
var GroundTruthManifest$ = [3, n0, _GTM,
    0,
    [_SO],
    [() => S3Object$]
];
var HumanLoopActivationOutput$ = [3, n0, _HLAO,
    0,
    [_HLA, _HLAR, _HLACER],
    [0, 64 | 0, [() => SynthesizedJsonHumanLoopActivationConditionsEvaluationResults, 0]]
];
var HumanLoopConfig$ = [3, n0, _HLC,
    0,
    [_HLN, _FDA, _DAa],
    [0, 0, () => HumanLoopDataAttributes$], 2
];
var HumanLoopDataAttributes$ = [3, n0, _HLDA,
    0,
    [_CC],
    [64 | 0]
];
var HumanLoopQuotaExceededException$ = [-3, n0, _HLQEE,
    { [_e]: _c, [_hE]: 402 },
    [_RT, _QC, _SCe, _M, _C, _L],
    [0, 0, 0, 0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(HumanLoopQuotaExceededException$, HumanLoopQuotaExceededException);
var IdempotentParameterMismatchException$ = [-3, n0, _IPME,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(IdempotentParameterMismatchException$, IdempotentParameterMismatchException);
var Image$ = [3, n0, _Im,
    0,
    [_B, _SO],
    [21, () => S3Object$]
];
var ImageQuality$ = [3, n0, _IQ,
    0,
    [_Br, _Sh],
    [1, 1]
];
var ImageTooLargeException$ = [-3, n0, _ITLE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(ImageTooLargeException$, ImageTooLargeException);
var IndexFacesRequest$ = [3, n0, _IFR,
    0,
    [_CI, _Im, _EII, _DAe, _MF, _QF],
    [0, () => Image$, 0, 64 | 0, 1, 0], 2
];
var IndexFacesResponse$ = [3, n0, _IFRn,
    0,
    [_FRa, _OCr, _FMV, _UFn],
    [() => FaceRecordList, 0, 0, () => UnindexedFaces]
];
var Instance$ = [3, n0, _Ins,
    0,
    [_BB, _Con, _DCo],
    [() => BoundingBox$, 1, () => DominantColors]
];
var InternalServerError$ = [-3, n0, _ISE,
    { [_e]: _s },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(InternalServerError$, InternalServerError);
var InvalidImageFormatException$ = [-3, n0, _IIFE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(InvalidImageFormatException$, InvalidImageFormatException);
var InvalidManifestException$ = [-3, n0, _IME,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(InvalidManifestException$, InvalidManifestException);
var InvalidPaginationTokenException$ = [-3, n0, _IPTE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(InvalidPaginationTokenException$, InvalidPaginationTokenException);
var InvalidParameterException$ = [-3, n0, _IPE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(InvalidParameterException$, InvalidParameterException);
var InvalidPolicyRevisionIdException$ = [-3, n0, _IPRIE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(InvalidPolicyRevisionIdException$, InvalidPolicyRevisionIdException);
var InvalidS3ObjectException$ = [-3, n0, _ISOE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(InvalidS3ObjectException$, InvalidS3ObjectException);
var KinesisDataStream$ = [3, n0, _KDS,
    0,
    [_Ar],
    [0]
];
var KinesisVideoStream$ = [3, n0, _KVS,
    0,
    [_Ar],
    [0]
];
var KinesisVideoStreamStartSelector$ = [3, n0, _KVSSS,
    0,
    [_PT, _FN],
    [1, 0]
];
var KnownGender$ = [3, n0, _KG,
    0,
    [_Ty],
    [0]
];
var Label$ = [3, n0, _Labe,
    0,
    [_N, _Con, _Inst, _Pa, _Al, _Ca],
    [0, 1, () => Instances, () => Parents, () => LabelAliases, () => LabelCategories]
];
var LabelAlias$ = [3, n0, _LA,
    0,
    [_N],
    [0]
];
var LabelCategory$ = [3, n0, _LC,
    0,
    [_N],
    [0]
];
var LabelDetection$ = [3, n0, _LD,
    0,
    [_Ti, _Labe, _STM, _ETM, _DM],
    [1, () => Label$, 1, 1, 1]
];
var LabelDetectionSettings$ = [3, n0, _LDS,
    0,
    [_GL],
    [() => GeneralLabelsSettings$]
];
var Landmark$ = [3, n0, _Lan,
    0,
    [_Ty, _X, _Y_],
    [0, 1, 1]
];
var LimitExceededException$ = [-3, n0, _LEE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(LimitExceededException$, LimitExceededException);
var ListCollectionsRequest$ = [3, n0, _LCR,
    0,
    [_NT, _MR],
    [0, 1]
];
var ListCollectionsResponse$ = [3, n0, _LCRi,
    0,
    [_CIo, _NT, _FMVa],
    [64 | 0, 0, 64 | 0]
];
var ListDatasetEntriesRequest$ = [3, n0, _LDER,
    0,
    [_DA, _CLo, _Label, _SRC, _HE, _NT, _MR],
    [0, 64 | 0, 2, 0, 2, 0, 1], 1
];
var ListDatasetEntriesResponse$ = [3, n0, _LDERi,
    0,
    [_DE, _NT],
    [64 | 0, 0]
];
var ListDatasetLabelsRequest$ = [3, n0, _LDLR,
    0,
    [_DA, _NT, _MR],
    [0, 0, 1], 1
];
var ListDatasetLabelsResponse$ = [3, n0, _LDLRi,
    0,
    [_DLDa, _NT],
    [() => DatasetLabelDescriptions, 0]
];
var ListFacesRequest$ = [3, n0, _LFR,
    0,
    [_CI, _NT, _MR, _UI, _FIa],
    [0, 0, 1, 0, 64 | 0], 1
];
var ListFacesResponse$ = [3, n0, _LFRi,
    0,
    [_Fa, _NT, _FMV],
    [() => FaceList, 0, 0]
];
var ListMediaAnalysisJobsRequest$ = [3, n0, _LMAJR,
    0,
    [_NT, _MR],
    [0, 1]
];
var ListMediaAnalysisJobsResponse$ = [3, n0, _LMAJRi,
    0,
    [_MAJ, _NT],
    [() => MediaAnalysisJobDescriptions, 0], 1
];
var ListProjectPoliciesRequest$ = [3, n0, _LPPR,
    0,
    [_PA, _NT, _MR],
    [0, 0, 1], 1
];
var ListProjectPoliciesResponse$ = [3, n0, _LPPRi,
    0,
    [_PPr, _NT],
    [() => ProjectPolicies, 0]
];
var ListStreamProcessorsRequest$ = [3, n0, _LSPR,
    0,
    [_NT, _MR],
    [0, 1]
];
var ListStreamProcessorsResponse$ = [3, n0, _LSPRi,
    0,
    [_NT, _SP],
    [0, () => StreamProcessorList]
];
var ListTagsForResourceRequest$ = [3, n0, _LTFRR,
    0,
    [_RAe],
    [0], 1
];
var ListTagsForResourceResponse$ = [3, n0, _LTFRRi,
    0,
    [_Ta],
    [128 | 0]
];
var ListUsersRequest$ = [3, n0, _LUR,
    0,
    [_CI, _MR, _NT],
    [0, 1, 0], 1
];
var ListUsersResponse$ = [3, n0, _LURi,
    0,
    [_Us, _NT],
    [() => UserList, 0]
];
var LivenessOutputConfig$ = [3, n0, _LOC,
    0,
    [_SBu, _SKP],
    [0, 0], 1
];
var MalformedPolicyDocumentException$ = [-3, n0, _MPDE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(MalformedPolicyDocumentException$, MalformedPolicyDocumentException);
var MatchedUser$ = [3, n0, _MU,
    0,
    [_UI, _US],
    [0, 0]
];
var MediaAnalysisDetectModerationLabelsConfig$ = [3, n0, _MADMLC,
    0,
    [_MCi, _PV],
    [1, 0]
];
var MediaAnalysisInput$ = [3, n0, _MAI,
    0,
    [_SO],
    [() => S3Object$], 1
];
var MediaAnalysisJobDescription$ = [3, n0, _MAJD,
    0,
    [_JI, _OCp, _St, _CTr, _In, _OC, _JN, _FDai, _CTom, _KKI, _Re, _MS],
    [0, () => MediaAnalysisOperationsConfig$, 0, 4, () => MediaAnalysisInput$, () => MediaAnalysisOutputConfig$, 0, () => MediaAnalysisJobFailureDetails$, 4, 0, () => MediaAnalysisResults$, () => MediaAnalysisManifestSummary$], 6
];
var MediaAnalysisJobFailureDetails$ = [3, n0, _MAJFD,
    0,
    [_C, _M],
    [0, 0]
];
var MediaAnalysisManifestSummary$ = [3, n0, _MAMS,
    0,
    [_SO],
    [() => S3Object$]
];
var MediaAnalysisModelVersions$ = [3, n0, _MAMV,
    0,
    [_Mo],
    [0]
];
var MediaAnalysisOperationsConfig$ = [3, n0, _MAOC,
    0,
    [_DML],
    [() => MediaAnalysisDetectModerationLabelsConfig$]
];
var MediaAnalysisOutputConfig$ = [3, n0, _MAOCe,
    0,
    [_SBu, _SKP],
    [0, 0], 1
];
var MediaAnalysisResults$ = [3, n0, _MAR,
    0,
    [_SO, _MV],
    [() => S3Object$, () => MediaAnalysisModelVersions$]
];
var ModerationLabel$ = [3, n0, _ML,
    0,
    [_Con, _N, _PNa, _TLa],
    [1, 0, 0, 1]
];
var MouthOpen$ = [3, n0, _MO,
    0,
    [_V, _Con],
    [2, 1]
];
var Mustache$ = [3, n0, _Mu,
    0,
    [_V, _Con],
    [2, 1]
];
var NotificationChannel$ = [3, n0, _NC,
    0,
    [_SNSTA, _RA],
    [0, 0], 2
];
var OutputConfig$ = [3, n0, _OC,
    0,
    [_SBu, _SKP],
    [0, 0]
];
var Parent$ = [3, n0, _Par,
    0,
    [_N],
    [0]
];
var PersonDetail$ = [3, n0, _PDe,
    0,
    [_Ind, _BB, _F],
    [1, () => BoundingBox$, () => FaceDetail$]
];
var PersonDetection$ = [3, n0, _PDer,
    0,
    [_Ti, _Per],
    [1, () => PersonDetail$]
];
var PersonMatch$ = [3, n0, _PM,
    0,
    [_Ti, _Per, _FM],
    [1, () => PersonDetail$, () => FaceMatchList]
];
var Point$ = [3, n0, _Poi,
    0,
    [_X, _Y_],
    [1, 1]
];
var Pose$ = [3, n0, _P,
    0,
    [_Ro, _Y, _Pi],
    [1, 1, 1]
];
var ProjectDescription$ = [3, n0, _PDr,
    0,
    [_PA, _CTr, _St, _D, _Fe, _AU],
    [0, 4, 0, () => DatasetMetadataList, 0, 0]
];
var ProjectPolicy$ = [3, n0, _PPro,
    0,
    [_PA, _PNo, _PRI, _PDo, _CTr, _LUT],
    [0, 0, 0, 0, 4, 4]
];
var ProjectVersionDescription$ = [3, n0, _PVDr,
    0,
    [_PVA, _CTr, _MIU, _St, _SM, _BTTIS, _TET, _OC, _TDR, _TDRe, _ER, _MS, _KKI, _MIUa, _SPVA, _VD, _Fe, _BMV, _FC],
    [0, 4, 1, 0, 0, 1, 4, () => OutputConfig$, () => TrainingDataResult$, () => TestingDataResult$, () => EvaluationResult$, () => GroundTruthManifest$, 0, 1, 0, 0, 0, 0, () => CustomizationFeatureConfig$]
];
var ProtectiveEquipmentBodyPart$ = [3, n0, _PEBP,
    0,
    [_N, _Con, _EDq],
    [0, 1, () => EquipmentDetections]
];
var ProtectiveEquipmentPerson$ = [3, n0, _PEP,
    0,
    [_BP, _BB, _Con, _I],
    [() => BodyParts, () => BoundingBox$, 1, 1]
];
var ProtectiveEquipmentSummarizationAttributes$ = [3, n0, _PESA,
    0,
    [_MCi, _RET],
    [1, 64 | 0], 2
];
var ProtectiveEquipmentSummary$ = [3, n0, _PES,
    0,
    [_PWRE, _PWREe, _PI],
    [64 | 1, 64 | 1, 64 | 1]
];
var ProvisionedThroughputExceededException$ = [-3, n0, _PTEE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(ProvisionedThroughputExceededException$, ProvisionedThroughputExceededException);
var PutProjectPolicyRequest$ = [3, n0, _PPPR,
    0,
    [_PA, _PNo, _PDo, _PRI],
    [0, 0, 0, 0], 3
];
var PutProjectPolicyResponse$ = [3, n0, _PPPRu,
    0,
    [_PRI],
    [0]
];
var RecognizeCelebritiesRequest$ = [3, n0, _RCR,
    0,
    [_Im],
    [() => Image$], 1
];
var RecognizeCelebritiesResponse$ = [3, n0, _RCRe,
    0,
    [_CFe, _UFnr, _OCr],
    [() => CelebrityList, () => ComparedFaceList, 0]
];
var RegionOfInterest$ = [3, n0, _ROIe,
    0,
    [_BB, _Po],
    [() => BoundingBox$, () => Polygon]
];
var ResourceAlreadyExistsException$ = [-3, n0, _RAEE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(ResourceAlreadyExistsException$, ResourceAlreadyExistsException);
var ResourceInUseException$ = [-3, n0, _RIUE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(ResourceInUseException$, ResourceInUseException);
var ResourceNotFoundException$ = [-3, n0, _RNFE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(ResourceNotFoundException$, ResourceNotFoundException);
var ResourceNotReadyException$ = [-3, n0, _RNRE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(ResourceNotReadyException$, ResourceNotReadyException);
var S3Destination$ = [3, n0, _SD,
    0,
    [_Bu, _KP],
    [0, 0]
];
var S3Object$ = [3, n0, _SO,
    0,
    [_Bu, _N, _Ve],
    [0, 0, 0]
];
var SearchedFace$ = [3, n0, _SF,
    0,
    [_FI],
    [0]
];
var SearchedFaceDetails$ = [3, n0, _SFD,
    0,
    [_FDa],
    [() => FaceDetail$]
];
var SearchedUser$ = [3, n0, _SU,
    0,
    [_UI],
    [0]
];
var SearchFacesByImageRequest$ = [3, n0, _SFBIR,
    0,
    [_CI, _Im, _MF, _FMT, _QF],
    [0, () => Image$, 1, 1, 0], 2
];
var SearchFacesByImageResponse$ = [3, n0, _SFBIRe,
    0,
    [_SFBB, _SFC, _FM, _FMV],
    [() => BoundingBox$, 1, () => FaceMatchList, 0]
];
var SearchFacesRequest$ = [3, n0, _SFR,
    0,
    [_CI, _FI, _MF, _FMT],
    [0, 0, 1, 1], 2
];
var SearchFacesResponse$ = [3, n0, _SFRe,
    0,
    [_SFI, _FM, _FMV],
    [0, () => FaceMatchList, 0]
];
var SearchUsersByImageRequest$ = [3, n0, _SUBIR,
    0,
    [_CI, _Im, _UMT, _MUa, _QF],
    [0, () => Image$, 1, 1, 0], 2
];
var SearchUsersByImageResponse$ = [3, n0, _SUBIRe,
    0,
    [_UM, _FMV, _SF, _UFns],
    [() => UserMatchList, 0, () => SearchedFaceDetails$, () => UnsearchedFacesList]
];
var SearchUsersRequest$ = [3, n0, _SUR,
    0,
    [_CI, _UI, _FI, _UMT, _MUa],
    [0, 0, 0, 1, 1], 1
];
var SearchUsersResponse$ = [3, n0, _SURe,
    0,
    [_UM, _FMV, _SF, _SU],
    [() => UserMatchList, 0, () => SearchedFace$, () => SearchedUser$]
];
var SegmentDetection$ = [3, n0, _SDe,
    0,
    [_Ty, _STM, _ETM, _DM, _STSMPTE, _ETSMPTE, _DSMPTE, _TCS, _SS, _SFN, _EFN, _DFu],
    [0, 1, 1, 1, 0, 0, 0, () => TechnicalCueSegment$, () => ShotSegment$, 1, 1, 1]
];
var SegmentTypeInfo$ = [3, n0, _STI,
    0,
    [_Ty, _MVo],
    [0, 0]
];
var ServiceQuotaExceededException$ = [-3, n0, _SQEE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(ServiceQuotaExceededException$, ServiceQuotaExceededException);
var SessionNotFoundException$ = [-3, n0, _SNFE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(SessionNotFoundException$, SessionNotFoundException);
var ShotSegment$ = [3, n0, _SS,
    0,
    [_Ind, _Con],
    [1, 1]
];
var Smile$ = [3, n0, _S,
    0,
    [_V, _Con],
    [2, 1]
];
var StartCelebrityRecognitionRequest$ = [3, n0, _SCRR,
    0,
    [_Vi, _CRT, _NC, _JT],
    [() => Video$, 0, () => NotificationChannel$, 0], 1
];
var StartCelebrityRecognitionResponse$ = [3, n0, _SCRRt,
    0,
    [_JI],
    [0]
];
var StartContentModerationRequest$ = [3, n0, _SCMR,
    0,
    [_Vi, _MCi, _CRT, _NC, _JT],
    [() => Video$, 1, 0, () => NotificationChannel$, 0], 1
];
var StartContentModerationResponse$ = [3, n0, _SCMRt,
    0,
    [_JI],
    [0]
];
var StartFaceDetectionRequest$ = [3, n0, _SFDR,
    0,
    [_Vi, _CRT, _NC, _FA, _JT],
    [() => Video$, 0, () => NotificationChannel$, 0, 0], 1
];
var StartFaceDetectionResponse$ = [3, n0, _SFDRt,
    0,
    [_JI],
    [0]
];
var StartFaceSearchRequest$ = [3, n0, _SFSR,
    0,
    [_Vi, _CI, _CRT, _FMT, _NC, _JT],
    [() => Video$, 0, 0, 1, () => NotificationChannel$, 0], 2
];
var StartFaceSearchResponse$ = [3, n0, _SFSRt,
    0,
    [_JI],
    [0]
];
var StartLabelDetectionRequest$ = [3, n0, _SLDR,
    0,
    [_Vi, _CRT, _MCi, _NC, _JT, _Fea, _Se],
    [() => Video$, 0, 1, () => NotificationChannel$, 0, 64 | 0, () => LabelDetectionSettings$], 1
];
var StartLabelDetectionResponse$ = [3, n0, _SLDRt,
    0,
    [_JI],
    [0]
];
var StartMediaAnalysisJobRequest$ = [3, n0, _SMAJR,
    0,
    [_OCp, _In, _OC, _CRT, _JN, _KKI],
    [() => MediaAnalysisOperationsConfig$, () => MediaAnalysisInput$, () => MediaAnalysisOutputConfig$, [0, 4], 0, 0], 3
];
var StartMediaAnalysisJobResponse$ = [3, n0, _SMAJRt,
    0,
    [_JI],
    [0], 1
];
var StartPersonTrackingRequest$ = [3, n0, _SPTR,
    0,
    [_Vi, _CRT, _NC, _JT],
    [() => Video$, 0, () => NotificationChannel$, 0], 1
];
var StartPersonTrackingResponse$ = [3, n0, _SPTRt,
    0,
    [_JI],
    [0]
];
var StartProjectVersionRequest$ = [3, n0, _SPVR,
    0,
    [_PVA, _MIU, _MIUa],
    [0, 1, 1], 2
];
var StartProjectVersionResponse$ = [3, n0, _SPVRt,
    0,
    [_St],
    [0]
];
var StartSegmentDetectionFilters$ = [3, n0, _SSDF,
    0,
    [_TCF, _SFh],
    [() => StartTechnicalCueDetectionFilter$, () => StartShotDetectionFilter$]
];
var StartSegmentDetectionRequest$ = [3, n0, _SSDR,
    0,
    [_Vi, _STe, _CRT, _NC, _JT, _Fi],
    [() => Video$, 64 | 0, 0, () => NotificationChannel$, 0, () => StartSegmentDetectionFilters$], 2
];
var StartSegmentDetectionResponse$ = [3, n0, _SSDRt,
    0,
    [_JI],
    [0]
];
var StartShotDetectionFilter$ = [3, n0, _SSDFt,
    0,
    [_MSC],
    [1]
];
var StartStreamProcessorRequest$ = [3, n0, _SSPR,
    0,
    [_N, _SSt, _SSto],
    [0, () => StreamProcessingStartSelector$, () => StreamProcessingStopSelector$], 1
];
var StartStreamProcessorResponse$ = [3, n0, _SSPRt,
    0,
    [_SIe],
    [0]
];
var StartTechnicalCueDetectionFilter$ = [3, n0, _STCDF,
    0,
    [_MSC, _BF],
    [1, () => BlackFrame$]
];
var StartTextDetectionFilters$ = [3, n0, _STDF,
    0,
    [_WF, _ROI],
    [() => DetectionFilter$, () => RegionsOfInterest]
];
var StartTextDetectionRequest$ = [3, n0, _STDR,
    0,
    [_Vi, _CRT, _NC, _JT, _Fi],
    [() => Video$, 0, () => NotificationChannel$, 0, () => StartTextDetectionFilters$], 1
];
var StartTextDetectionResponse$ = [3, n0, _STDRt,
    0,
    [_JI],
    [0]
];
var StopProjectVersionRequest$ = [3, n0, _SPVRto,
    0,
    [_PVA],
    [0], 1
];
var StopProjectVersionResponse$ = [3, n0, _SPVRtop,
    0,
    [_St],
    [0]
];
var StopStreamProcessorRequest$ = [3, n0, _SSPRto,
    0,
    [_N],
    [0], 1
];
var StopStreamProcessorResponse$ = [3, n0, _SSPRtop,
    0,
    [],
    []
];
var StreamProcessingStartSelector$ = [3, n0, _SPSS,
    0,
    [_KVSSSS],
    [() => KinesisVideoStreamStartSelector$]
];
var StreamProcessingStopSelector$ = [3, n0, _SPSSt,
    0,
    [_MDIS],
    [1]
];
var StreamProcessor$ = [3, n0, _SPt,
    0,
    [_N, _St],
    [0, 0]
];
var StreamProcessorDataSharingPreference$ = [3, n0, _SPDSP,
    0,
    [_OI],
    [2], 1
];
var StreamProcessorInput$ = [3, n0, _SPI,
    0,
    [_KVS],
    [() => KinesisVideoStream$]
];
var StreamProcessorNotificationChannel$ = [3, n0, _SPNC,
    0,
    [_SNSTA],
    [0], 1
];
var StreamProcessorOutput$ = [3, n0, _SPO,
    0,
    [_KDS, _SD],
    [() => KinesisDataStream$, () => S3Destination$]
];
var StreamProcessorSettings$ = [3, n0, _SPS,
    0,
    [_FSa, _CH],
    [() => FaceSearchSettings$, () => ConnectedHomeSettings$]
];
var StreamProcessorSettingsForUpdate$ = [3, n0, _SPSFU,
    0,
    [_CHFU],
    [() => ConnectedHomeSettingsForUpdate$]
];
var Summary$ = [3, n0, _Su,
    0,
    [_SO],
    [() => S3Object$]
];
var Sunglasses$ = [3, n0, _Sun,
    0,
    [_V, _Con],
    [2, 1]
];
var TagResourceRequest$ = [3, n0, _TRR,
    0,
    [_RAe, _Ta],
    [0, 128 | 0], 2
];
var TagResourceResponse$ = [3, n0, _TRRa,
    0,
    [],
    []
];
var TechnicalCueSegment$ = [3, n0, _TCS,
    0,
    [_Ty, _Con],
    [0, 1]
];
var TestingData$ = [3, n0, _TDe,
    0,
    [_As, _AC],
    [() => Assets, 2]
];
var TestingDataResult$ = [3, n0, _TDRe,
    0,
    [_In, _O, _Va],
    [() => TestingData$, () => TestingData$, () => ValidationData$]
];
var TextDetection$ = [3, n0, _TDext,
    0,
    [_DTe, _Ty, _I, _PIa, _Con, _G],
    [0, 0, 1, 1, 1, () => Geometry$]
];
var TextDetectionResult$ = [3, n0, _TDRex,
    0,
    [_Ti, _TDext],
    [1, () => TextDetection$]
];
var ThrottlingException$ = [-3, n0, _TEh,
    { [_e]: _s },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(ThrottlingException$, ThrottlingException);
var TrainingData$ = [3, n0, _TD,
    0,
    [_As],
    [() => Assets]
];
var TrainingDataResult$ = [3, n0, _TDR,
    0,
    [_In, _O, _Va],
    [() => TrainingData$, () => TrainingData$, () => ValidationData$]
];
var UnindexedFace$ = [3, n0, _UFni,
    0,
    [_Rea, _FDa],
    [64 | 0, () => FaceDetail$]
];
var UnsearchedFace$ = [3, n0, _UFnse,
    0,
    [_FD, _Rea],
    [() => FaceDetail$, 64 | 0]
];
var UnsuccessfulFaceAssociation$ = [3, n0, _UFAn,
    0,
    [_FI, _UI, _Con, _Rea],
    [0, 0, 1, 64 | 0]
];
var UnsuccessfulFaceDeletion$ = [3, n0, _UFDns,
    0,
    [_FI, _UI, _Rea],
    [0, 0, 64 | 0]
];
var UnsuccessfulFaceDisassociation$ = [3, n0, _UFDnsu,
    0,
    [_FI, _UI, _Rea],
    [0, 0, 64 | 0]
];
var UntagResourceRequest$ = [3, n0, _URR,
    0,
    [_RAe, _TK],
    [0, 64 | 0], 2
];
var UntagResourceResponse$ = [3, n0, _URRn,
    0,
    [],
    []
];
var UpdateDatasetEntriesRequest$ = [3, n0, _UDER,
    0,
    [_DA, _Cha],
    [0, () => DatasetChanges$], 2
];
var UpdateDatasetEntriesResponse$ = [3, n0, _UDERp,
    0,
    [],
    []
];
var UpdateStreamProcessorRequest$ = [3, n0, _USPR,
    0,
    [_N, _SFU, _ROIFU, _DSPFU, _PTD],
    [0, () => StreamProcessorSettingsForUpdate$, () => RegionsOfInterest, () => StreamProcessorDataSharingPreference$, 64 | 0], 1
];
var UpdateStreamProcessorResponse$ = [3, n0, _USPRp,
    0,
    [],
    []
];
var User$ = [3, n0, _Use,
    0,
    [_UI, _US],
    [0, 0]
];
var UserMatch$ = [3, n0, _UMs,
    0,
    [_Si, _Use],
    [1, () => MatchedUser$]
];
var ValidationData$ = [3, n0, _VDa,
    0,
    [_As],
    [() => Assets]
];
var Versions$ = [3, n0, _Ver,
    0,
    [_Mi, _Ma],
    [0, 0]
];
var Video$ = [3, n0, _Vi,
    0,
    [_SO],
    [() => S3Object$]
];
var VideoMetadata$ = [3, n0, _VM,
    0,
    [_Co, _DM, _For, _FRr, _FH, _FW, _CRo],
    [0, 1, 0, 1, 1, 1, 0]
];
var VideoTooLargeException$ = [-3, n0, _VTLE,
    { [_e]: _c },
    [_M, _C, _L],
    [0, 0, 0]
];
schema.TypeRegistry.for(n0).registerError(VideoTooLargeException$, VideoTooLargeException);
var RekognitionServiceException$ = [-3, _sm, "RekognitionServiceException", 0, [], []];
schema.TypeRegistry.for(_sm).registerError(RekognitionServiceException$, RekognitionServiceException);
var Assets = [1, n0, _As,
    0, () => Asset$
];
var AssociatedFacesList = [1, n0, _AFL,
    0, () => AssociatedFace$
];
var AudioMetadataList = [1, n0, _AML,
    0, () => AudioMetadata$
];
var AuditImages = [1, n0, _AIu,
    0, [() => AuditImage$,
        0]
];
var BodyParts = [1, n0, _BP,
    0, () => ProtectiveEquipmentBodyPart$
];
var CelebrityList = [1, n0, _CLe,
    0, () => Celebrity$
];
var CelebrityRecognitions = [1, n0, _CRe,
    0, () => CelebrityRecognition$
];
var ChallengePreferences = [1, n0, _CPh,
    0, () => ChallengePreference$
];
var ComparedFaceList = [1, n0, _CFL,
    0, () => ComparedFace$
];
var CompareFacesMatchList = [1, n0, _CFML,
    0, () => CompareFacesMatch$
];
var CompareFacesUnmatchList = [1, n0, _CFUL,
    0, () => ComparedFace$
];
var ContentModerationDetections = [1, n0, _CMDo,
    0, () => ContentModerationDetection$
];
var ContentTypes = [1, n0, _CT,
    0, () => ContentType$
];
var CustomLabels = [1, n0, _CLu,
    0, () => CustomLabel$
];
var DatasetLabelDescriptions = [1, n0, _DLDa,
    0, () => DatasetLabelDescription$
];
var DatasetMetadataList = [1, n0, _DMLa,
    0, () => DatasetMetadata$
];
var DisassociatedFacesList = [1, n0, _DFL,
    0, () => DisassociatedFace$
];
var DistributeDatasetMetadataList = [1, n0, _DDML,
    0, () => DistributeDataset$
];
var DominantColors = [1, n0, _DCo,
    0, () => DominantColor$
];
var Emotions = [1, n0, _E,
    0, () => Emotion$
];
var EquipmentDetections = [1, n0, _EDq,
    0, () => EquipmentDetection$
];
var FaceDetailList = [1, n0, _FDL,
    0, () => FaceDetail$
];
var FaceDetections = [1, n0, _FDace,
    0, () => FaceDetection$
];
var FaceList = [1, n0, _FL,
    0, () => Face$
];
var FaceMatchList = [1, n0, _FML,
    0, () => FaceMatch$
];
var FaceRecordList = [1, n0, _FRL,
    0, () => FaceRecord$
];
var Instances = [1, n0, _Inst,
    0, () => Instance$
];
var LabelAliases = [1, n0, _LAa,
    0, () => LabelAlias$
];
var LabelCategories = [1, n0, _LCa,
    0, () => LabelCategory$
];
var LabelDetections = [1, n0, _LDa,
    0, () => LabelDetection$
];
var Labels = [1, n0, _Lab,
    0, () => Label$
];
var Landmarks = [1, n0, _La,
    0, () => Landmark$
];
var MediaAnalysisJobDescriptions = [1, n0, _MAJDe,
    0, () => MediaAnalysisJobDescription$
];
var ModerationLabels = [1, n0, _MLo,
    0, () => ModerationLabel$
];
var Parents = [1, n0, _Pa,
    0, () => Parent$
];
var PersonDetections = [1, n0, _PDers,
    0, () => PersonDetection$
];
var PersonMatches = [1, n0, _PMe,
    0, () => PersonMatch$
];
var Polygon = [1, n0, _Po,
    0, () => Point$
];
var ProjectDescriptions = [1, n0, _PD,
    0, () => ProjectDescription$
];
var ProjectPolicies = [1, n0, _PPr,
    0, () => ProjectPolicy$
];
var ProjectVersionDescriptions = [1, n0, _PVD,
    0, () => ProjectVersionDescription$
];
var ProtectiveEquipmentPersons = [1, n0, _PEPr,
    0, () => ProtectiveEquipmentPerson$
];
var RegionsOfInterest = [1, n0, _ROI,
    0, () => RegionOfInterest$
];
var SegmentDetections = [1, n0, _SDeg,
    0, () => SegmentDetection$
];
var SegmentTypesInfo = [1, n0, _STIe,
    0, () => SegmentTypeInfo$
];
var StreamProcessorList = [1, n0, _SPL,
    0, () => StreamProcessor$
];
var TextDetectionList = [1, n0, _TDL,
    0, () => TextDetection$
];
var TextDetectionResults = [1, n0, _TDRext,
    0, () => TextDetectionResult$
];
var UnindexedFaces = [1, n0, _UFn,
    0, () => UnindexedFace$
];
var UnsearchedFacesList = [1, n0, _UFL,
    0, () => UnsearchedFace$
];
var UnsuccessfulFaceAssociationList = [1, n0, _UFAL,
    0, () => UnsuccessfulFaceAssociation$
];
var UnsuccessfulFaceDeletionsList = [1, n0, _UFDL,
    0, () => UnsuccessfulFaceDeletion$
];
var UnsuccessfulFaceDisassociationList = [1, n0, _UFDLn,
    0, () => UnsuccessfulFaceDisassociation$
];
var UserList = [1, n0, _UL,
    0, () => User$
];
var UserMatchList = [1, n0, _UML,
    0, () => UserMatch$
];
var VideoMetadataList = [1, n0, _VML,
    0, () => VideoMetadata$
];
var AssociateFaces$ = [9, n0, _AFss,
    0, () => AssociateFacesRequest$, () => AssociateFacesResponse$
];
var CompareFaces$ = [9, n0, _CFo,
    0, () => CompareFacesRequest$, () => CompareFacesResponse$
];
var CopyProjectVersion$ = [9, n0, _CPV,
    0, () => CopyProjectVersionRequest$, () => CopyProjectVersionResponse$
];
var CreateCollection$ = [9, n0, _CCr,
    0, () => CreateCollectionRequest$, () => CreateCollectionResponse$
];
var CreateDataset$ = [9, n0, _CDr,
    0, () => CreateDatasetRequest$, () => CreateDatasetResponse$
];
var CreateFaceLivenessSession$ = [9, n0, _CFLS,
    2, () => CreateFaceLivenessSessionRequest$, () => CreateFaceLivenessSessionResponse$
];
var CreateProject$ = [9, n0, _CPr,
    0, () => CreateProjectRequest$, () => CreateProjectResponse$
];
var CreateProjectVersion$ = [9, n0, _CPVr,
    0, () => CreateProjectVersionRequest$, () => CreateProjectVersionResponse$
];
var CreateStreamProcessor$ = [9, n0, _CSP,
    0, () => CreateStreamProcessorRequest$, () => CreateStreamProcessorResponse$
];
var CreateUser$ = [9, n0, _CU,
    0, () => CreateUserRequest$, () => CreateUserResponse$
];
var DeleteCollection$ = [9, n0, _DCe,
    0, () => DeleteCollectionRequest$, () => DeleteCollectionResponse$
];
var DeleteDataset$ = [9, n0, _DDe,
    0, () => DeleteDatasetRequest$, () => DeleteDatasetResponse$
];
var DeleteFaces$ = [9, n0, _DFel,
    0, () => DeleteFacesRequest$, () => DeleteFacesResponse$
];
var DeleteProject$ = [9, n0, _DP,
    0, () => DeleteProjectRequest$, () => DeleteProjectResponse$
];
var DeleteProjectPolicy$ = [9, n0, _DPP,
    0, () => DeleteProjectPolicyRequest$, () => DeleteProjectPolicyResponse$
];
var DeleteProjectVersion$ = [9, n0, _DPV,
    0, () => DeleteProjectVersionRequest$, () => DeleteProjectVersionResponse$
];
var DeleteStreamProcessor$ = [9, n0, _DSPe,
    0, () => DeleteStreamProcessorRequest$, () => DeleteStreamProcessorResponse$
];
var DeleteUser$ = [9, n0, _DU,
    0, () => DeleteUserRequest$, () => DeleteUserResponse$
];
var DescribeCollection$ = [9, n0, _DCes,
    0, () => DescribeCollectionRequest$, () => DescribeCollectionResponse$
];
var DescribeDataset$ = [9, n0, _DDes,
    0, () => DescribeDatasetRequest$, () => DescribeDatasetResponse$
];
var DescribeProjects$ = [9, n0, _DPe,
    0, () => DescribeProjectsRequest$, () => DescribeProjectsResponse$
];
var DescribeProjectVersions$ = [9, n0, _DPVe,
    0, () => DescribeProjectVersionsRequest$, () => DescribeProjectVersionsResponse$
];
var DescribeStreamProcessor$ = [9, n0, _DSPes,
    0, () => DescribeStreamProcessorRequest$, () => DescribeStreamProcessorResponse$
];
var DetectCustomLabels$ = [9, n0, _DCL,
    0, () => DetectCustomLabelsRequest$, () => DetectCustomLabelsResponse$
];
var DetectFaces$ = [9, n0, _DFet,
    0, () => DetectFacesRequest$, () => DetectFacesResponse$
];
var DetectLabels$ = [9, n0, _DL,
    0, () => DetectLabelsRequest$, () => DetectLabelsResponse$
];
var DetectModerationLabels$ = [9, n0, _DML,
    0, () => DetectModerationLabelsRequest$, () => DetectModerationLabelsResponse$
];
var DetectProtectiveEquipment$ = [9, n0, _DPE,
    0, () => DetectProtectiveEquipmentRequest$, () => DetectProtectiveEquipmentResponse$
];
var DetectText$ = [9, n0, _DTet,
    0, () => DetectTextRequest$, () => DetectTextResponse$
];
var DisassociateFaces$ = [9, n0, _DFisa,
    0, () => DisassociateFacesRequest$, () => DisassociateFacesResponse$
];
var DistributeDatasetEntries$ = [9, n0, _DDE,
    0, () => DistributeDatasetEntriesRequest$, () => DistributeDatasetEntriesResponse$
];
var GetCelebrityInfo$ = [9, n0, _GCI,
    0, () => GetCelebrityInfoRequest$, () => GetCelebrityInfoResponse$
];
var GetCelebrityRecognition$ = [9, n0, _GCR,
    0, () => GetCelebrityRecognitionRequest$, () => GetCelebrityRecognitionResponse$
];
var GetContentModeration$ = [9, n0, _GCM,
    0, () => GetContentModerationRequest$, () => GetContentModerationResponse$
];
var GetFaceDetection$ = [9, n0, _GFD,
    0, () => GetFaceDetectionRequest$, () => GetFaceDetectionResponse$
];
var GetFaceLivenessSessionResults$ = [9, n0, _GFLSR,
    0, () => GetFaceLivenessSessionResultsRequest$, () => GetFaceLivenessSessionResultsResponse$
];
var GetFaceSearch$ = [9, n0, _GFS,
    0, () => GetFaceSearchRequest$, () => GetFaceSearchResponse$
];
var GetLabelDetection$ = [9, n0, _GLD,
    0, () => GetLabelDetectionRequest$, () => GetLabelDetectionResponse$
];
var GetMediaAnalysisJob$ = [9, n0, _GMAJ,
    0, () => GetMediaAnalysisJobRequest$, () => GetMediaAnalysisJobResponse$
];
var GetPersonTracking$ = [9, n0, _GPT,
    0, () => GetPersonTrackingRequest$, () => GetPersonTrackingResponse$
];
var GetSegmentDetection$ = [9, n0, _GSD,
    0, () => GetSegmentDetectionRequest$, () => GetSegmentDetectionResponse$
];
var GetTextDetection$ = [9, n0, _GTD,
    0, () => GetTextDetectionRequest$, () => GetTextDetectionResponse$
];
var IndexFaces$ = [9, n0, _IF,
    0, () => IndexFacesRequest$, () => IndexFacesResponse$
];
var ListCollections$ = [9, n0, _LCi,
    0, () => ListCollectionsRequest$, () => ListCollectionsResponse$
];
var ListDatasetEntries$ = [9, n0, _LDE,
    0, () => ListDatasetEntriesRequest$, () => ListDatasetEntriesResponse$
];
var ListDatasetLabels$ = [9, n0, _LDL,
    0, () => ListDatasetLabelsRequest$, () => ListDatasetLabelsResponse$
];
var ListFaces$ = [9, n0, _LF,
    0, () => ListFacesRequest$, () => ListFacesResponse$
];
var ListMediaAnalysisJobs$ = [9, n0, _LMAJ,
    0, () => ListMediaAnalysisJobsRequest$, () => ListMediaAnalysisJobsResponse$
];
var ListProjectPolicies$ = [9, n0, _LPP,
    0, () => ListProjectPoliciesRequest$, () => ListProjectPoliciesResponse$
];
var ListStreamProcessors$ = [9, n0, _LSP,
    0, () => ListStreamProcessorsRequest$, () => ListStreamProcessorsResponse$
];
var ListTagsForResource$ = [9, n0, _LTFR,
    0, () => ListTagsForResourceRequest$, () => ListTagsForResourceResponse$
];
var ListUsers$ = [9, n0, _LU,
    0, () => ListUsersRequest$, () => ListUsersResponse$
];
var PutProjectPolicy$ = [9, n0, _PPP,
    0, () => PutProjectPolicyRequest$, () => PutProjectPolicyResponse$
];
var RecognizeCelebrities$ = [9, n0, _RC,
    0, () => RecognizeCelebritiesRequest$, () => RecognizeCelebritiesResponse$
];
var SearchFaces$ = [9, n0, _SFe,
    0, () => SearchFacesRequest$, () => SearchFacesResponse$
];
var SearchFacesByImage$ = [9, n0, _SFBI,
    0, () => SearchFacesByImageRequest$, () => SearchFacesByImageResponse$
];
var SearchUsers$ = [9, n0, _SUe,
    0, () => SearchUsersRequest$, () => SearchUsersResponse$
];
var SearchUsersByImage$ = [9, n0, _SUBI,
    0, () => SearchUsersByImageRequest$, () => SearchUsersByImageResponse$
];
var StartCelebrityRecognition$ = [9, n0, _SCR,
    2, () => StartCelebrityRecognitionRequest$, () => StartCelebrityRecognitionResponse$
];
var StartContentModeration$ = [9, n0, _SCM,
    2, () => StartContentModerationRequest$, () => StartContentModerationResponse$
];
var StartFaceDetection$ = [9, n0, _SFDt,
    2, () => StartFaceDetectionRequest$, () => StartFaceDetectionResponse$
];
var StartFaceSearch$ = [9, n0, _SFS,
    2, () => StartFaceSearchRequest$, () => StartFaceSearchResponse$
];
var StartLabelDetection$ = [9, n0, _SLD,
    2, () => StartLabelDetectionRequest$, () => StartLabelDetectionResponse$
];
var StartMediaAnalysisJob$ = [9, n0, _SMAJ,
    2, () => StartMediaAnalysisJobRequest$, () => StartMediaAnalysisJobResponse$
];
var StartPersonTracking$ = [9, n0, _SPT,
    2, () => StartPersonTrackingRequest$, () => StartPersonTrackingResponse$
];
var StartProjectVersion$ = [9, n0, _SPV,
    0, () => StartProjectVersionRequest$, () => StartProjectVersionResponse$
];
var StartSegmentDetection$ = [9, n0, _SSD,
    2, () => StartSegmentDetectionRequest$, () => StartSegmentDetectionResponse$
];
var StartStreamProcessor$ = [9, n0, _SSP,
    0, () => StartStreamProcessorRequest$, () => StartStreamProcessorResponse$
];
var StartTextDetection$ = [9, n0, _STD,
    2, () => StartTextDetectionRequest$, () => StartTextDetectionResponse$
];
var StopProjectVersion$ = [9, n0, _SPVt,
    0, () => StopProjectVersionRequest$, () => StopProjectVersionResponse$
];
var StopStreamProcessor$ = [9, n0, _SSPt,
    0, () => StopStreamProcessorRequest$, () => StopStreamProcessorResponse$
];
var TagResource$ = [9, n0, _TR,
    0, () => TagResourceRequest$, () => TagResourceResponse$
];
var UntagResource$ = [9, n0, _UR,
    0, () => UntagResourceRequest$, () => UntagResourceResponse$
];
var UpdateDatasetEntries$ = [9, n0, _UDE,
    0, () => UpdateDatasetEntriesRequest$, () => UpdateDatasetEntriesResponse$
];
var UpdateStreamProcessor$ = [9, n0, _USP,
    0, () => UpdateStreamProcessorRequest$, () => UpdateStreamProcessorResponse$
];

class AssociateFacesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "AssociateFaces", {})
    .n("RekognitionClient", "AssociateFacesCommand")
    .sc(AssociateFaces$)
    .build() {
}

class CompareFacesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "CompareFaces", {})
    .n("RekognitionClient", "CompareFacesCommand")
    .sc(CompareFaces$)
    .build() {
}

class CopyProjectVersionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "CopyProjectVersion", {})
    .n("RekognitionClient", "CopyProjectVersionCommand")
    .sc(CopyProjectVersion$)
    .build() {
}

class CreateCollectionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "CreateCollection", {})
    .n("RekognitionClient", "CreateCollectionCommand")
    .sc(CreateCollection$)
    .build() {
}

class CreateDatasetCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "CreateDataset", {})
    .n("RekognitionClient", "CreateDatasetCommand")
    .sc(CreateDataset$)
    .build() {
}

class CreateFaceLivenessSessionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "CreateFaceLivenessSession", {})
    .n("RekognitionClient", "CreateFaceLivenessSessionCommand")
    .sc(CreateFaceLivenessSession$)
    .build() {
}

class CreateProjectCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "CreateProject", {})
    .n("RekognitionClient", "CreateProjectCommand")
    .sc(CreateProject$)
    .build() {
}

class CreateProjectVersionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "CreateProjectVersion", {})
    .n("RekognitionClient", "CreateProjectVersionCommand")
    .sc(CreateProjectVersion$)
    .build() {
}

class CreateStreamProcessorCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "CreateStreamProcessor", {})
    .n("RekognitionClient", "CreateStreamProcessorCommand")
    .sc(CreateStreamProcessor$)
    .build() {
}

class CreateUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "CreateUser", {})
    .n("RekognitionClient", "CreateUserCommand")
    .sc(CreateUser$)
    .build() {
}

class DeleteCollectionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DeleteCollection", {})
    .n("RekognitionClient", "DeleteCollectionCommand")
    .sc(DeleteCollection$)
    .build() {
}

class DeleteDatasetCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DeleteDataset", {})
    .n("RekognitionClient", "DeleteDatasetCommand")
    .sc(DeleteDataset$)
    .build() {
}

class DeleteFacesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DeleteFaces", {})
    .n("RekognitionClient", "DeleteFacesCommand")
    .sc(DeleteFaces$)
    .build() {
}

class DeleteProjectCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DeleteProject", {})
    .n("RekognitionClient", "DeleteProjectCommand")
    .sc(DeleteProject$)
    .build() {
}

class DeleteProjectPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DeleteProjectPolicy", {})
    .n("RekognitionClient", "DeleteProjectPolicyCommand")
    .sc(DeleteProjectPolicy$)
    .build() {
}

class DeleteProjectVersionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DeleteProjectVersion", {})
    .n("RekognitionClient", "DeleteProjectVersionCommand")
    .sc(DeleteProjectVersion$)
    .build() {
}

class DeleteStreamProcessorCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DeleteStreamProcessor", {})
    .n("RekognitionClient", "DeleteStreamProcessorCommand")
    .sc(DeleteStreamProcessor$)
    .build() {
}

class DeleteUserCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DeleteUser", {})
    .n("RekognitionClient", "DeleteUserCommand")
    .sc(DeleteUser$)
    .build() {
}

class DescribeCollectionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DescribeCollection", {})
    .n("RekognitionClient", "DescribeCollectionCommand")
    .sc(DescribeCollection$)
    .build() {
}

class DescribeDatasetCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DescribeDataset", {})
    .n("RekognitionClient", "DescribeDatasetCommand")
    .sc(DescribeDataset$)
    .build() {
}

class DescribeProjectsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DescribeProjects", {})
    .n("RekognitionClient", "DescribeProjectsCommand")
    .sc(DescribeProjects$)
    .build() {
}

class DescribeProjectVersionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DescribeProjectVersions", {})
    .n("RekognitionClient", "DescribeProjectVersionsCommand")
    .sc(DescribeProjectVersions$)
    .build() {
}

class DescribeStreamProcessorCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DescribeStreamProcessor", {})
    .n("RekognitionClient", "DescribeStreamProcessorCommand")
    .sc(DescribeStreamProcessor$)
    .build() {
}

class DetectCustomLabelsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DetectCustomLabels", {})
    .n("RekognitionClient", "DetectCustomLabelsCommand")
    .sc(DetectCustomLabels$)
    .build() {
}

class DetectFacesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DetectFaces", {})
    .n("RekognitionClient", "DetectFacesCommand")
    .sc(DetectFaces$)
    .build() {
}

class DetectLabelsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DetectLabels", {})
    .n("RekognitionClient", "DetectLabelsCommand")
    .sc(DetectLabels$)
    .build() {
}

class DetectModerationLabelsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DetectModerationLabels", {})
    .n("RekognitionClient", "DetectModerationLabelsCommand")
    .sc(DetectModerationLabels$)
    .build() {
}

class DetectProtectiveEquipmentCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DetectProtectiveEquipment", {})
    .n("RekognitionClient", "DetectProtectiveEquipmentCommand")
    .sc(DetectProtectiveEquipment$)
    .build() {
}

class DetectTextCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DetectText", {})
    .n("RekognitionClient", "DetectTextCommand")
    .sc(DetectText$)
    .build() {
}

class DisassociateFacesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DisassociateFaces", {})
    .n("RekognitionClient", "DisassociateFacesCommand")
    .sc(DisassociateFaces$)
    .build() {
}

class DistributeDatasetEntriesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "DistributeDatasetEntries", {})
    .n("RekognitionClient", "DistributeDatasetEntriesCommand")
    .sc(DistributeDatasetEntries$)
    .build() {
}

class GetCelebrityInfoCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "GetCelebrityInfo", {})
    .n("RekognitionClient", "GetCelebrityInfoCommand")
    .sc(GetCelebrityInfo$)
    .build() {
}

class GetCelebrityRecognitionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "GetCelebrityRecognition", {})
    .n("RekognitionClient", "GetCelebrityRecognitionCommand")
    .sc(GetCelebrityRecognition$)
    .build() {
}

class GetContentModerationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "GetContentModeration", {})
    .n("RekognitionClient", "GetContentModerationCommand")
    .sc(GetContentModeration$)
    .build() {
}

class GetFaceDetectionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "GetFaceDetection", {})
    .n("RekognitionClient", "GetFaceDetectionCommand")
    .sc(GetFaceDetection$)
    .build() {
}

class GetFaceLivenessSessionResultsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "GetFaceLivenessSessionResults", {})
    .n("RekognitionClient", "GetFaceLivenessSessionResultsCommand")
    .sc(GetFaceLivenessSessionResults$)
    .build() {
}

class GetFaceSearchCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "GetFaceSearch", {})
    .n("RekognitionClient", "GetFaceSearchCommand")
    .sc(GetFaceSearch$)
    .build() {
}

class GetLabelDetectionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "GetLabelDetection", {})
    .n("RekognitionClient", "GetLabelDetectionCommand")
    .sc(GetLabelDetection$)
    .build() {
}

class GetMediaAnalysisJobCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "GetMediaAnalysisJob", {})
    .n("RekognitionClient", "GetMediaAnalysisJobCommand")
    .sc(GetMediaAnalysisJob$)
    .build() {
}

class GetPersonTrackingCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "GetPersonTracking", {})
    .n("RekognitionClient", "GetPersonTrackingCommand")
    .sc(GetPersonTracking$)
    .build() {
}

class GetSegmentDetectionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "GetSegmentDetection", {})
    .n("RekognitionClient", "GetSegmentDetectionCommand")
    .sc(GetSegmentDetection$)
    .build() {
}

class GetTextDetectionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "GetTextDetection", {})
    .n("RekognitionClient", "GetTextDetectionCommand")
    .sc(GetTextDetection$)
    .build() {
}

class IndexFacesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "IndexFaces", {})
    .n("RekognitionClient", "IndexFacesCommand")
    .sc(IndexFaces$)
    .build() {
}

class ListCollectionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "ListCollections", {})
    .n("RekognitionClient", "ListCollectionsCommand")
    .sc(ListCollections$)
    .build() {
}

class ListDatasetEntriesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "ListDatasetEntries", {})
    .n("RekognitionClient", "ListDatasetEntriesCommand")
    .sc(ListDatasetEntries$)
    .build() {
}

class ListDatasetLabelsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "ListDatasetLabels", {})
    .n("RekognitionClient", "ListDatasetLabelsCommand")
    .sc(ListDatasetLabels$)
    .build() {
}

class ListFacesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "ListFaces", {})
    .n("RekognitionClient", "ListFacesCommand")
    .sc(ListFaces$)
    .build() {
}

class ListMediaAnalysisJobsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "ListMediaAnalysisJobs", {})
    .n("RekognitionClient", "ListMediaAnalysisJobsCommand")
    .sc(ListMediaAnalysisJobs$)
    .build() {
}

class ListProjectPoliciesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "ListProjectPolicies", {})
    .n("RekognitionClient", "ListProjectPoliciesCommand")
    .sc(ListProjectPolicies$)
    .build() {
}

class ListStreamProcessorsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "ListStreamProcessors", {})
    .n("RekognitionClient", "ListStreamProcessorsCommand")
    .sc(ListStreamProcessors$)
    .build() {
}

class ListTagsForResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "ListTagsForResource", {})
    .n("RekognitionClient", "ListTagsForResourceCommand")
    .sc(ListTagsForResource$)
    .build() {
}

class ListUsersCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "ListUsers", {})
    .n("RekognitionClient", "ListUsersCommand")
    .sc(ListUsers$)
    .build() {
}

class PutProjectPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "PutProjectPolicy", {})
    .n("RekognitionClient", "PutProjectPolicyCommand")
    .sc(PutProjectPolicy$)
    .build() {
}

class RecognizeCelebritiesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "RecognizeCelebrities", {})
    .n("RekognitionClient", "RecognizeCelebritiesCommand")
    .sc(RecognizeCelebrities$)
    .build() {
}

class SearchFacesByImageCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "SearchFacesByImage", {})
    .n("RekognitionClient", "SearchFacesByImageCommand")
    .sc(SearchFacesByImage$)
    .build() {
}

class SearchFacesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "SearchFaces", {})
    .n("RekognitionClient", "SearchFacesCommand")
    .sc(SearchFaces$)
    .build() {
}

class SearchUsersByImageCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "SearchUsersByImage", {})
    .n("RekognitionClient", "SearchUsersByImageCommand")
    .sc(SearchUsersByImage$)
    .build() {
}

class SearchUsersCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "SearchUsers", {})
    .n("RekognitionClient", "SearchUsersCommand")
    .sc(SearchUsers$)
    .build() {
}

class StartCelebrityRecognitionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StartCelebrityRecognition", {})
    .n("RekognitionClient", "StartCelebrityRecognitionCommand")
    .sc(StartCelebrityRecognition$)
    .build() {
}

class StartContentModerationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StartContentModeration", {})
    .n("RekognitionClient", "StartContentModerationCommand")
    .sc(StartContentModeration$)
    .build() {
}

class StartFaceDetectionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StartFaceDetection", {})
    .n("RekognitionClient", "StartFaceDetectionCommand")
    .sc(StartFaceDetection$)
    .build() {
}

class StartFaceSearchCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StartFaceSearch", {})
    .n("RekognitionClient", "StartFaceSearchCommand")
    .sc(StartFaceSearch$)
    .build() {
}

class StartLabelDetectionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StartLabelDetection", {})
    .n("RekognitionClient", "StartLabelDetectionCommand")
    .sc(StartLabelDetection$)
    .build() {
}

class StartMediaAnalysisJobCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StartMediaAnalysisJob", {})
    .n("RekognitionClient", "StartMediaAnalysisJobCommand")
    .sc(StartMediaAnalysisJob$)
    .build() {
}

class StartPersonTrackingCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StartPersonTracking", {})
    .n("RekognitionClient", "StartPersonTrackingCommand")
    .sc(StartPersonTracking$)
    .build() {
}

class StartProjectVersionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StartProjectVersion", {})
    .n("RekognitionClient", "StartProjectVersionCommand")
    .sc(StartProjectVersion$)
    .build() {
}

class StartSegmentDetectionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StartSegmentDetection", {})
    .n("RekognitionClient", "StartSegmentDetectionCommand")
    .sc(StartSegmentDetection$)
    .build() {
}

class StartStreamProcessorCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StartStreamProcessor", {})
    .n("RekognitionClient", "StartStreamProcessorCommand")
    .sc(StartStreamProcessor$)
    .build() {
}

class StartTextDetectionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StartTextDetection", {})
    .n("RekognitionClient", "StartTextDetectionCommand")
    .sc(StartTextDetection$)
    .build() {
}

class StopProjectVersionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StopProjectVersion", {})
    .n("RekognitionClient", "StopProjectVersionCommand")
    .sc(StopProjectVersion$)
    .build() {
}

class StopStreamProcessorCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "StopStreamProcessor", {})
    .n("RekognitionClient", "StopStreamProcessorCommand")
    .sc(StopStreamProcessor$)
    .build() {
}

class TagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "TagResource", {})
    .n("RekognitionClient", "TagResourceCommand")
    .sc(TagResource$)
    .build() {
}

class UntagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "UntagResource", {})
    .n("RekognitionClient", "UntagResourceCommand")
    .sc(UntagResource$)
    .build() {
}

class UpdateDatasetEntriesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "UpdateDatasetEntries", {})
    .n("RekognitionClient", "UpdateDatasetEntriesCommand")
    .sc(UpdateDatasetEntries$)
    .build() {
}

class UpdateStreamProcessorCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("RekognitionService", "UpdateStreamProcessor", {})
    .n("RekognitionClient", "UpdateStreamProcessorCommand")
    .sc(UpdateStreamProcessor$)
    .build() {
}

const commands = {
    AssociateFacesCommand,
    CompareFacesCommand,
    CopyProjectVersionCommand,
    CreateCollectionCommand,
    CreateDatasetCommand,
    CreateFaceLivenessSessionCommand,
    CreateProjectCommand,
    CreateProjectVersionCommand,
    CreateStreamProcessorCommand,
    CreateUserCommand,
    DeleteCollectionCommand,
    DeleteDatasetCommand,
    DeleteFacesCommand,
    DeleteProjectCommand,
    DeleteProjectPolicyCommand,
    DeleteProjectVersionCommand,
    DeleteStreamProcessorCommand,
    DeleteUserCommand,
    DescribeCollectionCommand,
    DescribeDatasetCommand,
    DescribeProjectsCommand,
    DescribeProjectVersionsCommand,
    DescribeStreamProcessorCommand,
    DetectCustomLabelsCommand,
    DetectFacesCommand,
    DetectLabelsCommand,
    DetectModerationLabelsCommand,
    DetectProtectiveEquipmentCommand,
    DetectTextCommand,
    DisassociateFacesCommand,
    DistributeDatasetEntriesCommand,
    GetCelebrityInfoCommand,
    GetCelebrityRecognitionCommand,
    GetContentModerationCommand,
    GetFaceDetectionCommand,
    GetFaceLivenessSessionResultsCommand,
    GetFaceSearchCommand,
    GetLabelDetectionCommand,
    GetMediaAnalysisJobCommand,
    GetPersonTrackingCommand,
    GetSegmentDetectionCommand,
    GetTextDetectionCommand,
    IndexFacesCommand,
    ListCollectionsCommand,
    ListDatasetEntriesCommand,
    ListDatasetLabelsCommand,
    ListFacesCommand,
    ListMediaAnalysisJobsCommand,
    ListProjectPoliciesCommand,
    ListStreamProcessorsCommand,
    ListTagsForResourceCommand,
    ListUsersCommand,
    PutProjectPolicyCommand,
    RecognizeCelebritiesCommand,
    SearchFacesCommand,
    SearchFacesByImageCommand,
    SearchUsersCommand,
    SearchUsersByImageCommand,
    StartCelebrityRecognitionCommand,
    StartContentModerationCommand,
    StartFaceDetectionCommand,
    StartFaceSearchCommand,
    StartLabelDetectionCommand,
    StartMediaAnalysisJobCommand,
    StartPersonTrackingCommand,
    StartProjectVersionCommand,
    StartSegmentDetectionCommand,
    StartStreamProcessorCommand,
    StartTextDetectionCommand,
    StopProjectVersionCommand,
    StopStreamProcessorCommand,
    TagResourceCommand,
    UntagResourceCommand,
    UpdateDatasetEntriesCommand,
    UpdateStreamProcessorCommand,
};
class Rekognition extends RekognitionClient {
}
smithyClient.createAggregatedClient(commands, Rekognition);

const paginateDescribeProjects = core.createPaginator(RekognitionClient, DescribeProjectsCommand, "NextToken", "NextToken", "MaxResults");

const paginateDescribeProjectVersions = core.createPaginator(RekognitionClient, DescribeProjectVersionsCommand, "NextToken", "NextToken", "MaxResults");

const paginateGetCelebrityRecognition = core.createPaginator(RekognitionClient, GetCelebrityRecognitionCommand, "NextToken", "NextToken", "MaxResults");

const paginateGetContentModeration = core.createPaginator(RekognitionClient, GetContentModerationCommand, "NextToken", "NextToken", "MaxResults");

const paginateGetFaceDetection = core.createPaginator(RekognitionClient, GetFaceDetectionCommand, "NextToken", "NextToken", "MaxResults");

const paginateGetFaceSearch = core.createPaginator(RekognitionClient, GetFaceSearchCommand, "NextToken", "NextToken", "MaxResults");

const paginateGetLabelDetection = core.createPaginator(RekognitionClient, GetLabelDetectionCommand, "NextToken", "NextToken", "MaxResults");

const paginateGetPersonTracking = core.createPaginator(RekognitionClient, GetPersonTrackingCommand, "NextToken", "NextToken", "MaxResults");

const paginateGetSegmentDetection = core.createPaginator(RekognitionClient, GetSegmentDetectionCommand, "NextToken", "NextToken", "MaxResults");

const paginateGetTextDetection = core.createPaginator(RekognitionClient, GetTextDetectionCommand, "NextToken", "NextToken", "MaxResults");

const paginateListCollections = core.createPaginator(RekognitionClient, ListCollectionsCommand, "NextToken", "NextToken", "MaxResults");

const paginateListDatasetEntries = core.createPaginator(RekognitionClient, ListDatasetEntriesCommand, "NextToken", "NextToken", "MaxResults");

const paginateListDatasetLabels = core.createPaginator(RekognitionClient, ListDatasetLabelsCommand, "NextToken", "NextToken", "MaxResults");

const paginateListFaces = core.createPaginator(RekognitionClient, ListFacesCommand, "NextToken", "NextToken", "MaxResults");

const paginateListMediaAnalysisJobs = core.createPaginator(RekognitionClient, ListMediaAnalysisJobsCommand, "NextToken", "NextToken", "MaxResults");

const paginateListProjectPolicies = core.createPaginator(RekognitionClient, ListProjectPoliciesCommand, "NextToken", "NextToken", "MaxResults");

const paginateListStreamProcessors = core.createPaginator(RekognitionClient, ListStreamProcessorsCommand, "NextToken", "NextToken", "MaxResults");

const paginateListUsers = core.createPaginator(RekognitionClient, ListUsersCommand, "NextToken", "NextToken", "MaxResults");

const checkState$1 = async (client, input) => {
    let reason;
    try {
        let result = await client.send(new DescribeProjectVersionsCommand(input));
        reason = result;
        try {
            const returnComparator = () => {
                let flat_1 = [].concat(...result.ProjectVersionDescriptions);
                let projection_3 = flat_1.map((element_2) => {
                    return element_2.Status;
                });
                return projection_3;
            };
            let allStringEq_5 = (returnComparator().length > 0);
            for (let element_4 of returnComparator()) {
                allStringEq_5 = allStringEq_5 && (element_4 == "RUNNING");
            }
            if (allStringEq_5) {
                return { state: utilWaiter.WaiterState.SUCCESS, reason };
            }
        }
        catch (e) { }
        try {
            const returnComparator = () => {
                let flat_1 = [].concat(...result.ProjectVersionDescriptions);
                let projection_3 = flat_1.map((element_2) => {
                    return element_2.Status;
                });
                return projection_3;
            };
            for (let anyStringEq_4 of returnComparator()) {
                if (anyStringEq_4 == "FAILED") {
                    return { state: utilWaiter.WaiterState.FAILURE, reason };
                }
            }
        }
        catch (e) { }
    }
    catch (exception) {
        reason = exception;
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForProjectVersionRunning = async (params, input) => {
    const serviceDefaults = { minDelay: 30, maxDelay: 120 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$1);
};
const waitUntilProjectVersionRunning = async (params, input) => {
    const serviceDefaults = { minDelay: 30, maxDelay: 120 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$1);
    return utilWaiter.checkExceptions(result);
};

const checkState = async (client, input) => {
    let reason;
    try {
        let result = await client.send(new DescribeProjectVersionsCommand(input));
        reason = result;
        try {
            const returnComparator = () => {
                let flat_1 = [].concat(...result.ProjectVersionDescriptions);
                let projection_3 = flat_1.map((element_2) => {
                    return element_2.Status;
                });
                return projection_3;
            };
            let allStringEq_5 = (returnComparator().length > 0);
            for (let element_4 of returnComparator()) {
                allStringEq_5 = allStringEq_5 && (element_4 == "TRAINING_COMPLETED");
            }
            if (allStringEq_5) {
                return { state: utilWaiter.WaiterState.SUCCESS, reason };
            }
        }
        catch (e) { }
        try {
            const returnComparator = () => {
                let flat_1 = [].concat(...result.ProjectVersionDescriptions);
                let projection_3 = flat_1.map((element_2) => {
                    return element_2.Status;
                });
                return projection_3;
            };
            for (let anyStringEq_4 of returnComparator()) {
                if (anyStringEq_4 == "TRAINING_FAILED") {
                    return { state: utilWaiter.WaiterState.FAILURE, reason };
                }
            }
        }
        catch (e) { }
    }
    catch (exception) {
        reason = exception;
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForProjectVersionTrainingCompleted = async (params, input) => {
    const serviceDefaults = { minDelay: 120, maxDelay: 120 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState);
};
const waitUntilProjectVersionTrainingCompleted = async (params, input) => {
    const serviceDefaults = { minDelay: 120, maxDelay: 120 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState);
    return utilWaiter.checkExceptions(result);
};

const UnsuccessfulFaceAssociationReason = {
    ASSOCIATED_TO_A_DIFFERENT_USER: "ASSOCIATED_TO_A_DIFFERENT_USER",
    FACE_NOT_FOUND: "FACE_NOT_FOUND",
    LOW_MATCH_CONFIDENCE: "LOW_MATCH_CONFIDENCE",
};
const UserStatus = {
    ACTIVE: "ACTIVE",
    CREATED: "CREATED",
    CREATING: "CREATING",
    UPDATING: "UPDATING",
};
const Attribute = {
    AGE_RANGE: "AGE_RANGE",
    ALL: "ALL",
    BEARD: "BEARD",
    DEFAULT: "DEFAULT",
    EMOTIONS: "EMOTIONS",
    EYEGLASSES: "EYEGLASSES",
    EYES_OPEN: "EYES_OPEN",
    EYE_DIRECTION: "EYE_DIRECTION",
    FACE_OCCLUDED: "FACE_OCCLUDED",
    GENDER: "GENDER",
    MOUTH_OPEN: "MOUTH_OPEN",
    MUSTACHE: "MUSTACHE",
    SMILE: "SMILE",
    SUNGLASSES: "SUNGLASSES",
};
const BodyPart = {
    FACE: "FACE",
    HEAD: "HEAD",
    LEFT_HAND: "LEFT_HAND",
    RIGHT_HAND: "RIGHT_HAND",
};
const ProtectiveEquipmentType = {
    FACE_COVER: "FACE_COVER",
    HAND_COVER: "HAND_COVER",
    HEAD_COVER: "HEAD_COVER",
};
const EmotionName = {
    ANGRY: "ANGRY",
    CALM: "CALM",
    CONFUSED: "CONFUSED",
    DISGUSTED: "DISGUSTED",
    FEAR: "FEAR",
    HAPPY: "HAPPY",
    SAD: "SAD",
    SURPRISED: "SURPRISED",
    UNKNOWN: "UNKNOWN",
};
const LandmarkType = {
    chinBottom: "chinBottom",
    eyeLeft: "eyeLeft",
    eyeRight: "eyeRight",
    leftEyeBrowLeft: "leftEyeBrowLeft",
    leftEyeBrowRight: "leftEyeBrowRight",
    leftEyeBrowUp: "leftEyeBrowUp",
    leftEyeDown: "leftEyeDown",
    leftEyeLeft: "leftEyeLeft",
    leftEyeRight: "leftEyeRight",
    leftEyeUp: "leftEyeUp",
    leftPupil: "leftPupil",
    midJawlineLeft: "midJawlineLeft",
    midJawlineRight: "midJawlineRight",
    mouthDown: "mouthDown",
    mouthLeft: "mouthLeft",
    mouthRight: "mouthRight",
    mouthUp: "mouthUp",
    nose: "nose",
    noseLeft: "noseLeft",
    noseRight: "noseRight",
    rightEyeBrowLeft: "rightEyeBrowLeft",
    rightEyeBrowRight: "rightEyeBrowRight",
    rightEyeBrowUp: "rightEyeBrowUp",
    rightEyeDown: "rightEyeDown",
    rightEyeLeft: "rightEyeLeft",
    rightEyeRight: "rightEyeRight",
    rightEyeUp: "rightEyeUp",
    rightPupil: "rightPupil",
    upperJawlineLeft: "upperJawlineLeft",
    upperJawlineRight: "upperJawlineRight",
};
const KnownGenderType = {
    Female: "Female",
    Male: "Male",
    Nonbinary: "Nonbinary",
    Unlisted: "Unlisted",
};
const GenderType = {
    Female: "Female",
    Male: "Male",
};
const CelebrityRecognitionSortBy = {
    ID: "ID",
    TIMESTAMP: "TIMESTAMP",
};
const ChallengeType = {
    FACE_MOVEMENT_AND_LIGHT_CHALLENGE: "FaceMovementAndLightChallenge",
    FACE_MOVEMENT_CHALLENGE: "FaceMovementChallenge",
};
const QualityFilter = {
    AUTO: "AUTO",
    HIGH: "HIGH",
    LOW: "LOW",
    MEDIUM: "MEDIUM",
    NONE: "NONE",
};
const OrientationCorrection = {
    ROTATE_0: "ROTATE_0",
    ROTATE_180: "ROTATE_180",
    ROTATE_270: "ROTATE_270",
    ROTATE_90: "ROTATE_90",
};
const ContentClassifier = {
    FREE_OF_ADULT_CONTENT: "FreeOfAdultContent",
    FREE_OF_PERSONALLY_IDENTIFIABLE_INFORMATION: "FreeOfPersonallyIdentifiableInformation",
};
const ContentModerationAggregateBy = {
    SEGMENTS: "SEGMENTS",
    TIMESTAMPS: "TIMESTAMPS",
};
const ContentModerationSortBy = {
    NAME: "NAME",
    TIMESTAMP: "TIMESTAMP",
};
const DatasetType = {
    TEST: "TEST",
    TRAIN: "TRAIN",
};
const ProjectAutoUpdate = {
    DISABLED: "DISABLED",
    ENABLED: "ENABLED",
};
const CustomizationFeature = {
    CONTENT_MODERATION: "CONTENT_MODERATION",
    CUSTOM_LABELS: "CUSTOM_LABELS",
};
const DatasetStatus = {
    CREATE_COMPLETE: "CREATE_COMPLETE",
    CREATE_FAILED: "CREATE_FAILED",
    CREATE_IN_PROGRESS: "CREATE_IN_PROGRESS",
    DELETE_IN_PROGRESS: "DELETE_IN_PROGRESS",
    UPDATE_COMPLETE: "UPDATE_COMPLETE",
    UPDATE_FAILED: "UPDATE_FAILED",
    UPDATE_IN_PROGRESS: "UPDATE_IN_PROGRESS",
};
const DatasetStatusMessageCode = {
    CLIENT_ERROR: "CLIENT_ERROR",
    SERVICE_ERROR: "SERVICE_ERROR",
    SUCCESS: "SUCCESS",
};
const UnsuccessfulFaceDeletionReason = {
    ASSOCIATED_TO_AN_EXISTING_USER: "ASSOCIATED_TO_AN_EXISTING_USER",
    FACE_NOT_FOUND: "FACE_NOT_FOUND",
};
const ProjectStatus = {
    CREATED: "CREATED",
    CREATING: "CREATING",
    DELETING: "DELETING",
};
const ProjectVersionStatus = {
    COPYING_COMPLETED: "COPYING_COMPLETED",
    COPYING_FAILED: "COPYING_FAILED",
    COPYING_IN_PROGRESS: "COPYING_IN_PROGRESS",
    DELETING: "DELETING",
    DEPRECATED: "DEPRECATED",
    EXPIRED: "EXPIRED",
    FAILED: "FAILED",
    RUNNING: "RUNNING",
    STARTING: "STARTING",
    STOPPED: "STOPPED",
    STOPPING: "STOPPING",
    TRAINING_COMPLETED: "TRAINING_COMPLETED",
    TRAINING_FAILED: "TRAINING_FAILED",
    TRAINING_IN_PROGRESS: "TRAINING_IN_PROGRESS",
};
const StreamProcessorStatus = {
    FAILED: "FAILED",
    RUNNING: "RUNNING",
    STARTING: "STARTING",
    STOPPED: "STOPPED",
    STOPPING: "STOPPING",
    UPDATING: "UPDATING",
};
const DetectLabelsFeatureName = {
    GENERAL_LABELS: "GENERAL_LABELS",
    IMAGE_PROPERTIES: "IMAGE_PROPERTIES",
};
const TextTypes = {
    LINE: "LINE",
    WORD: "WORD",
};
const UnsuccessfulFaceDisassociationReason = {
    ASSOCIATED_TO_A_DIFFERENT_USER: "ASSOCIATED_TO_A_DIFFERENT_USER",
    FACE_NOT_FOUND: "FACE_NOT_FOUND",
};
const FaceAttributes = {
    ALL: "ALL",
    DEFAULT: "DEFAULT",
};
const FaceSearchSortBy = {
    INDEX: "INDEX",
    TIMESTAMP: "TIMESTAMP",
};
const VideoJobStatus = {
    FAILED: "FAILED",
    IN_PROGRESS: "IN_PROGRESS",
    SUCCEEDED: "SUCCEEDED",
};
const VideoColorRange = {
    FULL: "FULL",
    LIMITED: "LIMITED",
};
const LivenessSessionStatus = {
    CREATED: "CREATED",
    EXPIRED: "EXPIRED",
    FAILED: "FAILED",
    IN_PROGRESS: "IN_PROGRESS",
    SUCCEEDED: "SUCCEEDED",
};
const LabelDetectionAggregateBy = {
    SEGMENTS: "SEGMENTS",
    TIMESTAMPS: "TIMESTAMPS",
};
const LabelDetectionSortBy = {
    NAME: "NAME",
    TIMESTAMP: "TIMESTAMP",
};
const MediaAnalysisJobFailureCode = {
    ACCESS_DENIED: "ACCESS_DENIED",
    INTERNAL_ERROR: "INTERNAL_ERROR",
    INVALID_KMS_KEY: "INVALID_KMS_KEY",
    INVALID_MANIFEST: "INVALID_MANIFEST",
    INVALID_OUTPUT_CONFIG: "INVALID_OUTPUT_CONFIG",
    INVALID_S3_OBJECT: "INVALID_S3_OBJECT",
    RESOURCE_NOT_FOUND: "RESOURCE_NOT_FOUND",
    RESOURCE_NOT_READY: "RESOURCE_NOT_READY",
    THROTTLED: "THROTTLED",
};
const MediaAnalysisJobStatus = {
    CREATED: "CREATED",
    FAILED: "FAILED",
    IN_PROGRESS: "IN_PROGRESS",
    QUEUED: "QUEUED",
    SUCCEEDED: "SUCCEEDED",
};
const PersonTrackingSortBy = {
    INDEX: "INDEX",
    TIMESTAMP: "TIMESTAMP",
};
const TechnicalCueType = {
    BLACK_FRAMES: "BlackFrames",
    COLOR_BARS: "ColorBars",
    CONTENT: "Content",
    END_CREDITS: "EndCredits",
    OPENING_CREDITS: "OpeningCredits",
    SLATE: "Slate",
    STUDIO_LOGO: "StudioLogo",
};
const SegmentType = {
    SHOT: "SHOT",
    TECHNICAL_CUE: "TECHNICAL_CUE",
};
const Reason = {
    EXCEEDS_MAX_FACES: "EXCEEDS_MAX_FACES",
    EXTREME_POSE: "EXTREME_POSE",
    LOW_BRIGHTNESS: "LOW_BRIGHTNESS",
    LOW_CONFIDENCE: "LOW_CONFIDENCE",
    LOW_FACE_QUALITY: "LOW_FACE_QUALITY",
    LOW_SHARPNESS: "LOW_SHARPNESS",
    SMALL_BOUNDING_BOX: "SMALL_BOUNDING_BOX",
};
const LabelDetectionFeatureName = {
    GENERAL_LABELS: "GENERAL_LABELS",
};
const UnsearchedFaceReason = {
    EXCEEDS_MAX_FACES: "EXCEEDS_MAX_FACES",
    EXTREME_POSE: "EXTREME_POSE",
    FACE_NOT_LARGEST: "FACE_NOT_LARGEST",
    LOW_BRIGHTNESS: "LOW_BRIGHTNESS",
    LOW_CONFIDENCE: "LOW_CONFIDENCE",
    LOW_FACE_QUALITY: "LOW_FACE_QUALITY",
    LOW_SHARPNESS: "LOW_SHARPNESS",
    SMALL_BOUNDING_BOX: "SMALL_BOUNDING_BOX",
};
const StreamProcessorParameterToDelete = {
    ConnectedHomeMinConfidence: "ConnectedHomeMinConfidence",
    RegionsOfInterest: "RegionsOfInterest",
};

Object.defineProperty(exports, "$Command", {
    enumerable: true,
    get: function () { return smithyClient.Command; }
});
Object.defineProperty(exports, "__Client", {
    enumerable: true,
    get: function () { return smithyClient.Client; }
});
exports.AccessDeniedException = AccessDeniedException;
exports.AccessDeniedException$ = AccessDeniedException$;
exports.AgeRange$ = AgeRange$;
exports.Asset$ = Asset$;
exports.AssociateFaces$ = AssociateFaces$;
exports.AssociateFacesCommand = AssociateFacesCommand;
exports.AssociateFacesRequest$ = AssociateFacesRequest$;
exports.AssociateFacesResponse$ = AssociateFacesResponse$;
exports.AssociatedFace$ = AssociatedFace$;
exports.Attribute = Attribute;
exports.AudioMetadata$ = AudioMetadata$;
exports.AuditImage$ = AuditImage$;
exports.Beard$ = Beard$;
exports.BlackFrame$ = BlackFrame$;
exports.BodyPart = BodyPart;
exports.BoundingBox$ = BoundingBox$;
exports.Celebrity$ = Celebrity$;
exports.CelebrityDetail$ = CelebrityDetail$;
exports.CelebrityRecognition$ = CelebrityRecognition$;
exports.CelebrityRecognitionSortBy = CelebrityRecognitionSortBy;
exports.Challenge$ = Challenge$;
exports.ChallengePreference$ = ChallengePreference$;
exports.ChallengeType = ChallengeType;
exports.CompareFaces$ = CompareFaces$;
exports.CompareFacesCommand = CompareFacesCommand;
exports.CompareFacesMatch$ = CompareFacesMatch$;
exports.CompareFacesRequest$ = CompareFacesRequest$;
exports.CompareFacesResponse$ = CompareFacesResponse$;
exports.ComparedFace$ = ComparedFace$;
exports.ComparedSourceImageFace$ = ComparedSourceImageFace$;
exports.ConflictException = ConflictException;
exports.ConflictException$ = ConflictException$;
exports.ConnectedHomeSettings$ = ConnectedHomeSettings$;
exports.ConnectedHomeSettingsForUpdate$ = ConnectedHomeSettingsForUpdate$;
exports.ContentClassifier = ContentClassifier;
exports.ContentModerationAggregateBy = ContentModerationAggregateBy;
exports.ContentModerationDetection$ = ContentModerationDetection$;
exports.ContentModerationSortBy = ContentModerationSortBy;
exports.ContentType$ = ContentType$;
exports.CopyProjectVersion$ = CopyProjectVersion$;
exports.CopyProjectVersionCommand = CopyProjectVersionCommand;
exports.CopyProjectVersionRequest$ = CopyProjectVersionRequest$;
exports.CopyProjectVersionResponse$ = CopyProjectVersionResponse$;
exports.CoversBodyPart$ = CoversBodyPart$;
exports.CreateCollection$ = CreateCollection$;
exports.CreateCollectionCommand = CreateCollectionCommand;
exports.CreateCollectionRequest$ = CreateCollectionRequest$;
exports.CreateCollectionResponse$ = CreateCollectionResponse$;
exports.CreateDataset$ = CreateDataset$;
exports.CreateDatasetCommand = CreateDatasetCommand;
exports.CreateDatasetRequest$ = CreateDatasetRequest$;
exports.CreateDatasetResponse$ = CreateDatasetResponse$;
exports.CreateFaceLivenessSession$ = CreateFaceLivenessSession$;
exports.CreateFaceLivenessSessionCommand = CreateFaceLivenessSessionCommand;
exports.CreateFaceLivenessSessionRequest$ = CreateFaceLivenessSessionRequest$;
exports.CreateFaceLivenessSessionRequestSettings$ = CreateFaceLivenessSessionRequestSettings$;
exports.CreateFaceLivenessSessionResponse$ = CreateFaceLivenessSessionResponse$;
exports.CreateProject$ = CreateProject$;
exports.CreateProjectCommand = CreateProjectCommand;
exports.CreateProjectRequest$ = CreateProjectRequest$;
exports.CreateProjectResponse$ = CreateProjectResponse$;
exports.CreateProjectVersion$ = CreateProjectVersion$;
exports.CreateProjectVersionCommand = CreateProjectVersionCommand;
exports.CreateProjectVersionRequest$ = CreateProjectVersionRequest$;
exports.CreateProjectVersionResponse$ = CreateProjectVersionResponse$;
exports.CreateStreamProcessor$ = CreateStreamProcessor$;
exports.CreateStreamProcessorCommand = CreateStreamProcessorCommand;
exports.CreateStreamProcessorRequest$ = CreateStreamProcessorRequest$;
exports.CreateStreamProcessorResponse$ = CreateStreamProcessorResponse$;
exports.CreateUser$ = CreateUser$;
exports.CreateUserCommand = CreateUserCommand;
exports.CreateUserRequest$ = CreateUserRequest$;
exports.CreateUserResponse$ = CreateUserResponse$;
exports.CustomLabel$ = CustomLabel$;
exports.CustomizationFeature = CustomizationFeature;
exports.CustomizationFeatureConfig$ = CustomizationFeatureConfig$;
exports.CustomizationFeatureContentModerationConfig$ = CustomizationFeatureContentModerationConfig$;
exports.DatasetChanges$ = DatasetChanges$;
exports.DatasetDescription$ = DatasetDescription$;
exports.DatasetLabelDescription$ = DatasetLabelDescription$;
exports.DatasetLabelStats$ = DatasetLabelStats$;
exports.DatasetMetadata$ = DatasetMetadata$;
exports.DatasetSource$ = DatasetSource$;
exports.DatasetStats$ = DatasetStats$;
exports.DatasetStatus = DatasetStatus;
exports.DatasetStatusMessageCode = DatasetStatusMessageCode;
exports.DatasetType = DatasetType;
exports.DeleteCollection$ = DeleteCollection$;
exports.DeleteCollectionCommand = DeleteCollectionCommand;
exports.DeleteCollectionRequest$ = DeleteCollectionRequest$;
exports.DeleteCollectionResponse$ = DeleteCollectionResponse$;
exports.DeleteDataset$ = DeleteDataset$;
exports.DeleteDatasetCommand = DeleteDatasetCommand;
exports.DeleteDatasetRequest$ = DeleteDatasetRequest$;
exports.DeleteDatasetResponse$ = DeleteDatasetResponse$;
exports.DeleteFaces$ = DeleteFaces$;
exports.DeleteFacesCommand = DeleteFacesCommand;
exports.DeleteFacesRequest$ = DeleteFacesRequest$;
exports.DeleteFacesResponse$ = DeleteFacesResponse$;
exports.DeleteProject$ = DeleteProject$;
exports.DeleteProjectCommand = DeleteProjectCommand;
exports.DeleteProjectPolicy$ = DeleteProjectPolicy$;
exports.DeleteProjectPolicyCommand = DeleteProjectPolicyCommand;
exports.DeleteProjectPolicyRequest$ = DeleteProjectPolicyRequest$;
exports.DeleteProjectPolicyResponse$ = DeleteProjectPolicyResponse$;
exports.DeleteProjectRequest$ = DeleteProjectRequest$;
exports.DeleteProjectResponse$ = DeleteProjectResponse$;
exports.DeleteProjectVersion$ = DeleteProjectVersion$;
exports.DeleteProjectVersionCommand = DeleteProjectVersionCommand;
exports.DeleteProjectVersionRequest$ = DeleteProjectVersionRequest$;
exports.DeleteProjectVersionResponse$ = DeleteProjectVersionResponse$;
exports.DeleteStreamProcessor$ = DeleteStreamProcessor$;
exports.DeleteStreamProcessorCommand = DeleteStreamProcessorCommand;
exports.DeleteStreamProcessorRequest$ = DeleteStreamProcessorRequest$;
exports.DeleteStreamProcessorResponse$ = DeleteStreamProcessorResponse$;
exports.DeleteUser$ = DeleteUser$;
exports.DeleteUserCommand = DeleteUserCommand;
exports.DeleteUserRequest$ = DeleteUserRequest$;
exports.DeleteUserResponse$ = DeleteUserResponse$;
exports.DescribeCollection$ = DescribeCollection$;
exports.DescribeCollectionCommand = DescribeCollectionCommand;
exports.DescribeCollectionRequest$ = DescribeCollectionRequest$;
exports.DescribeCollectionResponse$ = DescribeCollectionResponse$;
exports.DescribeDataset$ = DescribeDataset$;
exports.DescribeDatasetCommand = DescribeDatasetCommand;
exports.DescribeDatasetRequest$ = DescribeDatasetRequest$;
exports.DescribeDatasetResponse$ = DescribeDatasetResponse$;
exports.DescribeProjectVersions$ = DescribeProjectVersions$;
exports.DescribeProjectVersionsCommand = DescribeProjectVersionsCommand;
exports.DescribeProjectVersionsRequest$ = DescribeProjectVersionsRequest$;
exports.DescribeProjectVersionsResponse$ = DescribeProjectVersionsResponse$;
exports.DescribeProjects$ = DescribeProjects$;
exports.DescribeProjectsCommand = DescribeProjectsCommand;
exports.DescribeProjectsRequest$ = DescribeProjectsRequest$;
exports.DescribeProjectsResponse$ = DescribeProjectsResponse$;
exports.DescribeStreamProcessor$ = DescribeStreamProcessor$;
exports.DescribeStreamProcessorCommand = DescribeStreamProcessorCommand;
exports.DescribeStreamProcessorRequest$ = DescribeStreamProcessorRequest$;
exports.DescribeStreamProcessorResponse$ = DescribeStreamProcessorResponse$;
exports.DetectCustomLabels$ = DetectCustomLabels$;
exports.DetectCustomLabelsCommand = DetectCustomLabelsCommand;
exports.DetectCustomLabelsRequest$ = DetectCustomLabelsRequest$;
exports.DetectCustomLabelsResponse$ = DetectCustomLabelsResponse$;
exports.DetectFaces$ = DetectFaces$;
exports.DetectFacesCommand = DetectFacesCommand;
exports.DetectFacesRequest$ = DetectFacesRequest$;
exports.DetectFacesResponse$ = DetectFacesResponse$;
exports.DetectLabels$ = DetectLabels$;
exports.DetectLabelsCommand = DetectLabelsCommand;
exports.DetectLabelsFeatureName = DetectLabelsFeatureName;
exports.DetectLabelsImageBackground$ = DetectLabelsImageBackground$;
exports.DetectLabelsImageForeground$ = DetectLabelsImageForeground$;
exports.DetectLabelsImageProperties$ = DetectLabelsImageProperties$;
exports.DetectLabelsImagePropertiesSettings$ = DetectLabelsImagePropertiesSettings$;
exports.DetectLabelsImageQuality$ = DetectLabelsImageQuality$;
exports.DetectLabelsRequest$ = DetectLabelsRequest$;
exports.DetectLabelsResponse$ = DetectLabelsResponse$;
exports.DetectLabelsSettings$ = DetectLabelsSettings$;
exports.DetectModerationLabels$ = DetectModerationLabels$;
exports.DetectModerationLabelsCommand = DetectModerationLabelsCommand;
exports.DetectModerationLabelsRequest$ = DetectModerationLabelsRequest$;
exports.DetectModerationLabelsResponse$ = DetectModerationLabelsResponse$;
exports.DetectProtectiveEquipment$ = DetectProtectiveEquipment$;
exports.DetectProtectiveEquipmentCommand = DetectProtectiveEquipmentCommand;
exports.DetectProtectiveEquipmentRequest$ = DetectProtectiveEquipmentRequest$;
exports.DetectProtectiveEquipmentResponse$ = DetectProtectiveEquipmentResponse$;
exports.DetectText$ = DetectText$;
exports.DetectTextCommand = DetectTextCommand;
exports.DetectTextFilters$ = DetectTextFilters$;
exports.DetectTextRequest$ = DetectTextRequest$;
exports.DetectTextResponse$ = DetectTextResponse$;
exports.DetectionFilter$ = DetectionFilter$;
exports.DisassociateFaces$ = DisassociateFaces$;
exports.DisassociateFacesCommand = DisassociateFacesCommand;
exports.DisassociateFacesRequest$ = DisassociateFacesRequest$;
exports.DisassociateFacesResponse$ = DisassociateFacesResponse$;
exports.DisassociatedFace$ = DisassociatedFace$;
exports.DistributeDataset$ = DistributeDataset$;
exports.DistributeDatasetEntries$ = DistributeDatasetEntries$;
exports.DistributeDatasetEntriesCommand = DistributeDatasetEntriesCommand;
exports.DistributeDatasetEntriesRequest$ = DistributeDatasetEntriesRequest$;
exports.DistributeDatasetEntriesResponse$ = DistributeDatasetEntriesResponse$;
exports.DominantColor$ = DominantColor$;
exports.Emotion$ = Emotion$;
exports.EmotionName = EmotionName;
exports.EquipmentDetection$ = EquipmentDetection$;
exports.EvaluationResult$ = EvaluationResult$;
exports.EyeDirection$ = EyeDirection$;
exports.EyeOpen$ = EyeOpen$;
exports.Eyeglasses$ = Eyeglasses$;
exports.Face$ = Face$;
exports.FaceAttributes = FaceAttributes;
exports.FaceDetail$ = FaceDetail$;
exports.FaceDetection$ = FaceDetection$;
exports.FaceMatch$ = FaceMatch$;
exports.FaceOccluded$ = FaceOccluded$;
exports.FaceRecord$ = FaceRecord$;
exports.FaceSearchSettings$ = FaceSearchSettings$;
exports.FaceSearchSortBy = FaceSearchSortBy;
exports.Gender$ = Gender$;
exports.GenderType = GenderType;
exports.GeneralLabelsSettings$ = GeneralLabelsSettings$;
exports.Geometry$ = Geometry$;
exports.GetCelebrityInfo$ = GetCelebrityInfo$;
exports.GetCelebrityInfoCommand = GetCelebrityInfoCommand;
exports.GetCelebrityInfoRequest$ = GetCelebrityInfoRequest$;
exports.GetCelebrityInfoResponse$ = GetCelebrityInfoResponse$;
exports.GetCelebrityRecognition$ = GetCelebrityRecognition$;
exports.GetCelebrityRecognitionCommand = GetCelebrityRecognitionCommand;
exports.GetCelebrityRecognitionRequest$ = GetCelebrityRecognitionRequest$;
exports.GetCelebrityRecognitionResponse$ = GetCelebrityRecognitionResponse$;
exports.GetContentModeration$ = GetContentModeration$;
exports.GetContentModerationCommand = GetContentModerationCommand;
exports.GetContentModerationRequest$ = GetContentModerationRequest$;
exports.GetContentModerationRequestMetadata$ = GetContentModerationRequestMetadata$;
exports.GetContentModerationResponse$ = GetContentModerationResponse$;
exports.GetFaceDetection$ = GetFaceDetection$;
exports.GetFaceDetectionCommand = GetFaceDetectionCommand;
exports.GetFaceDetectionRequest$ = GetFaceDetectionRequest$;
exports.GetFaceDetectionResponse$ = GetFaceDetectionResponse$;
exports.GetFaceLivenessSessionResults$ = GetFaceLivenessSessionResults$;
exports.GetFaceLivenessSessionResultsCommand = GetFaceLivenessSessionResultsCommand;
exports.GetFaceLivenessSessionResultsRequest$ = GetFaceLivenessSessionResultsRequest$;
exports.GetFaceLivenessSessionResultsResponse$ = GetFaceLivenessSessionResultsResponse$;
exports.GetFaceSearch$ = GetFaceSearch$;
exports.GetFaceSearchCommand = GetFaceSearchCommand;
exports.GetFaceSearchRequest$ = GetFaceSearchRequest$;
exports.GetFaceSearchResponse$ = GetFaceSearchResponse$;
exports.GetLabelDetection$ = GetLabelDetection$;
exports.GetLabelDetectionCommand = GetLabelDetectionCommand;
exports.GetLabelDetectionRequest$ = GetLabelDetectionRequest$;
exports.GetLabelDetectionRequestMetadata$ = GetLabelDetectionRequestMetadata$;
exports.GetLabelDetectionResponse$ = GetLabelDetectionResponse$;
exports.GetMediaAnalysisJob$ = GetMediaAnalysisJob$;
exports.GetMediaAnalysisJobCommand = GetMediaAnalysisJobCommand;
exports.GetMediaAnalysisJobRequest$ = GetMediaAnalysisJobRequest$;
exports.GetMediaAnalysisJobResponse$ = GetMediaAnalysisJobResponse$;
exports.GetPersonTracking$ = GetPersonTracking$;
exports.GetPersonTrackingCommand = GetPersonTrackingCommand;
exports.GetPersonTrackingRequest$ = GetPersonTrackingRequest$;
exports.GetPersonTrackingResponse$ = GetPersonTrackingResponse$;
exports.GetSegmentDetection$ = GetSegmentDetection$;
exports.GetSegmentDetectionCommand = GetSegmentDetectionCommand;
exports.GetSegmentDetectionRequest$ = GetSegmentDetectionRequest$;
exports.GetSegmentDetectionResponse$ = GetSegmentDetectionResponse$;
exports.GetTextDetection$ = GetTextDetection$;
exports.GetTextDetectionCommand = GetTextDetectionCommand;
exports.GetTextDetectionRequest$ = GetTextDetectionRequest$;
exports.GetTextDetectionResponse$ = GetTextDetectionResponse$;
exports.GroundTruthManifest$ = GroundTruthManifest$;
exports.HumanLoopActivationOutput$ = HumanLoopActivationOutput$;
exports.HumanLoopConfig$ = HumanLoopConfig$;
exports.HumanLoopDataAttributes$ = HumanLoopDataAttributes$;
exports.HumanLoopQuotaExceededException = HumanLoopQuotaExceededException;
exports.HumanLoopQuotaExceededException$ = HumanLoopQuotaExceededException$;
exports.IdempotentParameterMismatchException = IdempotentParameterMismatchException;
exports.IdempotentParameterMismatchException$ = IdempotentParameterMismatchException$;
exports.Image$ = Image$;
exports.ImageQuality$ = ImageQuality$;
exports.ImageTooLargeException = ImageTooLargeException;
exports.ImageTooLargeException$ = ImageTooLargeException$;
exports.IndexFaces$ = IndexFaces$;
exports.IndexFacesCommand = IndexFacesCommand;
exports.IndexFacesRequest$ = IndexFacesRequest$;
exports.IndexFacesResponse$ = IndexFacesResponse$;
exports.Instance$ = Instance$;
exports.InternalServerError = InternalServerError;
exports.InternalServerError$ = InternalServerError$;
exports.InvalidImageFormatException = InvalidImageFormatException;
exports.InvalidImageFormatException$ = InvalidImageFormatException$;
exports.InvalidManifestException = InvalidManifestException;
exports.InvalidManifestException$ = InvalidManifestException$;
exports.InvalidPaginationTokenException = InvalidPaginationTokenException;
exports.InvalidPaginationTokenException$ = InvalidPaginationTokenException$;
exports.InvalidParameterException = InvalidParameterException;
exports.InvalidParameterException$ = InvalidParameterException$;
exports.InvalidPolicyRevisionIdException = InvalidPolicyRevisionIdException;
exports.InvalidPolicyRevisionIdException$ = InvalidPolicyRevisionIdException$;
exports.InvalidS3ObjectException = InvalidS3ObjectException;
exports.InvalidS3ObjectException$ = InvalidS3ObjectException$;
exports.KinesisDataStream$ = KinesisDataStream$;
exports.KinesisVideoStream$ = KinesisVideoStream$;
exports.KinesisVideoStreamStartSelector$ = KinesisVideoStreamStartSelector$;
exports.KnownGender$ = KnownGender$;
exports.KnownGenderType = KnownGenderType;
exports.Label$ = Label$;
exports.LabelAlias$ = LabelAlias$;
exports.LabelCategory$ = LabelCategory$;
exports.LabelDetection$ = LabelDetection$;
exports.LabelDetectionAggregateBy = LabelDetectionAggregateBy;
exports.LabelDetectionFeatureName = LabelDetectionFeatureName;
exports.LabelDetectionSettings$ = LabelDetectionSettings$;
exports.LabelDetectionSortBy = LabelDetectionSortBy;
exports.Landmark$ = Landmark$;
exports.LandmarkType = LandmarkType;
exports.LimitExceededException = LimitExceededException;
exports.LimitExceededException$ = LimitExceededException$;
exports.ListCollections$ = ListCollections$;
exports.ListCollectionsCommand = ListCollectionsCommand;
exports.ListCollectionsRequest$ = ListCollectionsRequest$;
exports.ListCollectionsResponse$ = ListCollectionsResponse$;
exports.ListDatasetEntries$ = ListDatasetEntries$;
exports.ListDatasetEntriesCommand = ListDatasetEntriesCommand;
exports.ListDatasetEntriesRequest$ = ListDatasetEntriesRequest$;
exports.ListDatasetEntriesResponse$ = ListDatasetEntriesResponse$;
exports.ListDatasetLabels$ = ListDatasetLabels$;
exports.ListDatasetLabelsCommand = ListDatasetLabelsCommand;
exports.ListDatasetLabelsRequest$ = ListDatasetLabelsRequest$;
exports.ListDatasetLabelsResponse$ = ListDatasetLabelsResponse$;
exports.ListFaces$ = ListFaces$;
exports.ListFacesCommand = ListFacesCommand;
exports.ListFacesRequest$ = ListFacesRequest$;
exports.ListFacesResponse$ = ListFacesResponse$;
exports.ListMediaAnalysisJobs$ = ListMediaAnalysisJobs$;
exports.ListMediaAnalysisJobsCommand = ListMediaAnalysisJobsCommand;
exports.ListMediaAnalysisJobsRequest$ = ListMediaAnalysisJobsRequest$;
exports.ListMediaAnalysisJobsResponse$ = ListMediaAnalysisJobsResponse$;
exports.ListProjectPolicies$ = ListProjectPolicies$;
exports.ListProjectPoliciesCommand = ListProjectPoliciesCommand;
exports.ListProjectPoliciesRequest$ = ListProjectPoliciesRequest$;
exports.ListProjectPoliciesResponse$ = ListProjectPoliciesResponse$;
exports.ListStreamProcessors$ = ListStreamProcessors$;
exports.ListStreamProcessorsCommand = ListStreamProcessorsCommand;
exports.ListStreamProcessorsRequest$ = ListStreamProcessorsRequest$;
exports.ListStreamProcessorsResponse$ = ListStreamProcessorsResponse$;
exports.ListTagsForResource$ = ListTagsForResource$;
exports.ListTagsForResourceCommand = ListTagsForResourceCommand;
exports.ListTagsForResourceRequest$ = ListTagsForResourceRequest$;
exports.ListTagsForResourceResponse$ = ListTagsForResourceResponse$;
exports.ListUsers$ = ListUsers$;
exports.ListUsersCommand = ListUsersCommand;
exports.ListUsersRequest$ = ListUsersRequest$;
exports.ListUsersResponse$ = ListUsersResponse$;
exports.LivenessOutputConfig$ = LivenessOutputConfig$;
exports.LivenessSessionStatus = LivenessSessionStatus;
exports.MalformedPolicyDocumentException = MalformedPolicyDocumentException;
exports.MalformedPolicyDocumentException$ = MalformedPolicyDocumentException$;
exports.MatchedUser$ = MatchedUser$;
exports.MediaAnalysisDetectModerationLabelsConfig$ = MediaAnalysisDetectModerationLabelsConfig$;
exports.MediaAnalysisInput$ = MediaAnalysisInput$;
exports.MediaAnalysisJobDescription$ = MediaAnalysisJobDescription$;
exports.MediaAnalysisJobFailureCode = MediaAnalysisJobFailureCode;
exports.MediaAnalysisJobFailureDetails$ = MediaAnalysisJobFailureDetails$;
exports.MediaAnalysisJobStatus = MediaAnalysisJobStatus;
exports.MediaAnalysisManifestSummary$ = MediaAnalysisManifestSummary$;
exports.MediaAnalysisModelVersions$ = MediaAnalysisModelVersions$;
exports.MediaAnalysisOperationsConfig$ = MediaAnalysisOperationsConfig$;
exports.MediaAnalysisOutputConfig$ = MediaAnalysisOutputConfig$;
exports.MediaAnalysisResults$ = MediaAnalysisResults$;
exports.ModerationLabel$ = ModerationLabel$;
exports.MouthOpen$ = MouthOpen$;
exports.Mustache$ = Mustache$;
exports.NotificationChannel$ = NotificationChannel$;
exports.OrientationCorrection = OrientationCorrection;
exports.OutputConfig$ = OutputConfig$;
exports.Parent$ = Parent$;
exports.PersonDetail$ = PersonDetail$;
exports.PersonDetection$ = PersonDetection$;
exports.PersonMatch$ = PersonMatch$;
exports.PersonTrackingSortBy = PersonTrackingSortBy;
exports.Point$ = Point$;
exports.Pose$ = Pose$;
exports.ProjectAutoUpdate = ProjectAutoUpdate;
exports.ProjectDescription$ = ProjectDescription$;
exports.ProjectPolicy$ = ProjectPolicy$;
exports.ProjectStatus = ProjectStatus;
exports.ProjectVersionDescription$ = ProjectVersionDescription$;
exports.ProjectVersionStatus = ProjectVersionStatus;
exports.ProtectiveEquipmentBodyPart$ = ProtectiveEquipmentBodyPart$;
exports.ProtectiveEquipmentPerson$ = ProtectiveEquipmentPerson$;
exports.ProtectiveEquipmentSummarizationAttributes$ = ProtectiveEquipmentSummarizationAttributes$;
exports.ProtectiveEquipmentSummary$ = ProtectiveEquipmentSummary$;
exports.ProtectiveEquipmentType = ProtectiveEquipmentType;
exports.ProvisionedThroughputExceededException = ProvisionedThroughputExceededException;
exports.ProvisionedThroughputExceededException$ = ProvisionedThroughputExceededException$;
exports.PutProjectPolicy$ = PutProjectPolicy$;
exports.PutProjectPolicyCommand = PutProjectPolicyCommand;
exports.PutProjectPolicyRequest$ = PutProjectPolicyRequest$;
exports.PutProjectPolicyResponse$ = PutProjectPolicyResponse$;
exports.QualityFilter = QualityFilter;
exports.Reason = Reason;
exports.RecognizeCelebrities$ = RecognizeCelebrities$;
exports.RecognizeCelebritiesCommand = RecognizeCelebritiesCommand;
exports.RecognizeCelebritiesRequest$ = RecognizeCelebritiesRequest$;
exports.RecognizeCelebritiesResponse$ = RecognizeCelebritiesResponse$;
exports.RegionOfInterest$ = RegionOfInterest$;
exports.Rekognition = Rekognition;
exports.RekognitionClient = RekognitionClient;
exports.RekognitionServiceException = RekognitionServiceException;
exports.RekognitionServiceException$ = RekognitionServiceException$;
exports.ResourceAlreadyExistsException = ResourceAlreadyExistsException;
exports.ResourceAlreadyExistsException$ = ResourceAlreadyExistsException$;
exports.ResourceInUseException = ResourceInUseException;
exports.ResourceInUseException$ = ResourceInUseException$;
exports.ResourceNotFoundException = ResourceNotFoundException;
exports.ResourceNotFoundException$ = ResourceNotFoundException$;
exports.ResourceNotReadyException = ResourceNotReadyException;
exports.ResourceNotReadyException$ = ResourceNotReadyException$;
exports.S3Destination$ = S3Destination$;
exports.S3Object$ = S3Object$;
exports.SearchFaces$ = SearchFaces$;
exports.SearchFacesByImage$ = SearchFacesByImage$;
exports.SearchFacesByImageCommand = SearchFacesByImageCommand;
exports.SearchFacesByImageRequest$ = SearchFacesByImageRequest$;
exports.SearchFacesByImageResponse$ = SearchFacesByImageResponse$;
exports.SearchFacesCommand = SearchFacesCommand;
exports.SearchFacesRequest$ = SearchFacesRequest$;
exports.SearchFacesResponse$ = SearchFacesResponse$;
exports.SearchUsers$ = SearchUsers$;
exports.SearchUsersByImage$ = SearchUsersByImage$;
exports.SearchUsersByImageCommand = SearchUsersByImageCommand;
exports.SearchUsersByImageRequest$ = SearchUsersByImageRequest$;
exports.SearchUsersByImageResponse$ = SearchUsersByImageResponse$;
exports.SearchUsersCommand = SearchUsersCommand;
exports.SearchUsersRequest$ = SearchUsersRequest$;
exports.SearchUsersResponse$ = SearchUsersResponse$;
exports.SearchedFace$ = SearchedFace$;
exports.SearchedFaceDetails$ = SearchedFaceDetails$;
exports.SearchedUser$ = SearchedUser$;
exports.SegmentDetection$ = SegmentDetection$;
exports.SegmentType = SegmentType;
exports.SegmentTypeInfo$ = SegmentTypeInfo$;
exports.ServiceQuotaExceededException = ServiceQuotaExceededException;
exports.ServiceQuotaExceededException$ = ServiceQuotaExceededException$;
exports.SessionNotFoundException = SessionNotFoundException;
exports.SessionNotFoundException$ = SessionNotFoundException$;
exports.ShotSegment$ = ShotSegment$;
exports.Smile$ = Smile$;
exports.StartCelebrityRecognition$ = StartCelebrityRecognition$;
exports.StartCelebrityRecognitionCommand = StartCelebrityRecognitionCommand;
exports.StartCelebrityRecognitionRequest$ = StartCelebrityRecognitionRequest$;
exports.StartCelebrityRecognitionResponse$ = StartCelebrityRecognitionResponse$;
exports.StartContentModeration$ = StartContentModeration$;
exports.StartContentModerationCommand = StartContentModerationCommand;
exports.StartContentModerationRequest$ = StartContentModerationRequest$;
exports.StartContentModerationResponse$ = StartContentModerationResponse$;
exports.StartFaceDetection$ = StartFaceDetection$;
exports.StartFaceDetectionCommand = StartFaceDetectionCommand;
exports.StartFaceDetectionRequest$ = StartFaceDetectionRequest$;
exports.StartFaceDetectionResponse$ = StartFaceDetectionResponse$;
exports.StartFaceSearch$ = StartFaceSearch$;
exports.StartFaceSearchCommand = StartFaceSearchCommand;
exports.StartFaceSearchRequest$ = StartFaceSearchRequest$;
exports.StartFaceSearchResponse$ = StartFaceSearchResponse$;
exports.StartLabelDetection$ = StartLabelDetection$;
exports.StartLabelDetectionCommand = StartLabelDetectionCommand;
exports.StartLabelDetectionRequest$ = StartLabelDetectionRequest$;
exports.StartLabelDetectionResponse$ = StartLabelDetectionResponse$;
exports.StartMediaAnalysisJob$ = StartMediaAnalysisJob$;
exports.StartMediaAnalysisJobCommand = StartMediaAnalysisJobCommand;
exports.StartMediaAnalysisJobRequest$ = StartMediaAnalysisJobRequest$;
exports.StartMediaAnalysisJobResponse$ = StartMediaAnalysisJobResponse$;
exports.StartPersonTracking$ = StartPersonTracking$;
exports.StartPersonTrackingCommand = StartPersonTrackingCommand;
exports.StartPersonTrackingRequest$ = StartPersonTrackingRequest$;
exports.StartPersonTrackingResponse$ = StartPersonTrackingResponse$;
exports.StartProjectVersion$ = StartProjectVersion$;
exports.StartProjectVersionCommand = StartProjectVersionCommand;
exports.StartProjectVersionRequest$ = StartProjectVersionRequest$;
exports.StartProjectVersionResponse$ = StartProjectVersionResponse$;
exports.StartSegmentDetection$ = StartSegmentDetection$;
exports.StartSegmentDetectionCommand = StartSegmentDetectionCommand;
exports.StartSegmentDetectionFilters$ = StartSegmentDetectionFilters$;
exports.StartSegmentDetectionRequest$ = StartSegmentDetectionRequest$;
exports.StartSegmentDetectionResponse$ = StartSegmentDetectionResponse$;
exports.StartShotDetectionFilter$ = StartShotDetectionFilter$;
exports.StartStreamProcessor$ = StartStreamProcessor$;
exports.StartStreamProcessorCommand = StartStreamProcessorCommand;
exports.StartStreamProcessorRequest$ = StartStreamProcessorRequest$;
exports.StartStreamProcessorResponse$ = StartStreamProcessorResponse$;
exports.StartTechnicalCueDetectionFilter$ = StartTechnicalCueDetectionFilter$;
exports.StartTextDetection$ = StartTextDetection$;
exports.StartTextDetectionCommand = StartTextDetectionCommand;
exports.StartTextDetectionFilters$ = StartTextDetectionFilters$;
exports.StartTextDetectionRequest$ = StartTextDetectionRequest$;
exports.StartTextDetectionResponse$ = StartTextDetectionResponse$;
exports.StopProjectVersion$ = StopProjectVersion$;
exports.StopProjectVersionCommand = StopProjectVersionCommand;
exports.StopProjectVersionRequest$ = StopProjectVersionRequest$;
exports.StopProjectVersionResponse$ = StopProjectVersionResponse$;
exports.StopStreamProcessor$ = StopStreamProcessor$;
exports.StopStreamProcessorCommand = StopStreamProcessorCommand;
exports.StopStreamProcessorRequest$ = StopStreamProcessorRequest$;
exports.StopStreamProcessorResponse$ = StopStreamProcessorResponse$;
exports.StreamProcessingStartSelector$ = StreamProcessingStartSelector$;
exports.StreamProcessingStopSelector$ = StreamProcessingStopSelector$;
exports.StreamProcessor$ = StreamProcessor$;
exports.StreamProcessorDataSharingPreference$ = StreamProcessorDataSharingPreference$;
exports.StreamProcessorInput$ = StreamProcessorInput$;
exports.StreamProcessorNotificationChannel$ = StreamProcessorNotificationChannel$;
exports.StreamProcessorOutput$ = StreamProcessorOutput$;
exports.StreamProcessorParameterToDelete = StreamProcessorParameterToDelete;
exports.StreamProcessorSettings$ = StreamProcessorSettings$;
exports.StreamProcessorSettingsForUpdate$ = StreamProcessorSettingsForUpdate$;
exports.StreamProcessorStatus = StreamProcessorStatus;
exports.Summary$ = Summary$;
exports.Sunglasses$ = Sunglasses$;
exports.TagResource$ = TagResource$;
exports.TagResourceCommand = TagResourceCommand;
exports.TagResourceRequest$ = TagResourceRequest$;
exports.TagResourceResponse$ = TagResourceResponse$;
exports.TechnicalCueSegment$ = TechnicalCueSegment$;
exports.TechnicalCueType = TechnicalCueType;
exports.TestingData$ = TestingData$;
exports.TestingDataResult$ = TestingDataResult$;
exports.TextDetection$ = TextDetection$;
exports.TextDetectionResult$ = TextDetectionResult$;
exports.TextTypes = TextTypes;
exports.ThrottlingException = ThrottlingException;
exports.ThrottlingException$ = ThrottlingException$;
exports.TrainingData$ = TrainingData$;
exports.TrainingDataResult$ = TrainingDataResult$;
exports.UnindexedFace$ = UnindexedFace$;
exports.UnsearchedFace$ = UnsearchedFace$;
exports.UnsearchedFaceReason = UnsearchedFaceReason;
exports.UnsuccessfulFaceAssociation$ = UnsuccessfulFaceAssociation$;
exports.UnsuccessfulFaceAssociationReason = UnsuccessfulFaceAssociationReason;
exports.UnsuccessfulFaceDeletion$ = UnsuccessfulFaceDeletion$;
exports.UnsuccessfulFaceDeletionReason = UnsuccessfulFaceDeletionReason;
exports.UnsuccessfulFaceDisassociation$ = UnsuccessfulFaceDisassociation$;
exports.UnsuccessfulFaceDisassociationReason = UnsuccessfulFaceDisassociationReason;
exports.UntagResource$ = UntagResource$;
exports.UntagResourceCommand = UntagResourceCommand;
exports.UntagResourceRequest$ = UntagResourceRequest$;
exports.UntagResourceResponse$ = UntagResourceResponse$;
exports.UpdateDatasetEntries$ = UpdateDatasetEntries$;
exports.UpdateDatasetEntriesCommand = UpdateDatasetEntriesCommand;
exports.UpdateDatasetEntriesRequest$ = UpdateDatasetEntriesRequest$;
exports.UpdateDatasetEntriesResponse$ = UpdateDatasetEntriesResponse$;
exports.UpdateStreamProcessor$ = UpdateStreamProcessor$;
exports.UpdateStreamProcessorCommand = UpdateStreamProcessorCommand;
exports.UpdateStreamProcessorRequest$ = UpdateStreamProcessorRequest$;
exports.UpdateStreamProcessorResponse$ = UpdateStreamProcessorResponse$;
exports.User$ = User$;
exports.UserMatch$ = UserMatch$;
exports.UserStatus = UserStatus;
exports.ValidationData$ = ValidationData$;
exports.Versions$ = Versions$;
exports.Video$ = Video$;
exports.VideoColorRange = VideoColorRange;
exports.VideoJobStatus = VideoJobStatus;
exports.VideoMetadata$ = VideoMetadata$;
exports.VideoTooLargeException = VideoTooLargeException;
exports.VideoTooLargeException$ = VideoTooLargeException$;
exports.paginateDescribeProjectVersions = paginateDescribeProjectVersions;
exports.paginateDescribeProjects = paginateDescribeProjects;
exports.paginateGetCelebrityRecognition = paginateGetCelebrityRecognition;
exports.paginateGetContentModeration = paginateGetContentModeration;
exports.paginateGetFaceDetection = paginateGetFaceDetection;
exports.paginateGetFaceSearch = paginateGetFaceSearch;
exports.paginateGetLabelDetection = paginateGetLabelDetection;
exports.paginateGetPersonTracking = paginateGetPersonTracking;
exports.paginateGetSegmentDetection = paginateGetSegmentDetection;
exports.paginateGetTextDetection = paginateGetTextDetection;
exports.paginateListCollections = paginateListCollections;
exports.paginateListDatasetEntries = paginateListDatasetEntries;
exports.paginateListDatasetLabels = paginateListDatasetLabels;
exports.paginateListFaces = paginateListFaces;
exports.paginateListMediaAnalysisJobs = paginateListMediaAnalysisJobs;
exports.paginateListProjectPolicies = paginateListProjectPolicies;
exports.paginateListStreamProcessors = paginateListStreamProcessors;
exports.paginateListUsers = paginateListUsers;
exports.waitForProjectVersionRunning = waitForProjectVersionRunning;
exports.waitForProjectVersionTrainingCompleted = waitForProjectVersionTrainingCompleted;
exports.waitUntilProjectVersionRunning = waitUntilProjectVersionRunning;
exports.waitUntilProjectVersionTrainingCompleted = waitUntilProjectVersionTrainingCompleted;
