import { ExceptionFilter, Catch, ArgumentsHost, HttpException, HttpStatus } from '@nestjs/common';
import { Response } from 'express';

/**
 * Filtro global de exceções
 * Captura todos os erros e envia resposta formatada
 */
@Catch()
export class AllExceptionsFilter implements ExceptionFilter {
  catch(exception: any, host: ArgumentsHost) {
    const ctx = host.switchToHttp();
    const response = ctx.getResponse<Response>();
    const request = ctx.getRequest();

    // Log detalhado do erro
    console.error('========================================');
    console.error('❌ ERRO CAPTURADO PELO EXCEPTION FILTER');
    console.error('========================================');
    console.error('Timestamp:', new Date().toISOString());
    console.error('Path:', request.url);
    console.error('Method:', request.method);
    console.error('Body:', JSON.stringify(request.body, null, 2));
    console.error('');
    console.error('Erro:');
    console.error('  Message:', exception.message);
    console.error('  Name:', exception.name);
    console.error('  Code:', exception.code);
    console.error('  Detail:', exception.detail);
    console.error('');
    console.error('Stack:', exception.stack);
    console.error('========================================');

    const status =
      exception instanceof HttpException
        ? exception.getStatus()
        : HttpStatus.INTERNAL_SERVER_ERROR;

    const errorResponse = {
      statusCode: status,
      timestamp: new Date().toISOString(),
      path: request.url,
      message: exception.message || 'Internal server error',
      error: exception.name || 'Error',
      ...(process.env.NODE_ENV === 'development' && {
        detail: exception.detail,
        code: exception.code,
        stack: exception.stack,
      }),
    };

    response.status(status).json(errorResponse);
  }
}
