import { Controller, Post, Body, Get, UseGuards, Request } from '@nestjs/common';
import { AuthService } from './auth.service';
import { RegisterDto } from './dto/register.dto';
import { LoginDto } from './dto/login.dto';
import { JwtAuthGuard } from './guards/jwt-auth.guard';

/**
 * Controlador de Autenticação
 * 
 * Endpoints:
 * - POST /auth/register - Registo de novo utilizador
 * - POST /auth/login - Login com email/telefone + senha
 * - GET /auth/profile - Dados do utilizador autenticado
 * - POST /auth/google - Login via Google Identity
 * - POST /auth/2fa/send - Enviar código 2FA via SMS
 * - POST /auth/2fa/verify - Verificar código 2FA
 */
@Controller('auth')
export class AuthController {
  constructor(private readonly authService: AuthService) {}

  @Post('register')
  async register(@Body() registerDto: RegisterDto) {
    return this.authService.register(registerDto);
  }

  @Post('login')
  async login(@Body() loginDto: LoginDto) {
    return this.authService.login(loginDto);
  }

  @Get('profile')
  @UseGuards(JwtAuthGuard)
  getProfile(@Request() req: any) {
    return req.user;
  }

  @Post('google')
  async googleAuth(@Body() body: { idToken: string }) {
    // Integração com Google Identity
    return this.authService.googleLogin(body.idToken);
  }

  @Post('2fa/send')
  async send2FA(@Body() body: { telefone: string }) {
    // Enviar código via SMS/WhatsApp
    return this.authService.sendTwoFactorCode(body.telefone);
  }

  @Post('2fa/verify')
  async verify2FA(@Body() body: { telefone: string; codigo: string }) {
    // Verificar código 2FA
    return this.authService.verifyTwoFactorCode(body.telefone, body.codigo);
  }
}
