import { Injectable, UnauthorizedException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { JwtService } from '@nestjs/jwt';
import { User } from './entities/user.entity';
import { RegisterDto } from './dto/register.dto';
import { LoginDto } from './dto/login.dto';
import * as bcrypt from 'bcrypt';

@Injectable()
export class AuthService {
  constructor(
    @InjectRepository(User)
    private readonly userRepository: Repository<User>,
    private readonly jwtService: JwtService,
  ) {}

  /**
   * Registo de novo utilizador
   */
  async register(registerDto: RegisterDto) {
    try {
      const { email, senha, nome, telefone, tipo } = registerDto;

      // Verificar se o email já existe
      const userExists = await this.userRepository.findOne({ where: { email } });
      if (userExists) {
        throw new UnauthorizedException('Email já registado');
      }

      // Verificar se o telefone já existe
      const phoneExists = await this.userRepository.findOne({ where: { telefone } });
      if (phoneExists) {
        throw new UnauthorizedException('Telefone já registado');
      }

      // Hash da senha
      const senhaHash = await bcrypt.hash(senha, 10);

      // Criar novo utilizador
      const user = this.userRepository.create({
        email,
        senha: senhaHash,
        nome,
        telefone,
        tipo, // 'vendedor' ou 'comprador'
        seloVerificado: false,
        kycStatus: 'pendente',
      });

      await this.userRepository.save(user);

      // Gerar token JWT
      const token = this.jwtService.sign({ sub: user.id, email: user.email });

      return {
        message: 'Utilizador criado com sucesso',
        token,
        user: {
          id: user.id,
          nome: user.nome,
          email: user.email,
          tipo: user.tipo,
          seloVerificado: user.seloVerificado,
        },
      };
    } catch (error) {
      // Log detalhado para debug
      console.error('❌ Erro no registo:', error.message);
      console.error('   Stack:', error.stack);
      
      // Re-lançar erros de autorização
      if (error instanceof UnauthorizedException) {
        throw error;
      }
      
      // Tratar erro de constraint PostgreSQL
      if (error.code === '23505') {
        if (error.detail.includes('email')) {
          throw new UnauthorizedException('Email já registado');
        }
        if (error.detail.includes('telefone')) {
          throw new UnauthorizedException('Telefone já registado');
        }
      }
      
      // Erro genérico
      throw new Error(`Erro ao criar utilizador: ${error.message}`);
    }
  }

  /**
   * Login com email/telefone + senha
   */
  async login(loginDto: LoginDto) {
    const { email, senha } = loginDto;

    const user = await this.userRepository.findOne({ where: { email } });
    if (!user) {
      throw new UnauthorizedException('Credenciais inválidas');
    }

    // Verificar senha
    const senhaValida = await bcrypt.compare(senha, user.senha);
    if (!senhaValida) {
      throw new UnauthorizedException('Credenciais inválidas');
    }

    // Gerar token JWT
    const token = this.jwtService.sign({ sub: user.id, email: user.email });

    return {
      token,
      user: {
        id: user.id,
        nome: user.nome,
        email: user.email,
        tipo: user.tipo,
        seloVerificado: user.seloVerificado,
      },
    };
  }

  /**
   * Login via Google Identity
   */
  async googleLogin(idToken: string) {
    // TODO: Validar token com Google OAuth2
    // const payload = await this.verifyGoogleToken(idToken);
    
    // Por agora, retornar mock
    return { message: 'Google login - a implementar' };
  }

  /**
   * Enviar código 2FA via SMS
   */
  async sendTwoFactorCode(telefone: string) {
    // TODO: Integrar com serviço SMS angolano
    const codigo = Math.floor(100000 + Math.random() * 900000).toString();
    
    // Guardar código temporariamente (Redis com TTL de 5 minutos)
    console.log(`Código 2FA para ${telefone}: ${codigo}`);
    
    return { message: 'Código enviado via SMS' };
  }

  /**
   * Verificar código 2FA
   */
  async verifyTwoFactorCode(telefone: string, codigo: string) {
    // TODO: Verificar código no Redis
    return { message: 'Código verificado com sucesso' };
  }

  /**
   * Validar utilizador pelo ID (usado pelo JwtStrategy)
   */
  async validateUser(userId: string) {
    return this.userRepository.findOne({ where: { id: userId } });
  }
}
