import { Controller, Post, Get, Body, Param, UseGuards, UploadedFile, UseInterceptors } from '@nestjs/common';
import { FileInterceptor } from '@nestjs/platform-express';
import { KycService } from './kyc.service';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';

/**
 * Controlador KYC
 * 
 * Endpoints:
 * - POST /kyc/upload-bi - Upload de foto do BI
 * - POST /kyc/upload-selfie - Upload de selfie (Liveness Detection)
 * - POST /kyc/upload-alvara - Upload de Alvará (Business)
 * - GET /kyc/status/:userId - Estado da verificação
 * - POST /kyc/analyze - Análise completa via Amazon Rekognition
 */
@Controller('kyc')
export class KycController {
  constructor(private readonly kycService: KycService) {}

  @Post('upload-bi')
  @UseGuards(JwtAuthGuard)
  @UseInterceptors(FileInterceptor('file'))
  async uploadBI(
    @UploadedFile() file: Express.Multer.File,
    @Body('userId') userId: string,
  ) {
    return this.kycService.processBI(userId, file);
  }

  @Post('upload-selfie')
  @UseGuards(JwtAuthGuard)
  @UseInterceptors(FileInterceptor('file'))
  async uploadSelfie(
    @UploadedFile() file: Express.Multer.File,
    @Body('userId') userId: string,
  ) {
    return this.kycService.processSelfie(userId, file);
  }

  @Post('upload-alvara')
  @UseGuards(JwtAuthGuard)
  @UseInterceptors(FileInterceptor('file'))
  async uploadAlvara(
    @UploadedFile() file: Express.Multer.File,
    @Body('userId') userId: string,
  ) {
    return this.kycService.processAlvara(userId, file);
  }

  @Get('status/:userId')
  @UseGuards(JwtAuthGuard)
  async getKycStatus(@Param('userId') userId: string) {
    return this.kycService.getKycStatus(userId);
  }

  @Post('analyze')
  @UseGuards(JwtAuthGuard)
  async analyzeDocuments(@Body('userId') userId: string) {
    // Chama Amazon Rekognition para análise completa
    return this.kycService.analyzeWithAI(userId);
  }
}
