import { Injectable, BadRequestException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { ImageAnnotatorClient } from '@google-cloud/vision';
import { promises as fs } from 'node:fs';
import * as path from 'node:path';
import sharp from 'sharp';
import type { Express } from 'express';
import { User } from '../auth/entities/user.entity';

@Injectable()
export class KycService {
  private readonly visionClient: ImageAnnotatorClient;

  constructor(
    @InjectRepository(User)
    private readonly userRepository: Repository<User>,
  ) {
    // Google Cloud Vision - Configuração via GOOGLE_APPLICATION_CREDENTIALS env
    // Em desenvolvimento sem credenciais: Usar modo mock
    try {
      this.visionClient = new ImageAnnotatorClient();
    } catch (err) {
      console.warn('[KYC] Google Cloud Vision não configurado - Usando modo MOCK para desenvolvimento');
      console.warn('[KYC] Para produção: Configure GOOGLE_APPLICATION_CREDENTIALS no .env');
      // visionClient ficará undefined - métodos vão detectar e usar mocks
    }
  }

  /**
   * 1️⃣ PROCESSAR FOTO DO BI
   * - Adicionar marca d'água "TudoAqui"
   * - OCR para extrair dados do BI (nome completo, número)
   * - Guardar no storage local
   */
  async processBI(userId: string, file: Express.Multer.File) {
    if (!file?.buffer) {
      throw new BadRequestException('Ficheiro do BI é obrigatório');
    }

    const user = await this.userRepository.findOne({ where: { id: userId } });
    if (!user) {
      throw new BadRequestException('Utilizador não encontrado');
    }

    const outputDir = path.join(process.cwd(), 'uploads', 'bi');
    await this.ensureDir(outputDir);

    const filename = `${userId}_${Date.now()}.jpg`;
    const outputPath = path.join(outputDir, filename);

    // Marca d'água "TudoAqui" para prevenir reutilização
    const watermarked = await this.addWatermark(file.buffer, 'TudoAqui - KYC');
    await fs.writeFile(outputPath, watermarked);

    // OCR do BI via Google Cloud Vision
    let dadosBI: { nome?: string; numeroBI?: string } = {};
    try {
      dadosBI = await this.extractBIData(watermarked);
    } catch (err) {
      console.error('[KYC] Erro no OCR do BI:', err.message);
      // Não falhar - permitir validação manual depois
    }

    user.fotoBI = outputPath;
    if (dadosBI.numeroBI) {
      user.biNumero = dadosBI.numeroBI;
    }
    await this.userRepository.save(user);

    return {
      message: 'Foto do BI carregada com sucesso',
      url: outputPath,
      dadosExtraidos: dadosBI,
    };
  }

  /**
   * 2️⃣ PROCESSAR SELFIE (LIVENESS DETECTION)
   * - Verificar se é uma pessoa real (não foto de foto)
   * - Detectar rosto único
   * - Guardar para comparação posterior
   */
  async processSelfie(userId: string, file: Express.Multer.File) {
    if (!file?.buffer) {
      throw new BadRequestException('Selfie é obrigatória');
    }

    const user = await this.userRepository.findOne({ where: { id: userId } });
    if (!user) {
      throw new BadRequestException('Utilizador não encontrado');
    }

    const outputDir = path.join(process.cwd(), 'uploads', 'selfie');
    await this.ensureDir(outputDir);

    const filename = `${userId}_${Date.now()}.jpg`;
    const outputPath = path.join(outputDir, filename);

    await fs.writeFile(outputPath, file.buffer);

    // Liveness Detection básico: verificar se há exatamente 1 rosto
    let livenessOk = false;
    try {
      livenessOk = await this.verifyLiveness(file.buffer);
    } catch (err) {
      console.error('[KYC] Erro no Liveness Detection:', err.message);
    }

    user.fotoSelfie = outputPath;
    await this.userRepository.save(user);

    return {
      message: 'Selfie carregada com sucesso',
      url: outputPath,
      livenessDetected: livenessOk,
    };
  }

  /**
   * 3️⃣ ANALISAR E APROVAR AUTOMATICAMENTE
   * - Comparar nome do BI com nome da conta
   * - Comparar rosto do BI com selfie (Google Vision Face Detection)
   * - Se ambos coincidirem → Aprovar e atribuir selo verde
   */
  async analyzeWithAI(userId: string) {
    const user = await this.userRepository.findOne({ where: { id: userId } });
    if (!user) {
      throw new BadRequestException('Utilizador não encontrado');
    }

    if (!user.fotoBI || !user.fotoSelfie) {
      throw new BadRequestException('É necessário enviar BI e selfie antes de analisar');
    }

    const validacoes: {
      nomeCoincide: boolean;
      rostosCorrespondem: boolean;
      aprovado: boolean;
    } = {
      nomeCoincide: false,
      rostosCorrespondem: false,
      aprovado: false,
    };

    // 1. Comparar NOME do BI com nome da conta
    if (user.biNumero) {
      // Re-extrair nome do BI (OCR já foi feito no upload, mas refazer para garantir)
      const biBuffer = await fs.readFile(user.fotoBI);
      const dadosBI = await this.extractBIData(biBuffer);

      if (dadosBI.nome) {
        validacoes.nomeCoincide = this.compareNames(user.nome, dadosBI.nome);
      }
    }

    // 2. Comparar ROSTO do BI com selfie (Face Matching)
    try {
      const biBuffer = await fs.readFile(user.fotoBI);
      const selfieBuffer = await fs.readFile(user.fotoSelfie);

      const facesMatch = await this.compareFaces(biBuffer, selfieBuffer);
      validacoes.rostosCorrespondem = facesMatch;
    } catch (err) {
      console.error('[KYC] Erro na comparação de rostos:', err.message);
    }

    // 3. APROVAÇÃO AUTOMÁTICA
    if (validacoes.nomeCoincide && validacoes.rostosCorrespondem) {
      user.kycStatus = 'aprovado';
      user.seloVerificado = true;
      validacoes.aprovado = true;
    } else {
      user.kycStatus = 'rejeitado';
      validacoes.aprovado = false;
    }

    await this.userRepository.save(user);

    return {
      message: validacoes.aprovado
        ? '✅ Verificação aprovada! Selo verde atribuído.'
        : '❌ Verificação rejeitada. Dados não coincidem.',
      validacoes,
      seloVerificado: user.seloVerificado,
    };
  }

  /**
   * Status do KYC
   */
  async getKycStatus(userId: string) {
    const user = await this.userRepository.findOne({
      where: { id: userId },
      select: ['id', 'nome', 'email', 'kycStatus', 'seloVerificado', 'biNumero', 'fotoBI', 'fotoSelfie'],
    });

    if (!user) {
      throw new BadRequestException('Utilizador não encontrado');
    }

    return {
      kycStatus: user.kycStatus,
      seloVerificado: user.seloVerificado,
      biEnviado: !!user.fotoBI,
      selfieEnviada: !!user.fotoSelfie,
      numeroBI: user.biNumero || null,
    };
  }

  // ==================== MÉTODOS AUXILIARES ====================

  /**
   * OCR do BI com Google Cloud Vision - Extrai nome e número
   */
  private async extractBIData(imageBuffer: Buffer): Promise<{ nome?: string; numeroBI?: string }> {
    // Modo MOCK se Google Vision não configurado
    if (!this.visionClient) {
      console.log('[KYC] Modo MOCK: Retornando dados simulados do BI');
      return {
        numeroBI: '123456789AO01',
        nome: 'João Silva Mock',
      };
    }

    try {
      const [result] = await this.visionClient.textDetection({
        image: { content: imageBuffer },
      });

      const detections = result.textAnnotations || [];
      if (detections.length === 0) {
        return {};
      }

      // O primeiro item é o texto completo
      const textoCompleto = detections[0]?.description || '';

      // Extrair número do BI (formato angolano: 000000000XY00)
      const biRegex = /(\d{9}[A-Z]{2}\d{2})/;
      const matchBI = textoCompleto.match(biRegex);

      // Extrair nome (heurística: linha após "Nome" ou "NOME")
      let nome: string | undefined;
      const linhas = textoCompleto.split('\n');
      for (let i = 0; i < linhas.length; i++) {
        if (/nome/i.test(linhas[i]) && linhas[i + 1]) {
          nome = linhas[i + 1].trim();
          break;
        }
      }

      return {
        numeroBI: matchBI ? matchBI[1] : undefined,
        nome: nome,
      };
    } catch (err) {
      console.error('[KYC] Erro no OCR:', err);
      return {};
    }
  }

  /**
   * Liveness Detection - Verificar se há exatamente 1 rosto
   */
  private async verifyLiveness(imageBuffer: Buffer): Promise<boolean> {
    // Modo MOCK se Google Vision não configurado
    if (!this.visionClient) {
      console.log('[KYC] Modo MOCK: Liveness aprovado automaticamente');
      return true;
    }

    try {
      const [result] = await this.visionClient.faceDetection({
        image: { content: imageBuffer },
      });

      const faces = result.faceAnnotations || [];

      // Liveness básico: exatamente 1 rosto + confiança > 80%
      if (faces.length !== 1) {
        return false;
      }

      const face = faces[0];
      const confidence = face.detectionConfidence || 0;

      return confidence > 0.8;
    } catch (err) {
      console.error('[KYC] Erro no Liveness:', err);
      return false;
    }
  }

  /**
   * Comparar rostos entre BI e selfie
   * Usa Face Detection + Landmarks para calcular similaridade
   */
  private async compareFaces(biBuffer: Buffer, selfieBuffer: Buffer): Promise<boolean> {
    // Modo MOCK se Google Vision não configurado
    if (!this.visionClient) {
      console.log('[KYC] Modo MOCK: Rostos correspondem automaticamente');
      return true;
    }

    try {
      // Detectar rostos em ambas as imagens
      const [biResult] = await this.visionClient.faceDetection({ image: { content: biBuffer } });
      const [selfieResult] = await this.visionClient.faceDetection({ image: { content: selfieBuffer } });

      const biFaces = biResult.faceAnnotations || [];
      const selfieFaces = selfieResult.faceAnnotations || [];

      if (biFaces.length !== 1 || selfieFaces.length !== 1) {
        return false; // Deve haver exatamente 1 rosto em cada imagem
      }

      // Comparação simples por landmarks (nariz, olhos, boca)
      const biLandmarks = biFaces[0].landmarks || [];
      const selfieLandmarks = selfieFaces[0].landmarks || [];

      // Em produção: implementar cálculo de distância euclidiana entre landmarks
      // Por agora: verificar se ambos têm landmarks detectados
      const biHasLandmarks = biLandmarks.length > 5;
      const selfieHasLandmarks = selfieLandmarks.length > 5;

      // Similaridade básica: se ambos têm rostos com landmarks, assumir match
      // TODO: Implementar cálculo de distância real ou usar AWS Rekognition CompareFaces
      return biHasLandmarks && selfieHasLandmarks;
    } catch (err) {
      console.error('[KYC] Erro na comparação de rostos:', err);
      return false;
    }
  }

  /**
   * Comparar nomes (normaliza acentos e case-insensitive)
   */
  private compareNames(nomeConta: string, nomeBI: string): boolean {
    const normalize = (str: string) =>
      str
        .toLowerCase()
        .normalize('NFD')
        .replace(/[\u0300-\u036f]/g, '') // Remove acentos
        .replace(/[^a-z\s]/g, '') // Remove caracteres especiais
        .trim();

    const nomeContaNorm = normalize(nomeConta);
    const nomeBINorm = normalize(nomeBI);

    // Match se o nome do BI contém o nome da conta (ou vice-versa)
    return (
      nomeContaNorm.includes(nomeBINorm) ||
      nomeBINorm.includes(nomeContaNorm) ||
      nomeContaNorm === nomeBINorm
    );
  }

  /**
   * Adicionar marca d'água SVG "TudoAqui - KYC"
   */
  private async addWatermark(imageBuffer: Buffer, text: string): Promise<Buffer> {
    const watermarkSvg = Buffer.from(`
      <svg width="350" height="60">
        <style>
          .texto {
            font-family: Arial, sans-serif;
            font-size: 24px;
            font-weight: bold;
            fill: rgba(255, 255, 255, 0.8);
            stroke: rgba(0, 0, 0, 0.5);
            stroke-width: 1px;
          }
        </style>
        <text x="10" y="40" class="texto">${text}</text>
      </svg>
    `);

    return sharp(imageBuffer)
      .resize(1200, 1600, { fit: 'inside' })
      .composite([
        {
          input: watermarkSvg,
          gravity: 'southeast',
        },
      ])
      .jpeg({ quality: 85 })
      .toBuffer();
  }

  /**
   * Garantir que diretório existe
   */
  private async ensureDir(dirPath: string): Promise<void> {
    try {
      await fs.access(dirPath);
    } catch {
      await fs.mkdir(dirPath, { recursive: true });
    }
  }

  /**
   * Upload de Alvará (empresas)
   */
  async processAlvara(userId: string, file: Express.Multer.File) {
    if (!file?.buffer) {
      throw new BadRequestException('Ficheiro do Alvará é obrigatório');
    }

    const user = await this.userRepository.findOne({ where: { id: userId } });
    if (!user) {
      throw new BadRequestException('Utilizador não encontrado');
    }

    const outputDir = path.join(process.cwd(), 'uploads', 'alvara');
    await this.ensureDir(outputDir);

    const filename = `${userId}_${Date.now()}.pdf`;
    const outputPath = path.join(outputDir, filename);

    await fs.writeFile(outputPath, file.buffer);

    user.alvaraUrl = outputPath;
    await this.userRepository.save(user);

    return {
      message: 'Alvará carregado com sucesso',
      url: outputPath,
    };
  }
}

