import { Injectable, HttpException, HttpStatus } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';

interface EstimativaEntregaDto {
  restauranteLatitude: number;
  restauranteLongitude: number;
  clienteLatitude: number;
  clienteLongitude: number;
}

interface EstimativaResponse {
  distanciaKm: number;
  tempoEstimadoMinutos: number;
  precoEntrega: number;
  disponivel: boolean;
}

/**
 * Serviço de integração com Tupuca (Logística Angolana)
 * API: https://api.tupuca.ao
 */
@Injectable()
export class TupucaService {
  private readonly apiKey: string;
  private readonly baseUrl: string;

  constructor(private readonly configService: ConfigService) {
    this.apiKey = this.configService.get<string>('TUPUCA_API_KEY') || 'mock-api-key-dev';
    this.baseUrl = this.configService.get<string>('TUPUCA_BASE_URL') || 'https://api.tupuca.ao';
  }

  /**
   * Calcular estimativa de entrega
   * TODO: Integrar com API real da Tupuca quando credenciais estiverem disponíveis
   */
  async calcularEstimativaEntrega(dados: EstimativaEntregaDto): Promise<EstimativaResponse> {
    try {
      // Calcular distância usando fórmula de Haversine
      const distanciaKm = this.calcularDistanciaHaversine(
        dados.restauranteLatitude,
        dados.restauranteLongitude,
        dados.clienteLatitude,
        dados.clienteLongitude,
      );

      // Lógica de pricing (mock - substituir com API real)
      const precoBase = 500; // 500 Kz base
      const precoPorKm = 150; // 150 Kz por km
      const precoEntrega = precoBase + (distanciaKm * precoPorKm);

      // Tempo estimado: 5 min por km + 10 min de preparação
      const tempoEstimadoMinutos = Math.ceil((distanciaKm * 5) + 10);

      // Disponível apenas se < 15km
      const disponivel = distanciaKm <= 15;

      return {
        distanciaKm: Number(distanciaKm.toFixed(2)),
        tempoEstimadoMinutos,
        precoEntrega: Math.ceil(precoEntrega),
        disponivel,
      };

      // TODO: Quando API estiver disponível, usar:
      // const response = await fetch(`${this.baseUrl}/v1/delivery/estimate`, {
      //   method: 'POST',
      //   headers: {
      //     'Authorization': `Bearer ${this.apiKey}`,
      //     'Content-Type': 'application/json',
      //   },
      //   body: JSON.stringify(dados),
      // });
      // return response.json();

    } catch (error) {
      console.error('Erro ao calcular estimativa Tupuca:', error);
      throw new HttpException(
        'Erro ao calcular estimativa de entrega',
        HttpStatus.SERVICE_UNAVAILABLE,
      );
    }
  }

  /**
   * Criar pedido de entrega
   */
  async criarPedidoEntrega(pedidoId: string, origem: any, destino: any) {
    // TODO: Implementar chamada real à API Tupuca
    console.log('📦 Pedido de entrega criado (MOCK):', { pedidoId, origem, destino });
    
    return {
      success: true,
      trackingId: `TUPUCA-${Date.now()}`,
      message: 'Entregador a caminho (funcionalidade em desenvolvimento)',
    };
  }

  /**
   * Rastrear entrega em tempo real
   */
  async rastrearEntrega(trackingId: string) {
    // TODO: Integrar com API Tupuca para GPS em tempo real via Redis
    return {
      trackingId,
      status: 'em_rota',
      latitude: -8.8383,
      longitude: 13.2344,
      tempoRestante: 15,
    };
  }

  /**
   * Calcular distância entre dois pontos (Haversine)
   */
  private calcularDistanciaHaversine(lat1: number, lon1: number, lat2: number, lon2: number): number {
    const R = 6371; // Raio da Terra em km
    const dLat = this.toRad(lat2 - lat1);
    const dLon = this.toRad(lon2 - lon1);
    
    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos(this.toRad(lat1)) * Math.cos(this.toRad(lat2)) *
      Math.sin(dLon / 2) * Math.sin(dLon / 2);
    
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return R * c;
  }

  private toRad(degrees: number): number {
    return degrees * (Math.PI / 180);
  }
}
