import {
  Controller,
  Post,
  Get,
  Body,
  Param,
  Query,
  UseGuards,
  Request,
  UseInterceptors,
  UploadedFiles,
} from '@nestjs/common';
import { FilesInterceptor } from '@nestjs/platform-express';
import { MarketplaceService } from './marketplace.service';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';
import { CriarAnuncioDto, FiltrosAnuncioDto } from './dto/criar-anuncio.dto';

@Controller('marketplace')
export class MarketplaceController {
  constructor(private readonly marketplaceService: MarketplaceService) {}

  /**
   * Criar novo anúncio (requer autenticação)
   * Upload de até 10 fotos
   */
  @Post('anuncios')
  @UseGuards(JwtAuthGuard)
  @UseInterceptors(FilesInterceptor('fotos', 10, {
    limits: { fileSize: 5 * 1024 * 1024 }, // 5MB por foto
    fileFilter: (req, file, cb) => {
      if (!file.mimetype.match(/\/(jpg|jpeg|png)$/)) {
        return cb(new Error('Apenas imagens JPG/PNG são permitidas'), false);
      }
      cb(null, true);
    },
  }))
  async criarAnuncio(
    @Request() req: any,
    @Body() dto: CriarAnuncioDto,
    @UploadedFiles() fotos: Express.Multer.File[],
  ) {
    return this.marketplaceService.criarAnuncio(req.user.userId, dto, fotos);
  }

  /**
   * Listar anúncios com filtros
   */
  @Get('anuncios')
  async listarAnuncios(@Query() filtros: FiltrosAnuncioDto) {
    return this.marketplaceService.listarAnuncios(filtros);
  }

  /**
   * Obter detalhes de anúncio específico
   */
  @Get('anuncios/:id')
  async obterAnuncio(@Param('id') id: string) {
    return this.marketplaceService.obterAnuncio(id);
  }

  /**
   * Busca por província/município (Angola)
   */
  @Get('anuncios/localidade/:provincia/:municipio')
  async buscarPorLocalidade(
    @Param('provincia') provincia: string,
    @Param('municipio') municipio: string,
    @Query() filtros: FiltrosAnuncioDto,
  ) {
    return this.marketplaceService.listarAnuncios({
      ...filtros,
      provincia,
      municipio,
    });
  }
}
