import { Injectable, NotFoundException, UnauthorizedException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import sharp from 'sharp';
import { readFile, writeFile, mkdir } from 'node:fs/promises';
import { join } from 'node:path';
import { Anuncio, CategoriaAnuncio, StatusAnuncio } from './entities/anuncio.entity';
import { DetalhesImovel } from './entities/detalhes-imovel.entity';
import { DetalhesVeiculo } from './entities/detalhes-veiculo.entity';
import { CriarAnuncioDto, FiltrosAnuncioDto } from './dto/criar-anuncio.dto';

@Injectable()
export class MarketplaceService {
  constructor(
    @InjectRepository(Anuncio)
    private readonly anuncioRepository: Repository<Anuncio>,
    @InjectRepository(DetalhesImovel)
    private readonly detalhesImovelRepository: Repository<DetalhesImovel>,
    @InjectRepository(DetalhesVeiculo)
    private readonly detalhesVeiculoRepository: Repository<DetalhesVeiculo>,
  ) {}

  /**
   * Criar novo anúncio
   * Apenas vendedores verificados podem publicar
   */
  async criarAnuncio(userId: string, dto: CriarAnuncioDto, fotos: Express.Multer.File[]) {
    // Processar fotos: aplicar marca d'água "TudoAqui"
    const fotosComMarcaDagua = await Promise.all(
      fotos.map(foto => this.aplicarMarcaDagua(foto.buffer, foto.originalname))
    );

    // Criar anúncio
    const anuncio = this.anuncioRepository.create({
      vendedorId: userId,
      categoria: dto.categoria,
      titulo: dto.titulo,
      descricao: dto.descricao,
      preco: dto.preco,
      provincia: dto.provincia,
      municipio: dto.municipio,
      bairro: dto.bairro,
      endereco: dto.endereco,
      localizacao: dto.latitude && dto.longitude
        ? { type: 'Point', coordinates: [dto.longitude, dto.latitude] } as any
        : null,
      fotos: fotosComMarcaDagua.map(f => f.urlComMarca),
      fotosOriginais: fotosComMarcaDagua.map(f => f.urlOriginal),
      status: StatusAnuncio.ATIVO,
    });

    await this.anuncioRepository.save(anuncio);

    // Criar detalhes específicos por categoria
    if (dto.categoria === CategoriaAnuncio.IMOVEIS && dto.tipoImovel) {
      const detalhes = this.detalhesImovelRepository.create({
        anuncioId: anuncio.id,
        tipo: dto.tipoImovel,
        tipoNegocio: dto.tipoNegocio,
        quartos: dto.quartos,
        banheiros: dto.banheiros,
        garagem: dto.garagem,
        area: dto.area,
        mobilado: dto.mobilado,
        piscina: dto.piscina,
      });
      await this.detalhesImovelRepository.save(detalhes);
    }

    if (dto.categoria === CategoriaAnuncio.VEICULOS && dto.tipoVeiculo) {
      const detalhes = this.detalhesVeiculoRepository.create({
        anuncioId: anuncio.id,
        tipo: dto.tipoVeiculo,
        marca: dto.marca,
        modelo: dto.modelo,
        ano: dto.ano,
        combustivel: dto.combustivel,
        transmissao: dto.transmissao,
        quilometragem: dto.quilometragem,
        cor: dto.cor,
      });
      await this.detalhesVeiculoRepository.save(detalhes);
    }

    return {
      message: 'Anúncio criado com sucesso',
      anuncio: {
        id: anuncio.id,
        titulo: anuncio.titulo,
        preco: anuncio.preco,
        fotos: anuncio.fotos,
      },
    };
  }

  /**
   * Listar anúncios com filtros
   */
  async listarAnuncios(filtros: FiltrosAnuncioDto) {
    const query = this.anuncioRepository
      .createQueryBuilder('anuncio')
      .leftJoinAndSelect('anuncio.vendedor', 'vendedor')
      .where('anuncio.status = :status', { status: StatusAnuncio.ATIVO })
      .orderBy('anuncio.destaque', 'DESC') // Anúncios em destaque primeiro
      .addOrderBy('anuncio.createdAt', 'DESC');

    if (filtros.categoria) {
      query.andWhere('anuncio.categoria = :categoria', { categoria: filtros.categoria });
    }

    if (filtros.provincia) {
      query.andWhere('anuncio.provincia = :provincia', { provincia: filtros.provincia });
    }

    if (filtros.municipio) {
      query.andWhere('anuncio.municipio = :municipio', { municipio: filtros.municipio });
    }

    if (filtros.precoMin) {
      query.andWhere('anuncio.preco >= :precoMin', { precoMin: filtros.precoMin });
    }

    if (filtros.precoMax) {
      query.andWhere('anuncio.preco <= :precoMax', { precoMax: filtros.precoMax });
    }

    if (filtros.busca) {
      query.andWhere('(anuncio.titulo ILIKE :busca OR anuncio.descricao ILIKE :busca)', {
        busca: `%${filtros.busca}%`,
      });
    }

    // Filtro "Perto de Mim" usando PostGIS
    if (filtros.latitude && filtros.longitude && filtros.raioKm) {
      const raioMetros = filtros.raioKm * 1000;
      query.andWhere(
        `ST_DWithin(anuncio.localizacao, ST_SetSRID(ST_MakePoint(:lng, :lat), 4326)::geography, :raio)`,
        {
          lat: filtros.latitude,
          lng: filtros.longitude,
          raio: raioMetros,
        }
      );
    }

    const anuncios = await query.getMany();

    return {
      total: anuncios.length,
      anuncios: anuncios.map(a => ({
        id: a.id,
        titulo: a.titulo,
        preco: a.preco,
        provincia: a.provincia,
        municipio: a.municipio,
        fotos: a.fotos.slice(0, 1), // Apenas primeira foto na listagem
        destaque: a.destaque,
        vendedor: {
          nome: a.vendedor.nome,
          seloVerificado: a.vendedor.seloVerificado,
        },
      })),
    };
  }

  /**
   * Obter detalhes de um anúncio
   */
  async obterAnuncio(id: string) {
    const anuncio = await this.anuncioRepository.findOne({
      where: { id },
      relations: ['vendedor'],
    });

    if (!anuncio) {
      throw new NotFoundException('Anúncio não encontrado');
    }

    // Incrementar visualizações
    await this.anuncioRepository.update(id, {
      visualizacoes: anuncio.visualizacoes + 1,
    });

    let detalhes = null;
    if (anuncio.categoria === CategoriaAnuncio.IMOVEIS) {
      detalhes = await this.detalhesImovelRepository.findOne({
        where: { anuncioId: id },
      });
    } else if (anuncio.categoria === CategoriaAnuncio.VEICULOS) {
      detalhes = await this.detalhesVeiculoRepository.findOne({
        where: { anuncioId: id },
      });
    }

    return {
      ...anuncio,
      detalhes,
      vendedor: {
        id: anuncio.vendedor.id,
        nome: anuncio.vendedor.nome,
        telefone: anuncio.vendedor.telefone,
        seloVerificado: anuncio.vendedor.seloVerificado,
      },
    };
  }

  /**
   * Aplicar marca d'água "TudoAqui" nas fotos
   */
  private async aplicarMarcaDagua(buffer: Buffer, nomeArquivo: string) {
    const timestamp = Date.now();
    const nomeUnico = `${timestamp}-${nomeArquivo}`;
    const caminhoOriginal = join(process.cwd(), 'uploads', 'anuncios', 'originais', nomeUnico);
    const caminhoMarca = join(process.cwd(), 'uploads', 'anuncios', 'marca-dagua', nomeUnico);

    // Criar diretórios se não existirem
    await mkdir(join(process.cwd(), 'uploads', 'anuncios', 'originais'), { recursive: true });
    await mkdir(join(process.cwd(), 'uploads', 'anuncios', 'marca-dagua'), { recursive: true });

    // Salvar original
    await writeFile(caminhoOriginal, buffer);

    // Aplicar marca d'água com Sharp
    const marcaDagua = Buffer.from(
      `<svg width="300" height="50">
        <style>
          .texto { font-family: Arial, sans-serif; font-size: 28px; font-weight: bold; fill: white; opacity: 0.7; }
        </style>
        <text x="10" y="35" class="texto">TudoAqui</text>
      </svg>`
    );

    const imagemComMarca = await sharp(buffer)
      .resize(1200, 800, { fit: 'inside', withoutEnlargement: true })
      .composite([
        {
          input: marcaDagua,
          gravity: 'southeast',
        },
      ])
      .jpeg({ quality: 85 })
      .toBuffer();

    await writeFile(caminhoMarca, imagemComMarca);

    return {
      urlOriginal: `/uploads/anuncios/originais/${nomeUnico}`,
      urlComMarca: `/uploads/anuncios/marca-dagua/${nomeUnico}`,
    };
  }
}
