import { Injectable, HttpException, HttpStatus } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';

interface PagamentoDto {
  valor: number;
  telefone: string;
  descricao: string;
  referencia: string; // ID do pedido/anúncio
}

interface StatusPagamentoResponse {
  transacaoId: string;
  status: 'pendente' | 'aprovado' | 'rejeitado' | 'cancelado';
  valor: number;
  dataHora: Date;
}

/**
 * Serviço de integração com Multicaixa Express e GPO
 * Para pagamentos via USSD e carteira digital
 */
@Injectable()
export class PagamentosService {
  private readonly multicaixaMerchantId: string;
  private readonly gpoApiKey: string;

  constructor(private readonly configService: ConfigService) {
    this.multicaixaMerchantId = this.configService.get<string>('MULTICAIXA_MERCHANT_ID') || 'mock-merchant-id';
    this.gpoApiKey = this.configService.get<string>('GPO_API_KEY') || 'mock-gpo-key';
  }

  /**
   * Iniciar pagamento via Multicaixa Express
   * Utilizador recebe notificação no telemóvel para confirmar
   */
  async iniciarPagamento(dados: PagamentoDto) {
    try {
      console.log('💳 Iniciando pagamento Multicaixa Express:', dados);

      // TODO: Integrar com API real Multicaixa/GPO
      // const response = await fetch('https://api.gpo.ao/v1/pagamentos', {
      //   method: 'POST',
      //   headers: {
      //     'Authorization': `Bearer ${this.gpoApiKey}`,
      //     'Content-Type': 'application/json',
      //   },
      //   body: JSON.stringify({
      //     merchant_id: this.multicaixaMerchantId,
      //     amount: dados.valor,
      //     phone: dados.telefone,
      //     description: dados.descricao,
      //     reference: dados.referencia,
      //   }),
      // });

      // Mock response
      const transacaoId = `MCX-${Date.now()}`;
      
      return {
        success: true,
        transacaoId,
        mensagem: 'Pagamento iniciado. Confirme no seu Multicaixa Express.',
        status: 'pendente',
        valor: dados.valor,
        telefone: dados.telefone,
      };

    } catch (error) {
      console.error('Erro ao iniciar pagamento:', error);
      throw new HttpException(
        'Erro ao processar pagamento',
        HttpStatus.SERVICE_UNAVAILABLE,
      );
    }
  }

  /**
   * Verificar status de pagamento
   */
  async verificarStatusPagamento(transacaoId: string): Promise<StatusPagamentoResponse> {
    // TODO: Consultar API real para status
    console.log('🔍 Verificando status de pagamento:', transacaoId);

    return {
      transacaoId,
      status: 'pendente',
      valor: 5000,
      dataHora: new Date(),
    };
  }

  /**
   * Webhook para receber confirmação de pagamento
   * Multicaixa/GPO notifica quando o utilizador confirma
   */
  async processarWebhookPagamento(payload: any) {
    console.log('📲 Webhook recebido:', payload);

    // Validar assinatura do webhook (segurança)
    // TODO: Implementar validação real

    const { transacao_id, status, valor, referencia } = payload;

    if (status === 'aprovado') {
      // Pagamento confirmado - liberar pedido ou ativar destaque
      console.log(`✅ Pagamento aprovado: ${transacao_id} - Valor: ${valor} Kz`);
      
      // TODO: Notificar via Firebase Cloud Messaging
      // TODO: Atualizar status no banco de dados
      
      return {
        success: true,
        message: 'Pagamento processado com sucesso',
      };
    }

    if (status === 'rejeitado') {
      console.log(`❌ Pagamento rejeitado: ${transacao_id}`);
      return {
        success: false,
        message: 'Pagamento rejeitado',
      };
    }

    return {
      success: true,
      message: 'Webhook processado',
    };
  }

  /**
   * Calcular taxa de transação (2-5% conforme plano de negócio)
   */
  calcularTaxaPlataforma(valorTotal: number): number {
    const percentualTaxa = 0.025; // 2.5%
    return Math.ceil(valorTotal * percentualTaxa);
  }
}
